!    XCASCADE software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains subroutines to deal with objects/types
MODULE Objects_and_types 
implicit none

!==============================================
! All photons in one object, arrays inside:
type Photons ! photon as an object, contains the following info:
   real(8), DIMENSION(:), ALLOCATABLE :: E ! energy [eV]
   real(8), DIMENSION(:), ALLOCATABLE :: ti ! time of impact
end type Photons

type, EXTENDS (Photons) :: Electrons ! electron as an object contains the same into as photon + more:
   real(8), DIMENSION(:), ALLOCATABLE :: t ! current time [eV]
end type Electrons 

type, EXTENDS (Electrons) :: Holes ! hole as an object, contains the same info as electron + more:
   integer, DIMENSION(:), ALLOCATABLE :: KOA ! kind of atom
   integer, DIMENSION(:), ALLOCATABLE :: Sh  ! shell
end type Holes
!==============================================

!==============================================
! One photon as an object, array of them:
type :: Photon ! photon as an object, contains the following info:
   real(8) E ! energy [eV]
   real(8) ti ! time of impact
end type Photon

type, EXTENDS (Photon) :: Electron ! electron as an object contains the same into as photon + more:
   real(8) t ! current time [eV]
end type Electron

type, EXTENDS (Electron) :: Hole ! hole as an object, contains the same info as electron + more:
   integer KOA ! kind of atom
   integer Sh  ! shell
end type Hole
!==============================================

!==============================================
! One atom as an object:
type :: Atom ! photon as an object, contains the following info:
   integer Z ! atomic number
   real(8) M ! atomic mass in [proton mass] units
   character(3) Name
   character(30) Full_Name
end type Atom

!==============================================
! Error handling as an object:
type Error_handling
   LOGICAL Err		! indicates that an error occured
   integer Err_Num	! assign a number to an error
   integer File_Num		! number of the file with error log
   character(100) Err_descript	! describes more details about the error
end type
!==============================================


!==============================================
! Complex dielectric function as an object:
type :: CDF ! parameters entering Ritchi-CDF:
   real(8), DIMENSION(:), ALLOCATABLE :: E0       ! parameter E0
   real(8), DIMENSION(:), ALLOCATABLE :: A        ! parameter A
   real(8), DIMENSION(:), ALLOCATABLE :: Gamma    ! parameter gamma
   character(3) Atomtype    ! if one wants to save atoms name
   character(11) Shelltype  ! if one wants to save shell's name
end type CDF
!==============================================


!==============================================
! Mean free path read from file:
type :: MFP_from_file ! parameters entering Ritchi-CDF:
   real(8), DIMENSION(:), ALLOCATABLE :: E	! energy [eV]
   real(8), DIMENSION(:), ALLOCATABLE :: MFP	! Mean Free Path [A]
end type MFP_from_file
!==============================================



!==============================================
! Distribution functions:
type Density_time
   real(8), DIMENSION(:), ALLOCATABLE :: t ! time instances
end type Density_time

type, extends (Density_time) :: Electron_density
   real(8), DIMENSION(:), ALLOCATABLE :: ne ! electron density
   real(8), DIMENSION(:), ALLOCATABLE :: Ee ! electron energy density
   real(8), DIMENSION(:,:), ALLOCATABLE :: Ee_vs_t	! transient electron distribution (1/E vs t)
   real(8), DIMENSION(:), ALLOCATABLE :: Ee_vs_t_grid	! grid for transient electron distribution (1/E vs t)
end type Electron_density

type, extends (Density_time) :: Hole_density
   real(8), dimension(:,:), allocatable :: nh ! holes in different shells
   real(8), DIMENSION(:,:), ALLOCATABLE :: Eh ! holes energy density
end type Hole_density
!==============================================

!CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
 contains


! How to write the log about an error:
subroutine Save_error_details(Err_name, Err_num, Err_data)
   class(Error_handling) :: Err_name
   integer, intent(in) :: Err_num
   character(100), intent(in) :: Err_data
   integer FN
   FN = Err_name%File_Num
   Err_name%Err = .true.
   Err_name%Err_Num = Err_num
   Err_name%Err_descript = Err_data
   write(FN, '(a,i2,1x,a)') 'Error #', Err_name%Err_Num, trim(adjustl(Err_name%Err_descript))
end subroutine Save_error_details


! How to update the distribution function:
subroutine Distribution_t(Distr, i, t, ne, nh, Ee, Eh)
   class(Density_time), intent(inout) :: Distr
   integer, intent(in) :: i ! timestep nmumber
   real(8), intent(in) :: t ! time [fs]
   real(8), optional, intent(in) :: ne ! electron density [1/A^3]
   real(8), optional, intent(in) :: Ee ! electron energy density [1/A^3]
   real(8), dimension(:), optional, intent(in) :: nh ! hole densities for each shell [1/A^3]
   real(8), dimension(:), optional, intent(in) :: Eh ! hole energy densities for each shell [1/A^3]

   select type (Distr)
      type is (Electron_density)
         print*, 'Electron_density'
         if (present(ne)) then ! it's electron distribution
            Distr%t(i) = t
            Distr%ne(i) = ne
            Distr%Ee(i) = Ee
         endif
         if (present(nh)) then ! it's hole distribution
            print*, 'Wrong type/value for electron distribution!'
         endif
      type is (Hole_density)
         print*, 'Hole_density'
         if (present(ne)) then ! it's electron distribution
            print*, 'Wrong type/value for hole distribution!'
         endif
         if (present(nh)) then ! it's hole distribution
            Distr%t(i) = t
            Distr%nh(i,:) = nh(:)
            Distr%Eh(i,:) = Eh(:)
         endif
   end select
end subroutine Distribution_t 


! How to initialize the electron, photon, or hole object, and assign the data:
subroutine Initialize_object(TP, Particl, SZ, N, E, ti, t, KOA, Sh)
   integer, intent(in), optional :: TP
   class(*), dimension(:), allocatable, intent(inout) :: Particl
   integer, intent(in) :: SZ ! size of the array
   integer, intent(in) :: N  ! number of this particle
   real(8), intent(in) :: E  ! energy [eV]
   real(8), intent(in) :: ti ! time of impact [fs]
   real(8), intent(in), optional :: t ! current time [fs]
   integer, intent(in), optional :: KOA ! kind of atom
   integer, intent(in), optional :: Sh  ! atomic shell for holes

   if (.not. allocated(Particl)) then
    if (present(TP)) then ! it's first time, initiate this:
      select case (TP)
         case (:1)
            allocate(Photon::Particl(SZ))
         case (2)
            allocate(Electron::Particl(SZ))
         case (3)
            allocate(Hole::Particl(SZ))
      end select
      select type (Particl)
         type is (Photon)
            Particl%E = 0.d0 ! define energy [eV]
            Particl%ti = 1.0d15 ! define time [fs]
         type is (Electron)
            Particl%E = 0.0d0 ! define energy [eV]
            Particl%ti = 1.0d15 ! define time [fs]
            Particl%t = 1.0d18 ! define current time [fs]
         type is (Hole)
            Particl%E = 0.0d0 ! define energy [eV]
            Particl%ti = 1.0d15 ! define time [fs]
            Particl%t = 0.0d0 ! define current time [fs]
            Particl%KOA = 0 ! kind of atom
            Particl%Sh = 0 ! shell of hole
      end select
    else ! present(TP)
      print*, 'ERROR! Array is not allocated, and the shape is not given!'
    endif ! present(TP)
   endif ! .not. allocated(Particl)

   select type (Particl)
      type is (Photon) ! if it is a photon:
         Particl(N)%E = E ! define energy [eV]
         Particl(N)%ti = ti ! define time [fs]
      type is (Electron) ! if it is an electron:
         Particl(N)%E = E ! define energy [eV]
         Particl(N)%ti = ti ! define time [fs]
         if (present(t)) then
            Particl(N)%t = t ! define current time [fs]
         endif
      type is (Hole) ! if it's a hole:
         Particl(N)%E = E ! define energy [eV]
         Particl(N)%ti = ti ! define time [fs]
         if (present(t)) then
            Particl(N)%t = t ! define current time [fs]
         endif
         if (present(KOA)) then
            Particl(N)%KOA = KOA ! kind of atom
         endif
         if (present(sh)) then
            Particl(N)%Sh = Sh ! shell of hole
         endif
   end select
end subroutine Initialize_object


! How to initialize the electron, photon, or hole object, and assign the data:
subroutine Initialize_object_array(Particl, SZ, N, E, ti, t, KOA, Sh)
   class(Photons) :: Particl
   integer, intent(in) :: SZ ! size of the array
   integer, intent(in) :: N  ! number of this particle
   real(8), intent(in) :: E  ! energy [eV]
   real(8), intent(in) :: ti ! time of impact [fs]
   real(8), intent(in), optional :: t ! current time [fs]
   integer, intent(in), optional :: KOA ! kind of atom
   integer, intent(in), optional :: Sh  ! atomic shell for holes

   if (.not. allocated(Particl%E)) then
      allocate(Particl%E(SZ))
      Particl%E = 0.0d0 ! [eV] to start with
   endif
   if (.not. allocated(Particl%ti)) then 
      allocate(Particl%ti(SZ))
      Particl%ti = 1.0d15 ! [fs] to start with
   endif

   select type (Particl)
   type is (Photons) ! if it is a photon:
      Particl%E(N) = E ! define energy [eV]
      Particl%ti(N) = ti ! define time [fs]
   class is (Electrons) ! if it is an electron:
      if (present(t)) then
         if (.not. allocated(Particl%t)) then
            allocate(Particl%t(SZ))
            Particl%t = 0.0d0 ! [fs] to start with
         endif
         Particl%t(N) = t ! define current time [fs]
      else
      endif
   class is (Holes)
      if (present(t)) then
         if (.not. allocated(Particl%t)) then 
            allocate(Particl%t(SZ))
            Particl%t = 0.0d0 ! [fs] to start with
         endif
         Particl%t(N) = t ! define current time [fs]
      else
      endif
      if (present(KOA)) then
         if (.not. allocated(Particl%KOA)) then
            allocate(Particl%KOA(SZ))
            Particl%KOA = 0 ! kind of atom
         endif
         Particl%KOA(N) = KOA ! kind of atom
      else
      endif
      if (present(sh)) then
         if (.not. allocated(Particl%Sh)) then
            allocate(Particl%Sh(SZ))
            Particl%Sh = 0 ! shell of hole 
         endif
         Particl%Sh(N) = Sh ! shell of hole
      else
      endif
   end select
end subroutine Initialize_object_array

end MODULE Objects_and_types