!    XCASCADE software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains all Monte-Carlo subroutines
MODULE Monte_Carlo

use Constants
use Laser_pulse_parameters
use Objects_and_types
use Cross_sections
use Files_dealing

implicit none

contains

subroutine MC(CSN, N_tim, Ne_max, Fluence, Eph, Z_elements, Element, M_elements, N_elements, Ip_elements, whether_valence, Effgap, Egap, Ekin_elements, Nel_elements, PQN_elements, t_Auger, Photoabs_sigma, At_dens, Photoabs_sigma_tot, KOP_inc, KOPulse, Pulse_max, Pulse_duration, Total_t, dt, El_distr, El_NRG, Temp_El_distr_E, Temp_El_distr_E_grid, Hole_distr, incl_elastic, print_distr, print_casc, Ecut, CurrIter, Error_message, IMFPs, Ritchi)
!    class(Photon), intent (inout) :: Ph		! class of photons
!    class(Electron), intent (inout) :: El	! class of electrons
!    class(Hole), intent (inout) :: Hol		! class of holes
   integer, DIMENSION(:,:), intent(in) :: CSN	! number of cross-seciton
   integer, intent(in) :: N_tim	!maximal time index
   integer, intent(in) :: Ne_max	!maximal number of electrons gives sizes of arrays
   real(8), intent(in) :: Fluence ! fluence
   real(8), intent(in) :: Eph	! [eV] photon energy
   integer, DIMENSION(:), intent(in) :: Z_elements	! atomic numbers of elements
   type(Atom), DIMENSION(:), INTENT(in) :: Element	! data about all elements
   REAL(8), DIMENSION(:), intent(in) :: M_elements	! masses of all elements in molecule
   integer, DIMENSION(:), intent(in) :: N_elements	! numbers of elements in 1 molecule
   real(8), DIMENSION(:,:), intent(in) :: Ip_elements ! [eV] ionization potentials of all shells, all elements
   logical, DIMENSION(:,:), intent(in) :: whether_valence ! whether the hole in certain atom at particular shell is valence or not 
   real(8), intent(in) :: Effgap  ! energy gap; used to find the kinetic energy of valence holes
   real(8), intent(in) :: Egap  ! energy gap; used to find the kinetic energy of valence holes
   real(8), DIMENSION(:,:), intent(in) :: Ekin_elements ! [eV] kinetic energies of all shells, all elements
   real(8), DIMENSION(:,:), intent(in) :: Nel_elements 	! number of electrons in all shells, all elements
   integer, DIMENSION(:,:), intent(in) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements
   real(8), DIMENSION(:,:), intent(in) :: t_Auger	! Auger-decays times [fs]
   real(8), DIMENSION(:,:), intent(in) :: Photoabs_sigma ! [A^2] photoionization cross-sections for all shells
   real(8), intent(in) :: At_dens	! [1/cm^3] atomic density
   real(8), intent(in) :: Photoabs_sigma_tot	! [A^2] total photoabsorption cross-section
   integer, intent(in) :: KOP_inc	! kind of particle: laser vs electron impact
   integer, intent(in) :: KOPulse	! shape of pulse: 0=flat-top, 1=Gaussian
   real(8), intent(in) :: Pulse_max ! position of the maximum of the pulse (mu-gauss)
   real(8), intent(in) :: Pulse_duration ! FWHM of the pulse (sigma-gauss)
   real(8), intent(in) :: Total_t	! [fs] total duration of analysis 
   real(8), intent(in) :: dt	! [fs] time-step
   real(8), dimension(:), intent(inout) :: El_distr ! electron density in time
   real(8), dimension(:), intent(inout) :: El_NRG   ! electron energy density in time
   real(8), dimension(:,:), intent(inout) :: Temp_El_distr_E    ! electron distribution in the energy space vs time
   real(8), dimension(:), intent(inout) :: Temp_El_distr_E_grid ! grid electron distribution in the energy space vs time
   real(8), dimension(:,:), intent(inout) :: Hole_distr   ! holes densities in time
   ! Local variables:
   class(*), DIMENSION(:), ALLOCATABLE :: Ph, El, Hol 	! 1 photon object, array of them
   logical incl_elastic ! include the effect of elastic collisions? otherwise only inelastic ones are included
   logical, intent(in) :: print_distr ! print out the transient electron distribution?
   logical, intent(in) :: print_casc ! print out the cascades?
   real(8), intent(in) :: Ecut 	! energy cutoff for printing cascades
   integer, intent(in) :: CurrIter ! number of the cascade/iteration   
   
   type(Error_handling), intent(inout) :: Error_message ! deals with errors, if any
   type(MFP_from_file), dimension(:,:), intent(inout), optional :: IMFPs	! inelastic mean free paths from files
   type(CDF), dimension(:,:), intent(inout), optional :: Ritchi    ! Ritchi CDF to use for cross-sections
   real(8) Phot_arr_time, tim, RN, t_cur, dE, temp, Sigma_tot, IMFP, E1, E2, t1, t_next_e1, t_next_e2
   integer Ph_max, Ne_CB, Sh1, Sh2, KOA1, KOA2, Nph
   integer my_id,OMP_GET_THREAD_NUM, i,j,k,l, koe, NOP, KOA, SHL, Atoms, Shells

   real(8) Eh   ! kinetic energy of created valence hole
   integer fn  ! each cascade is written in a separate file
   character(100) File_name
   character(10) file_number

   logical, dimension(:), allocatable :: elastic_collision		! whether the next collision will be elastic for a particular electron
   real(8) EMFP, MFP	! elastic mean-free path and total mean-free path
   real(8) sigma_el, sigma_in
   
   if (.not. allocated(elastic_collision)) allocate(elastic_collision(Ne_max))
   elastic_collision(:) = .false.

   if (print_casc) then
	   !~ my_id = OMP_GET_THREAD_NUM()
	   fn = CurrIter+50	! could be dangerous?

	   write (file_number, "(I7)") CurrIter
	   File_name = 'cascades/OUTPUT_cascade'//trim(adjustl(file_number))//'.dat'
	   
	   open(unit=fn, file=File_name, action="write", status="replace")
   endif

   Atoms = Size(Ip_elements,1) ! dimensions, how many spieces
   Shells = Size(Ip_elements,2) ! dimensions, how many shells

   !open(unit=6,form='formatted',carriagecontrol='fortran') ! formatted printing on the screen; works in ifort only, but seems to be unnecessary

   Phot_arr_time = Photon_arrival_time(KOPulse, Pulse_max, Pulse_duration) ! when the photon arrives [fs]


   select case (KOP_inc)
      case (:1, 3:)	! photon, laser pulse
         Ph_max = 1	! analyse only one cascade at a time for the simple code
         Ne_CB = 0 ! there are no free electrons at the start
	if (fluence .gt. 0.d0) then
		call Initialize_object(1, Ph, Ph_max, 1, Eph, Phot_arr_time) ! initialize photons
		t_cur = Phot_arr_time ! [fs] everything starts with 1st photon absorption
		tim = Pulse_max - 3.0d0*Pulse_duration ! [fs] starting time
	else
		call Initialize_object(1, Ph, Ph_max, 1, Eph, 0.d0) ! initialize photons
		t_cur=0.d0 ! [fs] everything starts from zero
		tim = 0.d0   ! [fs] calulation starting time
	endif

         call Initialize_object(2, El, Ne_max, 1, 0.0d0, 1.0d15, 1.0d15) ! just to start, initialize electrons
         call Initialize_object(3, Hol, Ne_max, 1, 0.0d0, 1.0d15, 1.0d15, 0, 0) ! just to start, initialize holes

      case (2)	! electron
         Ph_max = 1	! analyse only one cascade at a time for the simple code
         Ne_CB = 1 	! there is only one free electron at the start, incident electron
         call Initialize_object(1, Ph, Ph_max, 1, 0.0d0, 1.0d25) ! initialize photons
         if (Fluence .GT. 0) then
            call Initialize_object(2, El, Ne_max, 1, Eph, Phot_arr_time+dt, Phot_arr_time) ! just to start, initialize electrons
            t_cur = Phot_arr_time ! [fs] everything starts with 1st electron impact
            tim = Pulse_max - 3.0d0*Pulse_duration ! [fs] starting time
         else
            call Initialize_object(2, El, Ne_max, 1, Eph, 0.d0, Phot_arr_time) ! just to start, initialize electrons
            t_cur = 0.0d0  ! [fs] everything starts from zero
            tim = 0.0d0    ! [fs] starting time
         endif
         call Initialize_object(3, Hol, Ne_max, 1, 0.0d0, 1.0d15, 1.0d15, 0, 0) ! just to start, initialize holes
   endselect
   !t_next_e1 = 1.d19 !!!!!!!!!!!! TO CHANGE LATER !!!!!!!!!!!!!!!!

   i = 0
 
  if (print_casc)  write (fn, *) '#koe	', 'NOP	', 'Time	', 'Ei	', 'Ef	', 'E2	', 'Eh	', 'NeCB'
  

  do i=1,n_tim ! time propagation
  
       call Find_min_time_particle(Ph, El, Hol, koe, NOP, t_cur) ! finds which event will happen next
         if (koe.eq.2 .and. elastic_collision(nop) .eqv. .true.) koe=4	! the next collision of the electron number nop will be elastic if so was decided earlier

	Nph = 0	! number of absorbed photons on each timestep
!       write(6,'(a,f25.16,e25.16)') '+', tim, t_cur
!       call Printing_persents(tim,Total_t)

      tim = tim + dt	! [fs]
      ! Propagate particles until next time-grid point:
      do while (t_cur .LT. tim) ! propagate particles until the time grid point
!          print*, 'NEXT', t_cur, koe
         select case (koe)
            case (1) ! photon
         
               Ne_CB = Ne_CB + 1	! +one electron is excited
               !PPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPP
               select type (Ph)	! since it's free format, it has to be checked every time...
                  type is (Photon)
		  
                  call Select_shell_photoabsorbtion(Photoabs_sigma_tot, Photoabs_sigma, KOA, SHL, Error_message) ! which shell absorbes photon
                  call NRG_of_emmited_electron(Ip_elements, KOA, SHL, Eph, E2, Error_message) ! E2 is the energy given

		  !if (E2 .lt. Ecut) print *, 'energy threshold',Ecut,'is not implemented correctly:', E2
                  

                  call El_free_flight_time(CSN, E2, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, t_next_e1, Ritchi) ! next collision of newly created electron will happen in t_next_e1 fs
		  if (incl_elastic) call account_for_elastic_collisions(N_elements, Z_elements,at_dens,Sigma_tot,E2,elastic_collision(Ne_CB),t_next_e1)	! Sigma_tot and t_next_e1 are corrected to account for elastic collisions

		  if (E2.ge.Ecut) then
			call random_number(RN)
			t_next_e1 = -t_next_e1*log(RN)
		  else
			t_next_e1 = 1.d15	! if the energy is smaller than the threshold, we do not follow this electron anymore: The next collision time is infinity
		  endif
		  
                  call Initialize_object(2,El, Ne_max, Ne_CB, E2, Ph(NOP)%ti + t_next_e1, Ph(NOP)%ti) ! electron created

                  ! Hole treatement:
                  temp = Auger_decay_sampled(KOA, SHL, t_Auger) ! temp [fs] is Auger decay time sampled

                  call Initialize_object(3,Hol, Ne_max, Ne_CB, Ip_elements(KOA, SHL), Ph(NOP)%ti + temp, Ph(NOP)%ti, KOA, SHL) !hole is left
                  
                  if (whether_valence(KOA, SHL)) then
                        Eh = Ip_elements(KOA, SHL)-Egap
                  else
                        Eh = 0.d0
                  endif

                  if (print_casc)  write (fn,"(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6)")  koe, nop, t_cur,Eph,  0.d0,E2,Eh, Ne_CB        ! print in a file the microscopic information about the cascades, to add spatial resolution later with xcascade3d               

               endselect ! type (Ph)
               call Initialize_object(1,Ph, Ph_max, NOP, 0.0d0, 1.d18) ! photon is absorbed, it disappears
               Nph = Nph + 1	! number of absorbed photons
	       
            !EEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEE
            case (2) ! electron
	    
               select type (El)	! since it's free format, it has to be checked every time...
			type is (Electron)

				call Electron_which_shell(CSN, El(NOP)%E, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Error_message, KOA, SHL, IMFPs, Ritchi) ! which shell it excites
				call Energy_transfer(CSN, El(NOP)%E, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, KOA, SHL, dE, Error_message, IMFPs, At_dens, Ritchi) ! how much energy electron lose
				!                    print*, 'El(NOP)%E', NOP, El(NOP)%E, dE

				Ne_CB = Ne_CB + 1	! +one electron is excited
				E1 = El(NOP)%E-dE     ! remaining energy of incident electron
				E2 = dE - Ip_elements(KOA, SHL) ! energy given to the second electron
				t1 = El(NOP)%ti
				
				!if (El(NOP)%E .lt. Ecut) print *, 'energy threshold',Ecut,'is not implemented correctly:', El(NOP)%E


				! Incoming electron:
				call El_free_flight_time(CSN, E1, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, t_next_e1, Ritchi) ! incoming electron next collision will happen in t_next_e1 fs
				! if we include elastic collisions for the incoming electron, we correct Sigma_tot and t_next_e1:
				if (incl_elastic) call account_for_elastic_collisions(N_elements, Z_elements,at_dens,Sigma_tot,E1,elastic_collision(NOP),t_next_e1)
				
				if (E1.ge.Ecut) then
					call random_number(RN)
					t_next_e1 = -t_next_e1*log(RN)	! next collision time
				else
					t_next_e1 = 1.d15	! if the energy is smaller than the threshold, we do not follow this electron anymore: The next collision time is infinity
				endif

				! Excited electron:
				call El_free_flight_time(CSN, E2, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, t_next_e2, Ritchi) ! ! excited electron next collision will happen in t_next_e2 fs
				! we can account for the elastic collisions for the excited electron: correct Sigma_tot and t_next_e2
				if (incl_elastic) call account_for_elastic_collisions(N_elements, Z_elements,at_dens,Sigma_tot,E2,elastic_collision(Ne_CB),t_next_e2)
				
				if (E2.ge.Ecut) then
					call random_number(RN)
					t_next_e2 = -t_next_e2*log(RN)	! next collision time
				else
					t_next_e2 = 1.d15	! if the energy is smaller than the threshold, we do not consider this electron anymore: The next collision time is infinity
				endif


				! Hole is created:
				temp = Auger_decay_sampled(KOA, SHL, t_Auger) ! temp [fs] is Auger decay time sampled
				call Initialize_object(3,Hol, Ne_max, Ne_CB, Ip_elements(KOA, SHL), El(NOP)%ti + temp, El(NOP)%ti, KOA, SHL) !hole is created

                  if (whether_valence(KOA, SHL)) then
                        Eh = Ip_elements(KOA, SHL)-Egap
                  else
                        Eh = 0.d0
                  endif
                  if (print_casc)  write(fn,"(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6)")  koe, nop, t_cur,El(NOP)%E,E1, E2,Eh,Ne_CB

			endselect

		call Initialize_object(2, El, Ne_max, NOP, E1, t_cur+t_next_e1, t_cur)		! incoming electron changes
		call Initialize_object(2, El, Ne_max, Ne_CB, E2, t_cur+t_next_e2, t_cur) 	! excited electron

            !HHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHH
            case (3) ! hole
	    
               select type (Hol)	! since it's free format, it has to be checked every time...
			type is (Hole)

			call Auger_electron(Hol(NOP)%KOA, Hol(NOP)%Sh, Ip_elements, Nel_elements, Sh1, KOA1, Sh2, KOA2, E2, Error_message)
		        KOA = Hol(NOP)%KOA
			SHL = Hol(NOP)%Sh  
                        if (whether_valence(koa,shl)) print*, 'Warning: valence hole',koa,shl,'moved up at', t_cur
               endselect
	       
               if (Sh2 .GT. 0) then ! decay happened
			! original hole is shifted up in energy levels:
			temp = Auger_decay_sampled(KOA1, Sh1, t_Auger) ! temp [fs] is Auger decay time sampled
			call Initialize_object(3,Hol, Ne_max, NOP, Ip_elements(KOA1, Sh1), t_cur + temp, t_cur, KOA1, Sh1) ! this hole moved up
			
			! one more electron-hole is produced:
			
			Ne_CB = Ne_CB + 1	! +one electron is excited

			! second created hole:
			temp = Auger_decay_sampled(KOA2, Sh2, t_Auger) ! temp [fs] is Auger decay time sampled
			call Initialize_object(3,Hol, Ne_max, Ne_CB, Ip_elements(KOA2, Sh2), t_cur + temp, t_cur, KOA2, Sh2)

			! electron is ejected:
			call El_free_flight_time(CSN, E2, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, t_next_e1, Ritchi) ! electron next collision
			! we can account for the elastic collisions for the excited electron
			if (incl_elastic) call account_for_elastic_collisions(N_elements, Z_elements,at_dens,Sigma_tot,E2,elastic_collision(Ne_CB),t_next_e1)

				if (E2.ge.Ecut) then
					call random_number(RN)
					t_next_e1 = -t_next_e1*log(RN)	! next collision time
				else
					t_next_e1 = 1.d15	! if the energy is smaller than the threshold, we do not consider this electron anymore: the next collision time is infinity
				endif
			
			call Initialize_object(2,El, Ne_max, Ne_CB, E2, t_cur + t_next_e1, t_cur) ! electron created

                  if (whether_valence(KOA1, SH1)) then
                        Eh = Ip_elements(KOA1, Sh1)-Egap
                  else
                        Eh = 0.d0
                  endif
                  if (print_casc)  then
                       if (whether_valence(KOA2,SH2)) then
                         write (fn,"(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6)")  koe, nop,t_cur, dble(KOA), Ip_elements(KOA2, Sh2)-Egap,E2, Eh, Ne_CB 
                       else
                        write (fn,"(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6)")  koe,nop,t_cur, dble(KOA), 0.d0,E2, Eh, Ne_CB
                       endif
                  endif
               else !  decay is impossible
			call Initialize_object(3,Hol, Ne_max, NOP, Ip_elements(KOA, SHL), 1.0d17, 1.0d18, KOA, SHL) ! this hole stays there forever...
               endif

            !EEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEE
            case (4) ! elastic collision of an electron
	        
		select type (El)	! since it's free format, it has to be checked every time...
		type is (Electron)

			! Incoming electron:
			E1 = El(NOP)%E
			E2 = El(NOP)%E	! no energy loss
			dE = 0.d0

			call El_free_flight_time(CSN, E1, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, t_next_e1, Ritchi) ! incoming electron's next collision will happen in t_next_e1 fs

! we could follow the elastic collisions until the end of required time, but here we do until the energy is enough for inelastic collisions
			if (Sigma_tot .GT. 0.0d0) then
				call account_for_elastic_collisions(N_elements, Z_elements,at_dens,Sigma_tot,E1,elastic_collision(NOP),t_next_e1)
				call random_number(RN)
				t_next_e1 = -t_next_e1*log(RN)
				
				!if (E1 .lt. Ecut) print *, 'energy threshold',Ecut,'is not implemented correctly:', E1
				if (print_casc)  write (fn, "(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6)")  koe, nop, t_cur, E1, E2, dE, 0.d0, Ne_CB
			else
				MFP = 1.d99 ! electron with too low energy never makes impact ionization (imfp = infinity)
				t_next_e1 = 1.d15	! we stop following it, including for elastic collisions (emfp = infinity)
			endif

		endselect

		call Initialize_object(2, El, Ne_max, NOP, E1, t_cur+t_next_e1, t_cur) ! the next collison of the incoming electron will occur in t_next_e1 femtoseconds
	    
         endselect ! (koe)

         call Find_min_time_particle(Ph, El, Hol, koe, NOP, t_cur) ! finds which event will occur next
         if (koe.eq.2 .and. elastic_collision(nop) .eqv. .true.) koe=4	! the next collision of the electron number nop will be elastic
	 
      enddo ! (t_cur .LT. tim)
      El_distr(i) = El_distr(i) + Ne_CB ! total the number of particles, not normalized
      ! Save the values in the distributions on this timestep:
      select type (El)	! since it's free format, it has to be checked every time...
         type is (Electron)
            El_NRG(i) = El_NRG(i) + SUM(El(:)%E) ! total electron energy vs time, not normalized
            !print*, 'Trying to get electron distribution'
            if (print_distr) call electron_distribution(Ne_CB, Eph, i, El, Temp_El_distr_E, Temp_El_distr_E_grid)
            !print*, 'Got electron distribution'
      endselect
      select type (Hol)	! since it's free format, it has to be checked every time...
         type is (Hole)
            do j = 1, Atoms ! all spieces
               do k = 1, Shells ! all shells
                  !l = j + Atoms*(k-1) ! wrong!!
                  l = k + Shells*(j-1)
                  Hole_distr(l,i) = Hole_distr(l,i) + DBLE(COUNT((Hol%Sh .EQ. k) .AND. (Hol%KOA .EQ. j)))
               enddo
            enddo
      endselect

   enddo ! (tim .LT. Total_t)

!    do i = 1,size(Hole_distr,2)
!       print*, i, Hole_distr(:,i)
!    enddo

   if (print_casc)  close (fn) 
   
end subroutine MC



subroutine electron_distribution(Ne_CB, Eph, itim, El, Temp_El_distr_E, dE)
   integer, intent(in) :: Ne_CB ! number of electrons
   real(8), intent(in) :: Eph	! photon energy [eV]
   integer, intent(in) :: itim	! time-step number
   type(Electron), DIMENSION(:), intent(in) :: El	! arrays of electrons
   real(8), dimension(:,:), intent(inout) :: Temp_El_distr_E    ! electron distribution in the energy space vs time
   real(8), dimension(:), intent(inout) :: dE ! grid electron distribution in the energy space vs time
   integer :: j, i
    do i = 1,Ne_CB ! all electrons
       if (El(i)%E .GT. 0.0d0) then ! really existing
         call Find_in_monotonous_1D_array(dE, El(i)%E, j) ! find where in the distribution array it is
         if (j .GT. 1) then
            Temp_El_distr_E(itim,j) = Temp_El_distr_E(itim,j) + 1.0d0/(dE(j)-dE(j-1)) !/DBLE(Ne_CB) ! number of electrons in this ENERGY interval
         else    ! j = 1, energy interval is 1
            Temp_El_distr_E(itim,j) = Temp_El_distr_E(itim,j) + 1.0d0/dE(j) !/DBLE(Ne_CB) ! number of electrons in this ENERGY interval
         endif
      endif
   enddo
end subroutine electron_distribution


pure subroutine Find_in_monotonous_1D_array(Array, Value0, Number)
   REAL(8), dimension(:), INTENT(in) :: Array ! in which we are looking for the Value
   REAL(8), INTENT(in) :: Value0   ! to be found in the array as near as possible
   integer, INTENT(out) :: Number ! number of the element which we are looking for 
   integer i, N, i_cur, i_1, i_2, coun
   real(8) temp_val, val_1, val_2
   N = size(Array)
   i_1 = 1
   val_1 = Array(i_1)
   i_2 = N
   val_2 = Array(i_2)
   i_cur = FLOOR((i_1+i_2)*0.5d0)
   temp_val = Array(i_cur)
   if (isnan(Value0)) then
 !       print*, 'The subroutine Find_in_monotonous_1D_array'
  !      print*, 'cannot proceed, the value of Value0 is', Value0
   !     write(*, '(f25.16,f25.16,f25.16,f25.16)') Value0, Array(i_cur), Array(i_1), Array(i_2)
    !    stop 'STOPPED WORKING...'
   else
       if (Value0 .LT. Array(1)) then ! it's the first value, no need to search
           i_cur = 0
       else if (Value0 .GE. Array(N)) then ! it's the last value, no need to search
           i_cur = N-1
       else
           coun = 0
           do ! until the Value is in between Array(i_cur) and Array(i_cur+1) => we found i_cur
                if ((Value0 .GE. Array(i_cur)) .AND. (Value0 .LE. Array(i_cur+1))) exit ! when the Value is in between Array(i_cur) and Array(i_cur+1) => we found i_cur
                if (temp_val .LE. Value0) then
                   i_1 = i_cur
                   val_1 = Array(i_1)
                   i_cur = FLOOR((i_1+i_2)*0.5d0)
                   temp_val = Array(i_cur)
                else
                   i_2 = i_cur
                   val_2 = temp_val
                   i_cur = FLOOR((i_1+i_2)*0.5d0)
                   temp_val = Array(i_cur)
                endif
                coun = coun + 1
                if (coun .GT. 1.d3) then
 !                   print*, 'PROBLEM WITH CONVERGANCE IN'
  !                  print*, 'Find_in_monotonous_1D_array', coun
   !                 write(*, '(f25.16,f25.16,f25.16,f25.16)') Value0, Array(i_cur), Array(i_1), Array(i_2)
    !                stop 'STOPPED WORKING...'
                endif
           enddo
       endif
   endif    ! isnan
   Number = i_cur+1
end subroutine Find_in_monotonous_1D_array



!EEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEE
subroutine Electron_which_shell(Noc, Ee, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Error_message, KOA, Sh, IMFPs, Ritchi)
   integer, DIMENSION(:,:), intent(in) :: Noc	! number of the cross-section used
   real(8), intent(in) :: Ee 	! [eV] kinetic energy of the electron
   real(8), DIMENSION(:,:), INTENT(in) :: Ip_elements	! ionization potentials [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Ekin_elements	! kinetic energies [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Nel_elements	! numbers of electrons in shells
   integer, DIMENSION(:,:), INTENT(in) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements   
   integer, DIMENSION(:), INTENT(in) :: N_elements	! number of elements in 1 molecule
   integer, DIMENSION(:), INTENT(in) :: Z_elements ! atomic number of this element
   REAL(8), DIMENSION(:), INTENT(in) :: M_elements ! mass of this element
   real(8), intent(in) ::  At_dens	! atomic density [1/cm^3]
   !real(8), intent(out) :: IMFP		! [A] inelastic mean free path
   integer, intent (out) :: KOA, Sh ! kind of atom and shell
   type(Error_handling), optional, intent(inout) :: Error_message ! deals with errors, if any
   type(MFP_from_file), dimension(:,:), intent(inout), optional :: IMFPs	! inelastic mean free paths from files
   type(CDF), dimension(:,:), intent(inout), optional :: Ritchi    ! Ritchi CDF to use for cross-sections
   character(100) Writing_var
   real(8) RN, Sigma_rel, Sigma_tot, Sigma, Sigma_sum, IMFP
   integer i, j, N, M
   N = size(Ip_elements,1) ! number of elements
   M = size(Ip_elements,2) ! number of shells

   !call Total_cross_section(Noc, Ee, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFP)
   call Total_cross_section(Noc, Ee, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, Ritchi)
       !Total_cross_section(Noc, T, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFP)
   call random_number(RN)
   Sigma_rel = RN*Sigma_tot ! this one is realized now, find which shell it is:
   Sigma_sum = 0.0d0 ! to start the search
   do i = 1, N ! all spiecies
      KOA = i ! this kind of atoms
      do j = 1, M ! all shells
         if ((Nel_elements(i,j) .GT. 0.0d0) .AND. (Ee .GT. Ip_elements(i,j))) then
            Sh = j ! this shell
            call Cross_section(Noc, Ee, i, j, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, Sigma, IMFPs, At_dens, Ritchi)
            select case (Noc(i,j))
               !case (3)
               case (2,3)
                  Sigma_sum = Sigma_sum + Sigma !*N_elements(j)/SUM(N_elements) ! [A^2] cross-section
               case default
                  Sigma_sum = Sigma_sum + Sigma*N_elements(i)/SUM(N_elements)
            endselect
!             write(*,'(a,f25.16,f25.16,f25.16,i12,i12,f25.16,f25.16)') 'S', Sigma_sum, Sigma_rel, Sigma_tot, i, j, Ee, Ip_elements(i,j)
         endif
         if (Sigma_sum .GT. Sigma_rel) exit
      enddo ! j
      if (Sigma_sum .GT. Sigma_rel) exit
   enddo ! i

!    print*, ANY((Ip_elements .LT. Ee) .AND. Ip_elements .GT. 0.0d0)
!    print*, KOA, Sh, Ip_elements( KOA, Sh), Ee
!    stop
   if (Sigma_sum .LT. Sigma_rel) then
      print*, '============Strange...============'
      print*, '*' ,Sigma_sum, Sigma_rel, '*'
   endif

end subroutine Electron_which_shell



!PPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPP
subroutine Select_shell_photoabsorbtion(Photoabs_sigma_tot, Photoabs_sigma, koa, shl, Error_message)
   real(8), DIMENSION(:,:), intent(in) :: Photoabs_sigma ! [A^2] photoionization cross-sections for all shells
   real(8), intent(in) :: Photoabs_sigma_tot	! [A^2] total photoabsorbtion cross-section
   integer, intent(out) :: koa	! kind of atom
   integer, intent(out) :: shl	! shell
   type(Error_handling), optional, intent(inout) :: Error_message ! deals with errors, if any
   real(8) RN, Sigma, S
   integer i,j, N, M
   character(100) Writing_var

   call random_number(RN)
   Sigma = RN*Photoabs_sigma_tot ! realized cross-section
   i = 1
   j = 1
   koa = i
   N = size(Photoabs_sigma,1)
   M = size(Photoabs_sigma,2)
   S = 0.0d0
   do while (S .LT. Sigma)
      if (j .GT. M) then ! last shell, switch to the next element
         j = 1
         if (i .LE. N) then ! scanning all elements
            i = i + 1
         else
            Writing_var = 'All elements are finished, but no cross-section...'
            write(*,'(a,i12,i12,i12,i12)') Writing_var, i, j, N, M
            call Save_error_details(Error_message, 4, Writing_var)
         endif 
      endif
      S = S + Photoabs_sigma(i,j)
      shl = j
      koa = i
      j = j + 1	! next shell
   enddo
end subroutine Select_shell_photoabsorbtion


subroutine NRG_of_emmited_electron(Ip_elements, koa, shl, dE, Eel, Error_message)
   real(8), DIMENSION(:,:), intent(in) :: Ip_elements ! [eV] ionization potentials of all shells, all elements
   integer, intent(in) :: shl, koa	! shell number, and kind of atom
   real(8), intent(in) :: dE 	! transferred energy [eV]
   real(8), intent(out) :: Eel	! emmited energy [eV]
   type(Error_handling), optional, intent(inout) :: Error_message ! deals with errors, if any
   character(100) Writing_var
   real(8) Ip
   Ip = Ip_elements(koa, shl)
   if ((dE - Ip) .GE. 0.0d0) then
      Eel = dE - Ip
   else
      Writing_var = 'Energy of an emmited electron is negative!'
      print*, Writing_var
      call Save_error_details(Error_message, 3, Writing_var)
   endif
end subroutine NRG_of_emmited_electron



pure subroutine Find_min_time_particle(Ph, El, Hol, koe, NOP, t_cur)
   class(*), DIMENSION(:), intent(in), ALLOCATABLE :: Ph, El, Hol 	! arrays of particles as objects
   integer, intent(out) :: koe	! kind of particle
   integer, intent(out) :: NOP	! number of particle of this kind
   real(8), intent(out) :: t_cur	! time of collision of this particle
   integer i
      select type (Ph)
         type is (Photon)
         !write(*,'(e25.16)', advance = 'NO') minval(Ph%ti) !, minval(El%ti), minval(Hol%ti)
         select type (El)
            type is (Electron)
            !write(*,'(e25.16)', advance = 'NO') minval(El%ti) !, minval(Hol%ti)
            select type (Hol)
               type is (Hole) 
               koe = transfer( minloc((/minval(Ph%ti),minval(El%ti),minval(Hol%ti)/)) , i) ! kind of event determines the next select case: order of arrays must coinside with select-case!!!
               select case (koe)
                  case (1) ! photon excites new electron
                     NOP =  transfer( minloc(Ph%ti), i)  ! which photon
                     t_cur = minval(Ph%ti)	! impact time ti becomes the real time -- impact really occurs
                  case (2) ! electron excites new electron = impact ionization = inelastic collision; or electron is scattered without energy change = elastic collision
                     NOP =  transfer( minloc(El%ti), i)	 ! which electron from array
                     t_cur = minval(El%ti)	! impact time ti becomes the real time -- impact really occurs
                  case (3) ! hole decays = Auger recombination
                     NOP =  transfer( minloc(Hol%ti), i) ! which hole from array
                     t_cur = minval(Hol%ti)	! impact time ti becomes the real time -- impact really occurs
               endselect

               !write(*,'(e25.16,e25.16,e25.16,i12)', advance = 'YES') minval(Hol%ti), min(minval(Ph%ti),minval(El%ti),minval(Hol%ti)), minval((/minval(Ph%ti),minval(El%ti),minval(Hol%ti)/)), minloc((/minval(Ph%ti),minval(El%ti),minval(Hol%ti)/))
!                print*, koe, NOP, t_cur
            endselect
         endselect
      endselect
end subroutine Find_min_time_particle


!HHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHH
subroutine Auger_electron(KOA, SHL, Ip_elements, Nel_elements, Sh1, KOA1, Sh2, KOA2, Ee, Error_message)
   integer, intent(in) :: KOA, SHL ! atomic species and shell numbers
   real(8), DIMENSION(:,:), intent(in) :: Ip_elements ! [eV] ionization potentials of all shells, all elements
   real(8), DIMENSION(:,:), intent(in) :: Nel_elements 	! number of electrons in all shells, all elements
   integer, intent(out) :: KOA1, Sh1	! shell and atom to which deeper hole jumps up
   integer, intent(out) :: KOA2, Sh2	! shell and atom in which the hole is produced
   real(8), intent(out) :: Ee	! energy of ejected electron [eV]
   type(Error_handling), optional, intent(inout) :: Error_message ! deals with errors, if any
   character(100) Writing_var
   real(8) RN, Energy_diff, coun, Shel
   integer i, j, N, M, iter

   N = size(Ip_elements,1) ! number of elements
   M = size(Ip_elements,2) ! number of shells

   Sh1 = SHL
   Sh2 = 0
   KOA1 = KOA
   KOA2 = 0
   Ee = 0.d0
   iter = 0 ! count nmber of iterations
   coun = 0.d0
   do while (coun .LE. 0.d0)
      iter = iter + 1 ! count nmber of iterations
      !1111111111111111111111111111111111111111111111111111111111111111111111111111
      ! Hole one:
      ! count how many electrons on the shells that can participate in Auger-decay:
      coun = count_for_Auger_shells(Nel_elements, Ip_elements, Ip_elements(KOA, SHL))

      call random_number(RN)
      !RN = 0.9999999d0
      Shel = RN*coun ! this shell is where the hole jumps up
!       print*, 'one', coun, Shel

      call Choose_for_Auger_shell(Nel_elements, Ip_elements, Ip_elements(KOA, SHL), Shel, Sh1, KOA1)

      Energy_diff = Ip_elements(KOA, SHL) - Ip_elements(KOA1, Sh1)
!       print*, 'Energy_diff=', Energy_diff, KOA, SHL, KOA1, Sh1
      !2222222222222222222222222222222222222222222222222222222222222222222222222222
      ! Hole two:
      ! count how many electrons on the shells that can participate in Auger-decay:
      coun = count_for_Auger_shells(Nel_elements, Ip_elements, Energy_diff)

      if (coun .GT. 0) then
         call random_number(RN)
         Shel = RN*coun ! this shell is where the hole jumps up
!          print*, 'two', coun, Shel

         call Choose_for_Auger_shell(Nel_elements, Ip_elements, Energy_diff, Shel, Sh2, KOA2)

         Ee = Energy_diff - Ip_elements(KOA2,Sh2) 
      endif ! (coun .GT. 0)

      if (iter .GE. 100) then
         Writing_var = 'Impossible Auger-decay'
         write(*,'(a,i12,i12,i12,i12)') Writing_var, i, j,KOA, SHL, KOA1, Sh1 
         print*, Ip_elements(KOA, SHL), Ip_elements(KOA1, Sh1)
         call Save_error_details(Error_message, 5, Writing_var)
      endif
      if (iter .GE. 100) exit
   enddo ! while
end subroutine Auger_electron


pure function count_for_Auger_shells(Nel_elements, Ip_elements, NRG)
   real(8) count_for_Auger_shells
   real(8), DIMENSION(:,:), intent(in) :: Ip_elements ! [eV] ionization potentials of all shells, all elements
   real(8), DIMENSION(:,:), intent(in) :: Nel_elements 	! number of electrons in all shells, all elements
   real(8), intent(in) :: NRG
   integer i, j, N, M
   N = size(Ip_elements,1) ! number of elements
   M = size(Ip_elements,2) ! number of shells
   count_for_Auger_shells = 0.d0
   do i = 1, N
      do j = 1, M
         if ((Ip_elements(i,j) .LT. NRG) .AND. (Ip_elements(i,j) .GT. 0.0d0)) then ! it is possible
            count_for_Auger_shells = count_for_Auger_shells + Nel_elements(i, j)
         endif
      enddo
   enddo
end function count_for_Auger_shells


pure subroutine Choose_for_Auger_shell(Nel_elements, Ip_elements, NRG, Shel, Sh1, KOA1)
   real(8), DIMENSION(:,:), intent(in) :: Ip_elements ! [eV] ionization potentials of all shells, all elements
   real(8), DIMENSION(:,:), intent(in) :: Nel_elements 	! number of electrons in all shells, all elements
   real(8), intent(in) :: NRG
   real(8), intent(in) :: Shel
   integer, intent(out) :: Sh1, KOA1
   integer i, j, N, M
   real(8) coun_sh
   N = size(Ip_elements,1) ! number of elements
   M = size(Ip_elements,2) ! number of shells
   coun_sh = 0.d0
   do i = 1, N
      do j = 1, M
         if ((Ip_elements(i,j) .LT. NRG) .AND. (Ip_elements(i,j) .GT. 0.0d0)) then ! it is possible
            coun_sh = coun_sh + Nel_elements(i, j)
            !write(*,'(a,i12,i12,i12,i12,i12)') 'A', coun, Shel, coun_sh, i, j
            if (coun_sh .GE. Shel) then 
               Sh1 = j ! this is the shell
               KOA1 = i
            endif ! (coun_sh .EQ. Shel) then
         endif ! (Ip_elements(i,j) .LT. Ip_elements(KOA, SHL)) then ! it is possible
         !if (coun_sh .EQ. Shel) print*, i, j, Shel, coun_sh, 'one'
         if (coun_sh .GE. Shel) exit
      enddo ! j
      !if (coun_sh .EQ. Shel) print*, i, j, Shel, coun_sh, 'two'
      if (coun_sh .GE. Shel) exit
   enddo ! i
end subroutine Choose_for_Auger_shell


function Auger_decay_sampled(KOA, SHL, t_Auger)
   real(8) Auger_decay_sampled	! [fs] sampled Auger decay time
   integer, intent(in) :: KOA, SHL ! atomic species and shell numbers
   real(8), dimension(:,:), intent(in) :: t_Auger	! table of all auger-decay times [fs]
   real(8) RN
   call random_number(RN)
   Auger_decay_sampled = -log(RN)*t_Auger(KOA, SHL) ! [fs] with Poisson distribution
end function Auger_decay_sampled 


!GGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGG
! A collision accures with a particle:
subroutine Particle_event(Particl, N)
   class(*), DIMENSION(:), ALLOCATABLE :: Particl	! photon, or electron, or hole
   integer, intent (in) :: N	! number of particle from the array
   select type (Particl)
      type is (Photon)
         print*, 'Photon', Particl(N)%ti
      type is (Electron)
         print*, 'Electron'
      type is (Hole)
         print*, 'Hole'
   end select
end subroutine Particle_event



pure subroutine Distribbutions_update(Ne_max, N_shell_max, El_distr, Hole_distr, i, ne, nh, El_distr_over, El_E_distr_over, Hole_distr_over)
   integer, optional, intent (in) :: Ne_max,N_shell_max ! dimensions
   class(Electron_density), intent(inout) :: El_distr ! electron density in time
   class(Hole_density), intent(inout) :: Hole_distr   ! holes densities in time
   integer, optional, intent (in) :: i	! number of the element in the array
   real(8), optional, intent (in) :: ne	! chenge in the electron density
   real(8), dimension(:), optional, intent (in) :: nh	! chenge in the hole density 
   real(8), dimension(:), optional, intent (in) :: El_distr_over ! electron distribution to overwrite
   real(8), dimension(:), optional, intent (in) :: El_E_distr_over ! electron energy distribution
   real(8), dimension(:,:), optional, intent (in) :: Hole_distr_over ! hole distributions to overwrite
   ! allocate electron distribution:
   if (.not. allocated(El_distr%t)) allocate(El_distr%t(Ne_max))
   if (.not. allocated(El_distr%ne)) allocate(El_distr%ne(Ne_max))
   if (.not. allocated(El_distr%Ee)) allocate(El_distr%Ee(Ne_max))
   ! allocate holes distributions:
   if (.not. allocated(Hole_distr%t)) allocate(Hole_distr%t(Ne_max))
   if (.not. allocated(Hole_distr%nh)) allocate(Hole_distr%nh(N_shell_max,Ne_max))
   if (.not. allocated(Hole_distr%Eh)) allocate(Hole_distr%Eh(N_shell_max,Ne_max))

   if (present(El_distr_over)) El_distr%ne = El_distr_over
   if (present(El_distr_over)) El_distr%Ee = El_E_distr_over
   if (present(Hole_distr_over)) Hole_distr%nh = Hole_distr_over
end subroutine Distribbutions_update


pure subroutine Energy_distributions_update(N_tim, Eph, El_distr)  ! MC
   integer, intent(in) :: N_tim
   real(8), intent(in) :: Eph  ! [eV]
   class(Electron_density), intent(inout) :: El_distr ! electron density in time
   integer :: N, Ord, Va, i
   real(8) :: Emin, Emax, dE0

   if (.not. allocated(El_distr%Ee_vs_t)) then
    Emin = 1.0d0
    N = 0
    Ord = 0 ! start with 1
    Va = int(Emin)
    dE0 = Emin
    Emax = Eph
    do while (dE0 .LT. Emax)
        N = N + 1
        if (Va .GE. 100) then
            Va = Va - 90
            Ord = Ord + 1
        endif
        dE0 = dE0 + 10.0d0**Ord
        Va = Va + 1
    enddo   ! while (dE .LT. Emax)
    N = N + 1

    allocate(El_distr%Ee_vs_t_grid(N))
    El_distr%Ee_vs_t_grid = 0.0d0
    allocate(El_distr%Ee_vs_t(N_tim,N))
    El_distr%Ee_vs_t = 0.0d0

    Emin = 1.0d0    ! [eV] we start with this minimum
    Ord = 0 ! start with 1
    Va = int(Emin)
    El_distr%Ee_vs_t_grid(1) = Emin
    do i = 1, N-1
        if (Va .GE. 100) then
            Va = Va - 90
            Ord = Ord + 1
        endif
        El_distr%Ee_vs_t_grid(i+1) = El_distr%Ee_vs_t_grid(i) + 10.0d0**Ord
        Va = Va + 1
        !print*, i, El_distr%Ee_vs_t_grid(i) 
    enddo
   endif
end subroutine Energy_distributions_update


subroutine account_for_elastic_collisions(N_elements, Z_elements,at_dens,sigma_in,Ekin,whether_elastic_collision,fft)

   integer, DIMENSION(:), intent(in) :: N_elements	! numbers of elements in 1 molecule
   integer, DIMENSION(:), intent(in) :: Z_elements	! atomic numbers of elements
   real(8), intent(in) :: At_dens		! [1/cm^3] atomic density
   real(8), intent(INout) :: sigma_in		! [A^2] inelastic cross section; output is the total cross section, taking into account both elastic and inelastic collisions
   real(8), intent(IN) :: Ekin			! eV, current kinetic energy of the electron
   logical, intent(out) :: whether_elastic_collision	! the result is whether the next collision for a certain particle will be elastic
   real(8), intent(out) :: fft		! [fs] the time until the next collision, taking into account elastic and inelastic cross sections, i.e., free flight time

    real(8) sigma_el, sigma_tot_el_in, rn, EMFP, IMFP, MFP

	sigma_el = Total_elastic_cross_section(N_elements, Z_elements,Ekin,1.d0)
	sigma_tot_el_in = sigma_in+sigma_el

	if (sigma_el .gt. 1.d-40) then
		EMFP = 1.d0/(at_dens*sigma_el)*1.d24 		! n*lambda*sigma_tot = 1
	else 
		EMFP = 1.d99		! the elastic mean free path is infinite when the cross section is zero		
	endif

	if (sigma_in .gt. 1.d-40) then
		IMFP = 1.d0/(at_dens*sigma_in)*1.d24 		! n*lambda*sigma_tot = 1
	else
		IMFP = 1.d99		! the inelastic mean free path is infinite when the cross section is zero
	endif

	MFP = 1.d0/(1.d0/IMFP+1.d0/EMFP)	! total mean-free path: 1/mfp = 1/imfp + 1/emfp

	if (Ekin .gt. 1.d-40) then
		fft = MFP/dSQRT(Ekin*2.d0*g_e/(g_me))*1.d5 ! [fs] free-flight time of an electron taking into account both elastic and inelastic collisions
	else
		fft = 1.d99
	endif
	
	! checking if elastic collision will occur at the next step
	call random_number(RN)
	if (sigma_tot_el_in .gt. 1.d-40) then
		if (RN.lt.sigma_el/sigma_tot_el_in) then 
			whether_elastic_collision = .true. 		! for the particle number NOP, the next collision will be elastic:
		else 
			whether_elastic_collision = .false. 	! for the particle number NOP, the next collision will be inelastic:
		endif
	else 
		whether_elastic_collision = .false. 	! for the particle number NOP, the next collision will be inelastic:
	endif

	sigma_in = sigma_tot_el_in
	
end subroutine account_for_elastic_collisions



END MODULE Monte_Carlo 
