!    XCASCADE software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains subroutines to deal with laser pulse parameters
MODULE Laser_pulse_parameters
implicit none

contains
!LLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLL
! Laser pulse and photons: 

function Photon_arrival_time(KOP, mu, sigma, RN)
   real(8) Photon_arrival_time ! [fs] arrival time of the photon
   integer, intent(in) :: KOP	! which kind of pulse we use
   real(8), intent(in) :: mu, sigma  ! pulse max and pulse duration
   real(8), optional, intent(in) :: RN
   real(8) Tim, dt, Int_pulse, y, last_t, this_t, RN2, lim_Tim
   integer i, j

   if (present(RN)) then
      y = RN
   else
      call random_number(y)
   endif
   lim_Tim = mu + 3.0d0*sigma
   dt = sigma*0.01d0
   Tim = mu - 3.0d0*sigma
!    print*, '============='
!    print*, mu, sigma, dt, y
!    print*, KOP, mu, sigma, Tim, lim_Tim
!    print*, '============='
   Int_pulse = 0.0d0
   last_t = 0.0d0
   do while (Int_pulse .LT. y)
      Tim = Tim + dt
      this_t = Pulse_shape(KOP, mu, sigma, Tim)
      Int_pulse = Int_pulse + dt*(this_t + last_t)*0.5d0
      last_t = this_t
      if (Tim .GT. lim_Tim) then ! just for case if numerical integration isn't perfect:
         call random_number(RN2)
         Photon_arrival_time = mu - 3.0d0*sigma + 6.0d0*sigma*RN2
!          stop 'Aborted timeline...'
      endif
      if (Tim .GT. lim_Tim) exit
      Photon_arrival_time = Tim
      !print*, Tim, Int_pulse, this_t, last_t
   enddo
   call random_number(RN2)
   Photon_arrival_time = Photon_arrival_time - 0.5d0*dt + dt*RN2
!    print*, 'Int_pulse', Photon_arrival_time, Int_pulse, y
end function Photon_arrival_time


pure function Pulse_shape(KOP, mu, sigma, x)
   REAL(8) Pulse_shape
   integer, intent(in) :: KOP
   real(8), intent(in) :: mu, sigma, x
   select case (KOP)
      case(:0)
         Pulse_shape = Rect_pulse(mu, sigma, x)
      case(1)
         Pulse_shape = Gaus_pulse(mu, sigma, x)
      case(2)
         Pulse_shape = SASE_pulse(mu, sigma, x)
      case default
         Pulse_shape = Gaus_pulse(mu, sigma, x)
   end select
end function Pulse_shape 


pure function Rect_pulse(mu, sigma, x) ! number of photons at this timestep x, according to rectangular pulse of constant intensity
   real(8) Rect_pulse
   real(8), intent(in) :: mu, sigma, x
   real(8) Gaus_sigma
   Gaus_sigma = sigma*2.35482d0
   if ((x .GT. (mu - Gaus_sigma*0.5d0)) .AND. (x .LE. (mu + Gaus_sigma*0.5d0)) ) then
      Rect_pulse = 1.0d0/Gaus_sigma
   else
      Rect_pulse = 0.0d0
   endif
end function Rect_pulse

pure function Gaus_pulse(mu, sigma, x) ! number of photons at the time x according to Gaussian shape
   real(8) Gaus_pulse
   real(8), intent(in) :: mu, sigma, x
   real(8), parameter :: g_Pi = 3.1415926535897932384626433832795d0
   if ((x .GT. (mu - 2.5d0*sigma)) .AND. (x .LE. (mu + 2.5d0*sigma)) ) then
      Gaus_pulse = 1.0d0/(dsqrt(2.0d0*g_Pi)*sigma)*dexp(-(x-mu)*(x-mu)/(2.0d0*sigma*sigma))
   else
      Gaus_pulse = 0.0d0
   endif
end function Gaus_pulse

pure function SASE_pulse(mu, sigma, x) ! number of photons at the time x according to schematic SASE shape
   real(8) SASE_pulse
   real(8), intent(in) :: mu, sigma, x
   real(8), parameter :: g_Pi = 3.1415926535897932384626433832795d0
   integer i
   real(8) RN(4), Co(4), y, f, SASE
   if ((x .GT. (mu - sigma)) .AND. (x .LE. (mu + sigma)) ) then
      RN(1) = 0.4563449303d0
      RN(2) = 0.1271999433d0
      RN(3) = 1.0d0 - RN(1)
      RN(4) = 1.0d0 - RN(2)
      Co(1) = 2.0d0
      Co(2) = 4.0d0
      Co(3) = 5.0d0
      Co(4) = 9.0d0
      SASE = 0.0d0
      do i = 1,4
         y = Co(i)*g_Pi*(x+sigma-mu)/sigma
         f = sin(y)
         SASE = SASE + f*f/sigma*RN(i)
      enddo
      SASE_pulse = SASE*0.5d0 ! normalization to 1
   else
      SASE_pulse = 0.0d0
   endif
end function SASE_pulse

!LLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLLL

END MODULE Laser_pulse_parameters 