!    XCASCADE software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains subroutines to deal with files
MODULE Files_dealing
use Constants   ! universal constants 
use Objects_and_types	! object oriented stuff
implicit none

 contains

!IIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIIII
! Input files:
subroutine READING_INPUT_FILES(NoEiC, Z_elements, M_elements, N_elements, Ip_elements, whether_valence, Ekin_elements, Nel_elements, NelVB_elements, PQN_elements, Shell_elements, t_Auger, Photoabs_sigma, Dens, Effgap, Egap, mheff, Eph, Fluence, KOP, KOPulse, Pulse_max, Pulse_duration, Input_file_name, Error_occured, incl_elastic, print_distr, print_casc, Ecut, NMC, Num_th, Total_t, Global_dt, Element, Ritchi, CSN, IMFPs)
   integer, intent(out) :: NoEiC ! number of elements in the compound 
   real(8), intent(out) :: Dens  ! density [g/cm^3]
   integer, DIMENSION(:), ALLOCATABLE, INTENT(out) :: Z_elements ! atomic number of this element
   REAL(8), DIMENSION(:), ALLOCATABLE, INTENT(out) :: M_elements ! mass of this element
   integer, DIMENSION(:), ALLOCATABLE, INTENT(out) :: N_elements ! relative number of atoms in 1 molecule
   real(8), DIMENSION(:,:), ALLOCATABLE, INTENT(out) :: Ip_elements	! ionization potentials [eV]
   logical, DIMENSION(:,:), ALLOCATABLE, INTENT(out) :: whether_valence	! ionization potentials [eV]
   real(8), DIMENSION(:,:), ALLOCATABLE, INTENT(out) :: Ekin_elements	! kinetic energies [eV]
   real(8), DIMENSION(:,:), ALLOCATABLE, INTENT(out) :: Nel_elements	! numbers of electrons in shells
   real(8), DIMENSION(:), ALLOCATABLE, INTENT(out) :: NelVB_elements	! numbers of VALENCE electrons in shells
   integer, DIMENSION(:,:), ALLOCATABLE, INTENT(out) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements
   character(11), DIMENSION(:,:), ALLOCATABLE, intent(out) :: Shell_elements 	! names of all shells, all elements
   real(8), DIMENSION(:,:), ALLOCATABLE, INTENT(out) :: Photoabs_sigma ! [A^2] photoionization cross-sections for all shells
   real(8), DIMENSION(:,:), ALLOCATABLE, INTENT(out) :: t_Auger	! Auger-decays times [fs]
   real(8) :: Photoabs_sigma_TEMP ! [A^2] total photoionization cross-sections
   real(8), intent(out) :: Effgap	! [eV] effective bandgap
   real(8), intent(out) :: Egap	! [eV] real bandgap, if given
   real(8), intent(out) :: mheff	! [m_e] effective mass of a hole, if given
   real(8), intent(out) :: Eph	! [eV] photon energy
   real(8), intent(out) :: Fluence	! [J/cm^2] photon fluence
   integer, intent(out) :: KOP	! kind of particle impacting: photon vs electron
   integer, intent(out) :: KOPulse	! shape of pulse: 0=flat-top, 1=Gaussian
   real(8), intent(inout) :: Pulse_max ! position of the maximum of the mulse (mu-gauss)
   real(8), intent(inout) :: Pulse_duration ! FWHM of the mulse (will be transferred to sigma-gauss)
   character(100), intent(out) :: Input_file_name
   type(Error_handling), intent(inout) :: Error_occured ! deals with errors, if any
   logical, intent (out) :: incl_elastic
   logical, intent (out) :: print_distr
   logical, intent (out) :: print_casc
   real(8), intent (out) :: Ecut
   integer, intent (out) :: NMC
   integer, intent (out) :: Num_th
   real(8), intent(out) :: Total_t, Global_dt
   type(Atom), DIMENSION(:), ALLOCATABLE, INTENT(inout) :: Element	! data about all elements
   type(CDF), dimension(:,:), allocatable, intent(inout), optional :: Ritchi    ! Ritchi CDF to use for cross-sections
   integer, DIMENSION(:,:), allocatable, intent(out) :: CSN	! number of cross-section used
   type(MFP_from_file), dimension(:,:), allocatable, intent(inout), optional :: IMFPs	! inelastic mean free paths from files
   !===========================
   character(100), DIMENSION(:), ALLOCATABLE :: File_names_all
   character(100) File_name, File_name_EADL, File_name_EPDL, File_name3, Error_descript, CDF_parameter_file
   integer i, j, k, FN, FN1, FN2, FN3, Nsh, Nshells, Z, size_DTBS(2), NOSHs, End_file, READ1
   real(8) Photoabs_sigma_tot, NVB_el
   logical file_exist
   integer io,io2,io3
   real(8) E_min

   FN = 333
   FN2 = 444
   FN3 = 555
   inquire(file='INPUT_material.txt',exist=file_exist)
   if (file_exist) then
      open(unit = FN, FILE = 'INPUT_material.txt', status = 'old')
   else
      File_name = 'OUTPUT_Error_log.dat'
      open(unit = 100, FILE = trim(adjustl(File_name)))
      Error_occured%File_Num = 100	! file number with error-log
      Error_descript = 'File INPUT_material.txt is not found!'
      call Save_error_details(Error_occured, 1, Error_descript)
      print*, trim(adjustl(Error_descript))
      stop 'The program is going to crash now, sorry... Press return.'
   endif
   read(FN,*) Input_file_name	! read file name
   File_name = 'OUTPUT_'//trim(adjustl(Input_file_name))//'_Error_log.dat'
   !~ open(unit = 100, FILE = trim(adjustl(File_name)))
   Error_occured%File_Num = 100	! file number with error-log

   Photoabs_sigma_tot = 0.0d0

   ! Read material properties:
   File_name3 = 'Material_parameters/'//trim(adjustl(Input_file_name))//'.txt'
   inquire(file=trim(adjustl(File_name3)),exist=file_exist)
   open(unit = FN3, FILE = trim(adjustl(File_name3)), status = 'old')
!    print*, file_exist
   read(FN3,*) NoEiC	! read number of elements in the compound
   allocate(Z_elements(NoEiC))	! this many elements, allocate array
   allocate(M_elements(NoEiC))	! this many elements, allocate array
   allocate(N_elements(NoEiC))	! this many elements, allocate array
   allocate(NelVB_elements(NoEiC))	! this many VB-electrons elements, allocate array
   allocate(File_names_all(NoEiC)) ! this many elements, this many filenames
   do i = 1, NoEiC	! for each element, read data:
      read(FN3,*) Z_elements(i), N_elements(i)  ! atomic number Z, contribution
   enddo
   read(FN3,*, IOSTAT=io) Dens
   !call Sort_array_deccending(Z_elements, M_elements, N_elements) ! sort elements
   call Sort_array_deccending(Z_elements, M_elements, N_elements, NelVB_elements) ! sort elements
   do i = 1, NoEiC
      call Find_element_name(Z_elements(i), Element, i, NoEiC, NelVB_elements(i))
      M_elements(i) = Element(i)%M	! atomic mass
!       print*, i, Element(i)%Z, Element(i)%M, Element(i)%Name, NelVB_elements(i)
   enddo
!    stop 'READ' 

   !read(FN3,*, IOSTAT=io) Effgap	! [eV] effective bandgap
   Effgap = -1.d0	! switch off effective bandgap, use pure atomistic approximation
   
   if (io .gt. 0) then
	print *, 'something went wrong...'
	stop
   endif

   read(FN3,*, IOSTAT=io2) Egap	! [eV] real bandgap 
   
   if (io2 .gt. 0) then
	print *, 'something went wrong...'
	stop
   elseif (io2 .lt. 0) then
	Egap = 0.d0
	mheff = 0.d0
   else
	read(FN3,*, IOSTAT=io3) mheff	! [eV] real bandgap 
	if (io3 .lt. 0) then
		mheff = 0.d0
	endif
   endif

   ! Read back from the INPUT-file all other numerical parameters:
   read(FN,*) KOP	! 1 = photon, 2 = electron instead of laser pulse
   read(FN,*) Eph	! [eV] photon energy
   if (Eph .le. 0.d0) stop "Energy of the incident particle must be positive!"
   read(FN,*) Fluence	! [J/cm^2] incoming photon fluence
	   ! if ((Fluence .LE. 0) .AND. (KOP .EQ. 1)) then ! if no photons incoming
	      ! KOP = 2	! the start with an electron
	   ! endif
   read(FN,*) KOPulse	! shape of pulse: 0=flat-top, 1=Gaussian
   Pulse_max = 0.d0 
   !~ read(FN,*) Pulse_max ! position of the maximum of the mulse (mu-gauss)
   read(FN,*) Pulse_duration ! FWHM of the mulse (sigma-gauss)
   Pulse_duration = Pulse_duration/2.355d0	! transfer FWHM into Sigma of Gaussian
   read(FN,*) Total_t	! Time of analysis [fs]
   read(FN,*) Global_dt	! Timestep [fs], with which you will to have the time resolution
   
   incl_elastic = .false. ! don't
   read(FN,*) NMC	! include the effect of elastic collisions? (1=yes,0=no)
   if (NMC .EQ. 1) then
      incl_elastic = .true. ! include
   else if (NMC .EQ. 0) then
      incl_elastic = .false. ! don't
   else
      stop "It is not correctly specified in the input file whether we should include elastic corrections"
   endif

   
   read(FN,*) NMC	! printout electron distribution? (1=yes,0=no)
   if (NMC .EQ. 1) then
      print_distr = .true. ! print it
   else
      print_distr = .false. ! don't
   endif
   
   print_casc = .false. ! don't
   Ecut = -1.d0
   read(FN,*) NMC, Ecut	! printout all microscopical details of the cascades in cascades/ ? (1=yes,0=no)
    if (NMC .EQ. 1) then
      print_casc = .true. ! print it
   else
      print_casc = .false. ! don't
   endif
  
  
   read(FN,*) NMC		! number fo MC iterations
   read(FN,*) Num_th		! number of threads for OPENMP
   File_name_EADL = "EADL2017.all"
   File_name_EPDL = "EPDL2017.all"
   !~ read(FN,*) File_name_EADL 	! read EADL file name
   !~ read(FN,*) File_name_EPDL	! read EPDL file name
   

   write(*,'(a)') '*********************************************'
   write(*,'(a,a)') 'The calculations will be performed for ', trim(adjustl(Input_file_name))
   if (KOP .eq. 1) write(*,'(a,f8.2,a)') 'The photon energy is ', Eph, ' [eV]'
   if (KOP .eq. 2) write(*,'(a,f8.2,a)') 'The incident electron energy is ', Eph, ' [eV]'
   if (Fluence .ge. 0.01d0) then
	write(*,'(a,f8.2,a)') 'The incoming fluence is ', Fluence, ' [J/cm^2]'
	write(*,'(a,f8.2,a)') 'The pulse duration is ', Pulse_duration*2.355d0, ' [fs] (FWHM)'
	write(*,'(a,f8.2,a)') 'with the maximum at ', Pulse_max, ' [fs]'
   else if (Fluence .lt. 0.01d0 .and. fluence .gt. 0.d0) then
	write(*,'(a,e9.2,a)') 'The incoming fluence is ', Fluence, ' [J/cm^2]'
	write(*,'(a,f8.2,a)') 'The pulse duration is ', Pulse_duration*2.355d0, ' [fs] (FWHM)'
	write(*,'(a,f8.2,a)') 'with the maximum at ', Pulse_max, ' [fs]'
   else
      if (KOP .eq. 1) write(*,'(a)') 'Absorption of a single photon is simulated'
      if (KOP .eq. 2) write(*,'(a)') 'The cascade starts with a single incident electron'
   endif
   if (Ecut .ge. 0.d0) write(*,'(a,f8.2,a)') 'Kinetic energy cutoff for cascades is  ', Ecut, ' [eV]'
   write(*,'(a)') '*********************************************' 

   FN1 = 336
   inquire(file=trim(adjustl(File_name_EADL)),exist=file_exist) ! check if input file is there
   if (.not. file_exist) then
      write(Error_descript,'(a,a,a)') 'File ',trim(adjustl(File_name_EADL)),' is not found.'
      !Writing_var = 'File '//trim(adjustl(File_name_EADL))//' is not found.'
      call Save_error_details(Error_occured, 2, Error_descript)
      print*, trim(adjustl(Error_descript))
      stop 'The program is going to crash now, sorry...'
   endif
   inquire(file=trim(adjustl(File_name_EPDL)),exist=file_exist) ! check if input file is there
   if (.not. file_exist) then
      write(Error_descript,'(a,a,a)') 'File ',trim(adjustl(File_name_EPDL)),' is not found.'
      call Save_error_details(Error_occured, 2, Error_descript)
      print*, trim(adjustl(Error_descript))
      stop 'The program is going to crash now, sorry...'
   endif

   File_name3 = 'OUTPUT_'//trim(adjustl(Input_file_name))//'_Atomic_parameters.dat'
   open(102, FILE = trim(adjustl(File_name3)), action="write", status="replace")

   do i = 1, NoEiC	! for each element, read data:
      Z = Z_elements(i) 

      call READ_EADL_TYPE_FILE_int(FN1,File_name_EADL, Z, 912, NoEiC, i, PQN_elements, Shell_elements, Nel_elements) ! read binding energies
      if (i .EQ. 1) NOSHs = size(Nel_elements,2) ! 2d dimension
      call READ_EADL_TYPE_FILE_real(FN1,File_name_EADL, Z, 913, NoEiC, NOSHs, i, Ip_elements) ! read binding energies
      size_DTBS(1) = size(Ip_elements,1)
      size_DTBS(2) = size(Ip_elements,2)
      call READ_EADL_TYPE_FILE_real(FN1,File_name_EADL, Z, 914, NoEiC, NOSHs, i, Ekin_elements) ! read kinetic energies
      call READ_EADL_TYPE_FILE_real(FN1,File_name_EADL, Z, 922, NoEiC, NOSHs, i, t_Auger) ! read auger-times

      write(102,'(a,i12)') 'Z=', Z_elements(i)
      write(102,'(a)') 'Element; 	n quantum; 	N_e; 		I_p [eV]; 		Ekin [eV];		Auger [fs]'
      do j = 1,size(Nel_elements,2)
         if ((t_Auger(i,j) .LE. 0.0d0) .OR. (t_Auger(i,j) .GT. 1.0d23))then
            t_Auger(i,j)=1.d23 ! [fs] not possible process => infinite time
         else
            t_Auger(i,j)=1.d15*g_h/(g_e*t_Auger(i,j)) ! [fs]
         endif
         !t_Auger(i,j)=1.d23 ! [fs] not possible process => infinite time
         if (Ip_elements(i,j) .EQ. 0.0d0) exit
	
         write(102,'(i12,i12,f25.16,f25.16,f25.16,e25.16)') i, PQN_elements(i,j), Nel_elements(i,j), Ip_elements(i,j), Ekin_elements(i,j), t_Auger(i,j) !1.d15*g_h/(g_e*t_Auger(i,j))

      enddo

	! numbers 73, 0, and 91 are specific for this particular database, see the manual
	call READ_EPDL_TYPE_FILE_real(FN2,File_name_EPDL, Z, 73, 0, 0, NoEiC, i, size_DTBS, Photoabs_sigma, Eph, Photoabs_sigma_TEMP) ! total cross-section
	call READ_EPDL_TYPE_FILE_real(FN2,File_name_EPDL, Z, 73, 0, 91, NoEiC, i, size_DTBS, Photoabs_sigma, Eph, Photoabs_sigma_TEMP) ! by subshells

      write(102,'(a,i12)') 'Photoabsorption cross-sections [A^2] for: ', Z_elements(i)
      write(102,'(a,f25.16,a,e25.16)') 'For photon energy of', Eph, ' [eV]. Total: ', Photoabs_sigma_TEMP 
      do j = 1, size(Photoabs_sigma,2)
         if (Ip_elements(i,j) .EQ. 0.0d0) exit
         write(102,'(a,f25.16,a,e25.16)') 'Shell', Ip_elements(i,j), ' [eV], ', Photoabs_sigma(i,j)
      enddo 	! j = 1, size(Photoabs_sigma,3)
      
	Photoabs_sigma_tot = Photoabs_sigma_tot + N_elements(i)*Photoabs_sigma_TEMP ! total cross section sums over all elements in the solid
      
   enddo       ! for each element i

   ! Account for the contribution of sort of atoms into the molucele:
   do i = 1, NoEiC    ! for each element, read data:
         Photoabs_sigma(i,:) = N_elements(i)*Photoabs_sigma(i,:)
   enddo
   
   ! Check if the photoabsorption cross sections are consistent:
      if (abs (Photoabs_sigma_tot - SUM(Photoabs_sigma(:,:)))/SUM(Photoabs_sigma(:,:)) .gt. 1.d-7 ) then
         print*, '****Warning!********'
         print*, 'Inconsistency in EPDL97 database noticed!'
         print*, 'Total photabsorption cross section', Photoabs_sigma_tot, 'does not equal sum of partial ones:', SUM(Photoabs_sigma)
         print*, 'For concistency, replacing the total one with the sum of partial ones...'
         print*, '*********************'
         Photoabs_sigma_tot = SUM(Photoabs_sigma(:,:))
      endif

	! Check consistency for Auger decays:
	E_min = 2.0d0*MINVAL(Ip_elements, MASK=(Ip_elements > 1.0d-10)) ! minimal energy needed for an electron emission (uppermost shell or band gap)
	forall(i = 1:size(Ip_elements,1), j = 1:size(Ip_elements,2), Ip_elements(i, j) .LE. E_min) t_Auger(i,j)=1d25 ! [fs] not possible process => infinite decay time

   if (.not.allocated(whether_valence)) allocate(whether_valence(size(Ip_elements,1),size(Ip_elements,2)))
   call Introduce_bandgap(102, NelVB_elements, Effgap, Nel_elements, Ip_elements,whether_valence)
!    do i = 1, size(Photoabs_sigma,1)
!       do j = 1, size(Photoabs_sigma,2)
!          if (Ip_elements(i,j) .GT. 0.0d0) write(*,'(a,f25.16,f25.16,e)') 'Shell', Ip_elements(i,j), Nel_elements(i,j)
!       enddo
!    enddo

   NVB_el = 0.0d0
   do i = 1, NoEiC
      NVB_el = NVB_el + N_elements(i)*SUM(Nel_elements(i,:), mask = Ip_elements(i,:) .EQ. Effgap)
   enddo
   !print*, 'NVB_el', NVB_el
 
   allocate(CSN(NoEiC,size_DTBS(2)))
   CSN = 1	! first, use the default BEB cross-section
   READ(FN,*,IOSTAT=End_file) READ1, CDF_parameter_file    ! File name of the CDF parameters file
   IF ((End_file .NE. 0) .or. (READ1 .EQ. 1))  THEN !something's wrong ...
      !write(CDF_parameter_file,'(a)') trim(adjustl(Folder_name))//'/CDF_parameters.dat'
      !print*, 'There is nothing in file about CDF-parameters...'
      print*, 'The BEB cross-section is used.'
   ELSE     !do normal stuff ...
      print*, 'The CDF cross-section is used instead of BEB.'
!       PRINT*, TRIM(adjustl(CDF_parameter_file))
      SELECT CASE (READ1)
         CASE(:1)	! it is BEB cross-section
            CSN = 1	! first, use the default BEB cross-section
         CASE(2)	! Mean free paths are precalculated
            backspace(FN)
            do i = 1, size_DTBS(1)	! all elements
               do j = 1, size_DTBS(2)	! all shells
                  if (Ip_elements(i,j) .GT. 0.0d0) then
                     READ(FN,*,IOSTAT=End_file) READ1, CDF_parameter_file    ! File name of the CDF parameters
                     IF (End_file .NE. 0)  THEN !
                        CSN(i,j) = 1	! no data, use BEB
                     ELSE
                        inquire(file=trim(adjustl(CDF_parameter_file)),exist=file_exist) ! check if input file is
                        if (.not. file_exist) then
                           CSN(i,j) = 1	! no data, use BEB
                        else
                           CSN(i,j) = READ1	! there is a file, use it
                           call Reading_IMFPs_file(i,j,CDF_parameter_file, IMFPs, size_DTBS)
                        endif
                     ENDIF
                  endif
!                   PRINT*, 'A:', READ1, TRIM(adjustl(CDF_parameter_file))
               enddo
            enddo
         CASE(3)	! CDF parameters are precalculated
            call READ_CDF_INPUT_FILE(CDF_parameter_file, NoEiC, Ritchi, CSN)
!             FORALL(i=1:NoEiC, j=1:size(Nel_elements,2), Ip_elements(i,j) .EQ. Effgap) Ritchi(i,j)%A(:) = Ritchi(i,j)%A(:)*Nel_elements(i,j)/NVB_el  ! normalization
!             do j = 2,4
!                Ritchi(1,j)%A(:) = Ritchi(1,j)%A(:)*Nel_elements(1,j)/8.0d0
!             enddo
            CSN = 3	! first, use the default BEB cross-section
!             print*, Ritchi(1,7)%A(:)
         CASE default
            CSN = 1	! first, use the default BEB cross-section
      end select
   END IF
    !CSN = 1	! first, use the default BEB cross-section -- to test all the features!!
!     do i = 1, size_DTBS(1)	! all elements
!        do j = 1, size_DTBS(2)	! all shells
!           do k = 1, size(Ritchi(i,j)%E0)
!              if (Ip_elements(i, j) .GT. 0.0d0) write(*,'(i2,i2,f7.1,i2, f7.1,f7.1,f7.1)') i, j, Ip_elements(i, j), CSN(i, j), Ritchi(i,j)%E0(k), Ritchi(i,j)%A(k), Ritchi(i,j)%Gamma(k)
!           enddo
!        enddo
!     enddo
   flush(FN)
   close(FN)

   flush(102)
   flush(FN1)
   flush(FN2)

!    stop 'CLOSE 1'
   close(102)
   print*, 'All the atomic data from EADL2017 and EPDL2017 dabases are saved in: ', trim(adjustl(File_name3))
!    stop 'CLOSE 2'
   close(FN2)
!    stop 'CLOSE 3'
   close(FN)
!    stop 'CLOSE finally'
end subroutine READING_INPUT_FILES


subroutine Reading_IMFPs_file(i,j,CDF_parameter_file, IMFPs, size_DTBS)
   integer, intent(in) :: i, j
   character(100), intent (in) :: CDF_parameter_file
   type(MFP_from_file), dimension(:,:), allocatable, intent(inout), optional :: IMFPs	! inelastic mean free paths from files
   integer, dimension(:), intent(in), optional :: size_DTBS
   integer FN, count_lines, k, End_file

    fn = 42

   open(unit = FN, file=trim(adjustl(CDF_parameter_file)))	! open file with MFP
   count_lines = 0
   do
      READ(FN,*,IOSTAT=End_file) !READ1, CDF_parameter_file    ! File name of the CDF parameters file
      IF (End_file .EQ. 0) count_lines = count_lines + 1
      IF (End_file .NE. 0) exit
   enddo
!    print*, 'count_lines', count_lines
   rewind(FN)

   if (.not. allocated(IMFPs)) allocate(IMFPs(size_DTBS(1),size_DTBS(2)))	! all elements, all shells
   if (.not. allocated(IMFPs(i,j)%E)) allocate(IMFPs(i,j)%E(count_lines))	! energy array [eV]
   if (.not. allocated(IMFPs(i,j)%MFP)) allocate(IMFPs(i,j)%MFP(count_lines))	! MFP array [A]
   do k = 1, count_lines
     READ(FN,*) IMFPs(i,j)%E(k), IMFPs(i,j)%MFP(k)
!      print*, IMFPs(i,j)%E(k), IMFPs(i,j)%MFP(k)
   enddo
   close(FN)
end subroutine Reading_IMFPs_file


subroutine READ_CDF_INPUT_FILE(CDF_parameter_file, NoEiC, Ritchi, CSN)
   character(100), intent(in) :: CDF_parameter_file	! name of the file
   integer, intent(in) :: NoEiC	! number of elements
   type(CDF), dimension(:,:), allocatable, intent(inout) :: Ritchi    ! Ritchi CDF
   integer, dimension(:,:), intent(inout) :: CSN	! number of cross-section

   character(100) Material_name
   integer Num_elements
   integer, dimension(:), allocatable :: Atomic_numbers
   integer, dimension(:), allocatable :: Atomic_contrib
   real(8) Density
   integer, dimension(:), allocatable :: Num_shells
   integer N_CDF_fittings
   integer Shell_num
   real(8), dimension(:,:), allocatable :: Ionization_pots
   real(8), dimension(:,:), allocatable :: N_electron_inshell
   real(8), dimension(:,:), allocatable :: Auger_times
   type(Atom), DIMENSION(:), ALLOCATABLE :: Element	! data about all elements
   integer, DIMENSION(:,:), allocatable :: PQN	! principal quantum number
   character(11), dimension(:,:), allocatable :: Shell_names ! names

   integer FN0, i, j, k
   logical file_exist

   FN0 = 211
   inquire(file=trim(adjustl(CDF_parameter_file)),exist=file_exist) ! check if input file is there
   if (.not. file_exist) then
      print*, 'No CDF file is found...'
   else
      print*, 'The file with CDF named ', trim(adjustl(CDF_parameter_file)), ' is found.'
      open(unit = FN0, FILE = trim(adjustl(CDF_parameter_file)), status = 'old')
      read(FN0, '(a100)') Material_name	! name of the target material
!       write(*, '(a100)') Material_name	! name of the target material
      read(FN0,*) Num_elements	! number of atomic spiecies in a molecule
!       write(*,*) Num_elements	! number of atomic spiecies in a molecule
      if (.not. allocated(Atomic_numbers)) allocate(Atomic_numbers(Num_elements))
      if (.not. allocated(Atomic_contrib)) allocate(Atomic_contrib(Num_elements))
      do i = 1, Num_elements	! for all elements that are there
         read(FN0,*) Atomic_numbers(i), Atomic_contrib(i)	! Atomic number | persentage per molecule
!          write(*,*) Atomic_numbers(i), Atomic_contrib(i)
      enddo
      read(FN0,*) Density	! density of the target material [1/cm^3]
!       write(*,*) Density
      if (.not. allocated(Num_shells)) allocate(Num_shells(Num_elements))
      do i = 1, Num_elements	! for all elements
      	 read(FN0,*) Num_shells(i)	! number of shells
!       	 write(*,*) Num_shells(i)
      	 do j = 1, Num_shells(i)	! for all number of shells
             if (.not. allocated(Ionization_pots)) allocate(Ionization_pots(Num_elements, Num_shells(i)))
             if (.not. allocated(N_electron_inshell)) allocate(N_electron_inshell(Num_elements, Num_shells(i)))
             if (.not. allocated(Auger_times)) allocate(Auger_times(Num_elements, Num_shells(i)))
             if (.not. allocated(Ritchi)) allocate(Ritchi(Num_elements, Num_shells(i)))
             read(FN0, *) N_CDF_fittings, Shell_num, Ionization_pots(i,j), N_electron_inshell(i,j), Auger_times(i,j)
!              write(*,'(i12,i12,f25.16,f25.16,e25.16)') N_CDF_fittings, Shell_num, Ionization_pots(i,j), N_electron_inshell(i,j), Auger_times(i,j)
             if (.not. allocated(Ritchi(i,j)%E0)) allocate(Ritchi(i,j)%E0(N_CDF_fittings))
             if (.not. allocated(Ritchi(i,j)%A)) allocate(Ritchi(i,j)%A(N_CDF_fittings))
             if (.not. allocated(Ritchi(i,j)%Gamma)) allocate(Ritchi(i,j)%Gamma(N_CDF_fittings))
             call Find_element_name(Atomic_numbers(i), Element, i, Num_elements)
             Ritchi(i,j)%Atomtype = Element(i)%Name
!              print*, Ritchi(i,j)%Atomtype
             !write(Ritchi(i,j)%Atomtype, '(a)') = trim(adjusl(Element(i)%Name))
             if (.not. allocated(PQN)) allocate(PQN(Num_elements, Num_shells(i)))
             if (.not. allocated(Shell_names)) allocate(Shell_names(Num_elements, Num_shells(i)))
             call define_PQN(Shell_num, i, j, PQN, Shell_names)
!              print*, trim(adjustl(Shell_names(i,j)))

             do k = 1, N_CDF_fittings
                read(FN0, *) Ritchi(i,j)%E0(k), Ritchi(i,j)%A(k), Ritchi(i,j)%Gamma(k)
!                 write(*,*) Ritchi(i,j)%E0(k), Ritchi(i,j)%A(k), Ritchi(i,j)%Gamma(k)
             enddo ! k = 1, N_CDF_fittings
             !Ritchi(i,j)%A(:) = Ritchi(i,j)%A(:)  ! normalization
      	 enddo ! j = 1, Num_shells(i)
      enddo ! i = 1, Num_elements
   endif
   close (FN0)
end subroutine READ_CDF_INPUT_FILE



subroutine Introduce_bandgap(FN, NelVB_elements, Effgap, Nel_elements, Ip_elements, whether_valence)
   integer, intent(in) :: FN    ! file number
   real(8), dimension(:), intent(in) :: NelVB_elements  ! number of VB electrons
   real(8), intent(in) :: Effgap  ! bandgap [eV]
   real(8), dimension(:,:), intent(in) :: Nel_elements  ! number of electrons in shells
   real(8), dimension(:,:), intent(inout) :: Ip_elements    ! ionizatin potentials [eV]
   logical, dimension(:,:), intent(inout) :: whether_valence    ! ionizatin potentials [eV]
   integer i, j, N, M
   real(8) sum_ne
   N = size(Ip_elements,1)  ! elements
   M = size(Ip_elements,2)  ! shells
   whether_valence(:,:) = .false.
   
   
     do i = 1, N
         sum_ne = 0.0d0
         j = M ! start from top
         do while (sum_ne .LE. NelVB_elements(i) - 0.001d0)
            if (Ip_elements(i, j) .GT. 0.0d0) then
                sum_ne = sum_ne + Nel_elements(i,j)
                if (Effgap .GT. 0) Ip_elements(i, j) = Effgap
		whether_valence(i, j) = .true.
            endif
            !write(*,'(i12,f25.16,f25.16,f25.16,f25.16)') j, sum_ne, NelVB_elements(i),  Ip_elements(i, j),  Ip_elements(i, j-1)
           j = j - 1
         enddo
 	!	print *, i,  Ip_elements(i, :),whether_valence(i, :)
      enddo

   if (Effgap .GT. 0) then
	write(FN, '(a,f5.2, a)') 'The average excitation energy from valence band of the material used is ', Effgap, ' [eV]'
	write(*, '(a,f5.2, a)') 'The average excitation energy from valence band of the material used is ', Effgap, ' [eV]'
   else
	write(FN, '(a)') 'The band gap of the material is NOT used, working within atomic approximation'
   endif

end subroutine Introduce_bandgap


! Subroutine that sorts arrays deccending: heaviest element must be first, lightest - the last.
pure subroutine Sort_array_deccending(Z_elements, M_elements, N_elements, NVB_elements)
   integer, DIMENSION(:), INTENT(inout) :: Z_elements ! atomic number of this element
   REAL(8), DIMENSION(:), INTENT(inout) :: M_elements ! mass of this element
   integer, DIMENSION(:), INTENT(inout) :: N_elements ! relative number of atoms in 1 molecule
   real(8), DIMENSION(:), INTENT(inout), optional :: NVB_elements ! number of VB electrons
   integer i, j, k, M, ch
   real(8) a, b
   M = size(Z_elements)
   do i = 1, M-1
      ch = 0
      do j = 1,M-1
         if (Z_elements(j) .LT. Z_elements(j+1)) then
            ch = 1
            k=Z_elements(j)
            Z_elements(j) = Z_elements(j+1)
            Z_elements(j+1)=k
            a=Z_elements(j)
            M_elements(j) = M_elements(j+1)
            M_elements(j+1)=a
            k=N_elements(j)
            N_elements(j) = N_elements(j+1)
            N_elements(j+1)=k
            b=NVB_elements(j)
            NVB_elements(j) = NVB_elements(j+1)
            NVB_elements(j+1)=b
         endif
      enddo
      if (ch .EQ. 0) exit
   enddo
end subroutine Sort_array_deccending



! Subroutine that sorts arrays deccending: heaviest element must be first, lightest - the last.
subroutine Sort_array_deccending_old(Z_elements, M_elements, N_elements)
   integer, DIMENSION(:), INTENT(inout) :: Z_elements ! atomic number of this element
   REAL(8), DIMENSION(:), INTENT(inout) :: M_elements ! mass of this element
   integer, DIMENSION(:), INTENT(inout) :: N_elements ! relative number of atoms in 1 molecule
   integer i, j, k, M, ch
   real(8) a
   M = size(Z_elements)
   do i = 1, M-1
      ch = 0
      do j = 1,M-1
         if (Z_elements(j) .LT. Z_elements(j+1)) then
            ch = 1
            k=Z_elements(j)
            Z_elements(j) = Z_elements(j+1)
            Z_elements(j+1)=k
            a=Z_elements(j)
            M_elements(j) = M_elements(j+1)
            M_elements(j+1)=a
            k=N_elements(j)
            N_elements(j) = N_elements(j+1)
            N_elements(j+1)=k
         endif
      enddo
      if (ch .EQ. 0) exit
   enddo
end subroutine Sort_array_deccending_old


! Reading data from EPDL file: photoabsrotbtion cross-section for given photon energy
subroutine READ_EPDL_TYPE_FILE_real(FN2, File_name, Z_needed, C_needed, I_needed, S_needed, NoEiC, N_needed, size_DTBS, UARRAY, Eph, Sigma)
   integer, intent (inout) :: FN2
   character(100), intent(in) :: File_name
   integer, intent(in) :: Z_needed, C_needed, I_needed, S_needed, NoEiC, N_needed
   real(8), intent(in) :: Eph ! photon energy [eV]
   real(8), intent(inout) :: Sigma ! cross-section [A^2]
   integer, DIMENSION(:), INTENT(in) :: size_DTBS
   !~ real(8), DIMENSION(:,:), INTENT(inout) :: UARRAY
   real(8), DIMENSION(:,:), ALLOCATABLE, INTENT(inout) :: UARRAY
   integer Z, A, Yi, Yo, Date, Iflag, C, I, S, EndCheck, counter, run_i
   real(8) AW, X1
   real(8) READ1, READ2, READ3, READ4, E_photon
   logical File_opened

   !print*, 'Inside 0 ',  trim(adjustl(File_name))
   
   inquire(file=trim(adjustl(File_name)),opened=File_opened)
   !print*, 'File_opened=', File_opened
   if (.not. File_opened) then
      FN2 = 338
      open(unit = FN2, FILE = trim(adjustl(File_name)), status = 'old') ! EPDL97
   endif
!    open(unit = 123, FILE = trim(adjustl(File_name)), status = 'old') ! EPDL97
!    FN2 = 123
!     print*, 'Inside 1: ', trim(adjustl(File_name))

   E_photon = Eph/1d6 ! [MeV] units in the EPDL-database

   Z = 0
   do while (Z .NE. Z_needed)
      call read_throu_EPDL(FN2, Z, A, Yi, Yo, AW, Date, Iflag, C, I, S, X1, EndCheck)
      call How_many_lines(FN2, counter, EndCheck) ! to know how many lines are there
      if (Z .GE. 100) print*, 'The element is not found in the EPDL97 database...'
      if (Z .GE. 100) exit
   enddo

!    print*, 'Found?', Z

   if (.not. allocated(UARRAY)) then
      allocate(UARRAY(size_DTBS(1), size_DTBS(2))) ! now we know the dimensions...
      UARRAY = 0.0d0
   endif

   if (Z .EQ. Z_needed) then ! this is the element we were looking for
     run_i = 0.0d0
     do while (Z .LT. Z_needed + 1) ! do throughout the whole data for this element: all shells
!         print*, 'START searching'
        call read_throu_EPDL(FN2, Z, A, Yi, Yo, AW, Date, Iflag, C, I, S, X1, EndCheck)
!         print*, 'Found something', C, I, S
!         print*, 'Needed values:', C_needed, I_needed, S_needed
        if ((C .EQ. C_needed) .AND. (I .EQ. I_needed) .AND. (S .EQ. S_needed)) then ! read it:
           SELECT CASE (S_needed) ! which data are these:
              CASE (0) ! total cross-section
!                  print*, 'Looking up total CS'
                 call Find_value_while_reading(FN2, Iflag, counter, EndCheck, E_photon, READ2)
                 Sigma = READ2*1d-8 ! [A^2]
              CASE (91) ! cross-section by shell
!                  print*, 'Looking up partial CS'
                 call Find_value_while_reading(FN2, Iflag, counter, EndCheck, E_photon, READ2)
                 run_i = run_i + 1
                 UARRAY(N_needed,run_i) = READ2*1d-8 ! [A^2]
              CASE DEFAULT
                 call How_many_lines(FN2, counter, EndCheck) ! to know how many lines are there
                 print*, 'God knows what is going on...'               ! but you have to find out without asking Him!
           END SELECT
        else ! just skip all these lines:
           call How_many_lines(FN2, counter, EndCheck) ! to know how many lines are there
        endif ! ((C .EQ. C_needed) .AND. (I .EQ. I_needed) .AND. (S .EQ. S_needed))
     enddo !while (Z .LT. Z_needed + 1) ! do throughout the whole data for this element: all shells
   endif ! (Z .EQ. Z_needed)
   
!    print*, 'Inside 2'
   flush(FN2)
   !close(FN2)
   rewind(FN2)
end subroutine READ_EPDL_TYPE_FILE_real


! Reading EADL data for case of real arrays:
subroutine READ_EADL_TYPE_FILE_real(FN, File_name, Z_needed, I_needed, NoEiC, NOSHs, N_needed, UARRAY)
   integer, intent(inout) :: FN, NOSHs
   character(100), intent(in) :: File_name
   integer, intent(in) :: Z_needed, I_needed, NoEiC, N_needed
   real(8), DIMENSION(:,:), ALLOCATABLE, INTENT(inout) :: UARRAY
   integer Z, A, Yi, Yo, Date, C, I, S, EndCheck, counter
   real(8) AW, X1
   real(8) READ1, READ2, READ3, READ4
   integer run_i
   logical File_opened, Exist_val

   Exist_val = .true.

   inquire(file=trim(adjustl(File_name)),opened=File_opened)
   if (.not. File_opened) then
      FN = 336
      open(unit = FN, FILE = trim(adjustl(File_name)), status = 'old') ! EADL
   endif

   Z = 0
   do while (Z .NE. Z_needed)
      call read_throu(FN, Z, A, Yi, Yo, AW, Date, C, I, S, X1, EndCheck)
      call How_many_lines(FN, counter, EndCheck) ! to know how many lines are there
      if (Z .GE. 100) print*, 'The element is not found in the EADL database...'
      if (Z .GE. 100) exit
   enddo

   if (Z .EQ. Z_needed) then ! this is the element we were looking for
      do while (I .NE. I_needed) ! find the value that we need
         call read_throu(FN, Z, A, Yi, Yo, AW, Date, C, I, S, X1, EndCheck)
         call How_many_lines(FN, counter, EndCheck) ! to know how many lines are there
         if (Z .GT. Z_needed) then
            print*, 'The I-value is not found in the EADL database:'
            print*, Z, Z_needed, I_needed
            Exist_val = .false.
         endif
         if (Z .GT. Z_needed) exit
!          write(*,'(a,i,i,i,i,i,i)') 'check', Z, Z_needed, I, I_needed, EndCheck, counter
      enddo
      do run_i = 1, counter+1
         backspace(FN)
      enddo
      !print*, counter, EndCheck

      if (.not. allocated(UARRAY)) then
         !allocate(UARRAY(NoEiC, counter)) ! now we know the dimensions...
         allocate(UARRAY(NoEiC, NOSHs)) ! now we know the dimensions...
         UARRAY = 0.0d0
      endif
      !print*, size(UARRAY,1), size(UARRAY,2)

      if (Exist_val) then	! if the value is found in the database, use it, if not leave default
         do run_i = 1, counter
            read(FN,*) READ1, READ2
            UARRAY(N_needed,run_i) = READ2*1e6
         enddo
      endif
   endif
   !print*, 'Here it is !!'
   flush(FN)
   !close(FN)
   rewind(FN)
9998   format(I2,I3,I3,E11.4)		! 2st line
9997   format(71X,I1)			! last line
end subroutine READ_EADL_TYPE_FILE_real


! Reading EADL data for case of integer arrays:
subroutine READ_EADL_TYPE_FILE_int(FN, File_name, Z_needed, I_needed, NoEiC, N_needed, UARRAY2, Shell_elements, UARRAY)
!READ_EADL_TYPE_FILE_int(FN,File_name_EADL, Z, 912, NoEiC, i, PQN_elements, Shell_elements, Nel_elements)
   integer, intent (inout) :: FN
   character(100), intent(in) :: File_name
   integer, intent(in) :: Z_needed, I_needed, NoEiC, N_needed
   integer, DIMENSION(:,:), ALLOCATABLE, INTENT(inout) :: UARRAY2
   real(8), DIMENSION(:,:), ALLOCATABLE, INTENT(inout) :: UARRAY
   character(11), DIMENSION(:,:), ALLOCATABLE, intent(inout) :: Shell_elements 	! names of all shells, all elements
   integer Z, A, Yi, Yo, Date, C, I, S, EndCheck, counter
   real(8) AW, X1
   real(8) READ1, READ2, READ3, READ4
   integer run_i 
   logical File_opened
   !FN = 337
   !open(unit = FN, FILE = trim(adjustl(File_name)), status = 'old')

   inquire(file=trim(adjustl(File_name)),opened=File_opened)
   if (.not. File_opened) then
      FN = 336
      open(unit = FN, FILE = trim(adjustl(File_name)), status = 'old')
   endif

   Z = 0
   do while (Z .NE. Z_needed)
      call read_throu(FN, Z, A, Yi, Yo, AW, Date, C, I, S, X1, EndCheck)
      call How_many_lines(FN, counter, EndCheck) ! to know how many lines are there
      if (Z .GE. 100) print*, 'The element is not found in the EADL database...'
      if (Z .GE. 100) exit
   enddo

   if (Z .EQ. Z_needed) then ! this is the element we were looking for
      do while (I .NE. I_needed) ! find the value that we need
         call read_throu(FN, Z, A, Yi, Yo, AW, Date, C, I, S, X1, EndCheck)
         call How_many_lines(FN, counter, EndCheck) ! to know how many lines are there
         if (Z .GT. Z_needed) print*, 'INT: The I-value is not found in the EADL database...'
         if (Z .GT. Z_needed) exit
      enddo
      do run_i = 1, counter+1
         backspace(FN)
      enddo

      if (.not. allocated(UARRAY)) then
         allocate(UARRAY(NoEiC, counter)) ! now we know the dimensions...
         UARRAY = 0.0d0	! number of electron on shells
      endif
      if (.not. allocated(UARRAY2)) then
         allocate(UARRAY2(NoEiC, counter)) ! now we know the dimensions...
         UARRAY2 = 0	! principal quantum number for shells
      endif
      if (.not. allocated(Shell_elements)) then
         allocate(Shell_elements(NoEiC, counter)) ! now we know the dimensions...
         Shell_elements = '0'	! names of shells
      endif

      do run_i = 1, counter
         read(FN,*) READ1, READ2 
         !UARRAY(N_needed,run_i) = AINT(READ2)
         UARRAY(N_needed,run_i) = READ2
         call define_PQN(int(READ1), N_needed, run_i, UARRAY2, Shell_elements)
!          print*, Z, Shell_elements(N_needed, run_i)
      enddo
   endif
   flush(FN)
   !close(FN)
   rewind(FN)
end subroutine READ_EADL_TYPE_FILE_int


! Reading first 2 lines with the descriptions in EPDL database: 
subroutine read_throu_EPDL(FN, Z, A, Yi, Yo, AW, Date1, Iflag, C, I, S, X1, EndCheck)
   integer, intent (inout) :: FN, Z, A, Yi, Yo, Date1, Iflag, C, I, S, EndCheck
   real(8), intent (inout) :: AW, X1
   real(8) skipread
   integer :: INFO	! info whether file read well
   EndCheck = 0
   ! First header line:
   !read(FN,'(i3, i3,i2, i2, E11.4, 2x, i8, i1)') Z, A, Yi, Yo, AW, Date1, Iflag  	! 1st line
   read(FN,'(i3, i3, 1X, i2, 1X, i2, 1X, E11.4, 1x, i6, i1)',IOSTAT=INFO) Z, A, Yi, Yo, AW, Date1, Iflag
   ! Second header line:
   !read(FN,'(i2,i3,i3,E11.4,E11.4)') C, I, S, skipread, X1  			! 2st line
   read(FN,'(i2, i3, i3, 14X, E11.4)',IOSTAT=INFO) C, I, S, X1
   read(FN,9997) EndCheck  			! 3st line, cheking...
   
9997   format(71X,I1)			! last line
end subroutine read_throu_EPDL


! Reading first 2 lines with the descriptions in EADL database:
subroutine read_throu(FN, Z, A, Yi, Yo, AW, Date, C, I, S, X1, EndCheck)
   integer, intent (inout) :: FN, Z, A, Yi, Yo, Date, C, I, S, EndCheck
   real(8), intent (inout) :: AW, X1
   EndCheck = 0
   read(FN,9999) Z, A, Yi, Yo, AW, Date  	! 1st line
   read(FN,9998) C, I, S, X1  			! 2st line
   read(FN,9997) EndCheck  			! 3st line, cheking...

9999   format(I3,I3,I2,I2,E11.4,I6)	! 1st line
9998   format(I2,I3,I3,E11.4)		! 2st line
9997   format(71X,I1)			! last line
end subroutine read_throu 


! Counts how many lines contain the data in a block of EADL, EPDL-databases
subroutine How_many_lines(FN,counter, EndCheck)
   integer, intent(in) :: FN
   integer, intent(inout) :: counter, EndCheck
   counter = 0
   do while (EndCheck .NE. 1) ! until end of data for this sub-shell/input
      counter = counter + 1 ! how many lines are there
      read(FN,9997) EndCheck  			! 3st line, cheking...
      !print*, 'How_many_lines ', counter, EndCheck
      if (counter .GE. 5000) EndCheck = -1	! indicates some ERROR
      if (counter .GE. 5000) exit	! if something went wrong...
   enddo
9997   format(71X,I1)			! last line
end subroutine How_many_lines


! EADL database, that's how to find the cross-section for given photon energy for each subshell:
subroutine Find_value_while_reading(FN, Iflag, counter, EndCheck, E_photon, OUT_value)
   integer, intent(in) :: FN, Iflag
   integer, intent(inout) :: counter, EndCheck
   real(8), intent(in) :: E_photon
   real(8), intent(out) :: OUT_value
   real(8) READ1, READ2, E1, E2, Sigma1, Sigma2
   LOGICAL :: Firsttime

   Firsttime = .true.
   counter = 0
   backspace(FN)
   do while (EndCheck .NE. 1) ! until end of data for this sub-shell/input
      counter = counter + 1 ! how many lines are there
      read(FN,9997) EndCheck        ! 3d line, cheking...
      if (EndCheck .NE. 1) then
         backspace(FN)
         read(FN,*) READ1, READ2
      endif
      if ((READ1 >= E_photon) .AND. (Firsttime)) then
         Firsttime = .false.
         backspace(FN)
         backspace(FN)
         read(FN,*) READ1, READ2
         if (counter .EQ. 1) then ! too small energy!
            OUT_value = 0.0d0
            exit  ! no need to scan more
         else ! photon energy is above the ionization potential
            E1 = READ1
            Sigma1 = READ2
            read(FN,*) READ1, READ2
            E2 = READ1
            IF (E2 == E1) then  ! it is an energy just around a shell edge
               Firsttime = .true.  ! find the last of identical energies, continue the cycle
            else  ! we found the value on the higher-energy side of a shell edge
               Sigma2 = READ2
               call Interpolate_EPDL(Iflag, E1, E2, Sigma1, Sigma2, E_photon, OUT_value)
               exit  ! if we found the value
            endif
         endif
      endif

      if (counter .GE. 5000) then
         EndCheck = -1  ! indicates some ERROR
         exit  ! if something went wrong...
      endif
   enddo
9997   format(71X,I1)      ! last line
end subroutine Find_value_while_reading


! Interpolation of photoabsorbtion cross-section according to EADL database:
subroutine Interpolate_EPDL(Iflag, E1, E2, Sigma1, Sigma2, E_needed, OUT_value)
   integer, intent(in) :: Iflag
   real(8), intent(in) :: E1, E2, Sigma1, Sigma2, E_needed
   real(8), intent(out) :: OUT_value
   real(8) E2log, E1log, E_needed_log, Sigma1log, Sigma2log
   select case(Iflag) ! what interpolation to use:
      case(0,2) ! linear x and y
         OUT_value = Sigma1 + (Sigma2 - Sigma1)/(E2 - E1)*(E_needed - E1)
      case(3)	! logarithmic x, linear y
         E2log = log(E2)
         E1log = log(E1)
         E_needed_log = log(E_needed)
         OUT_value = Sigma1 + (Sigma2 - Sigma1)/(E2log - E1log)*(E_needed_log - E1log)
      case(4)	! linear x, logarithmic y
         Sigma1log = log(Sigma1)
         Sigma2log = log(Sigma2)
         OUT_value = Sigma1log + (Sigma2log - Sigma1log)/(E2 - E1)*(E_needed - E1)
         OUT_value = exp(OUT_value)
      case(5)	! logarithmic x and y
         E2log = log(E2)
         E1log = log(E1)
         E_needed_log = log(E_needed)
         Sigma1log = log(Sigma1)
         Sigma2log = log(Sigma2)
         OUT_value = Sigma1log + (Sigma2log - Sigma1log)/(E2log - E1log)*(E_needed_log - E1log)
         OUT_value = exp(OUT_value)
      case default ! linear x and y
         OUT_value = Sigma1 + (Sigma2 - Sigma1)/(E2 - E1)*(E_needed - E1) 
   end select
end subroutine Interpolate_EPDL


! According to EADL-data format, the atomic shells are numerated as:
subroutine define_PQN(READ1, N_needed, run_i, UARRAY2, Uarray3)
   integer, intent(in) :: READ1
   integer, intent(in) :: run_i, N_needed
   integer, DIMENSION(:,:), INTENT(inout) :: UARRAY2	! principal quantum number
   character(11), dimension(:,:), intent(inout), optional :: Uarray3 ! names
   SELECT CASE(READ1)
   CASE ( : 1)
      UARRAY2(N_needed,run_i) = 1 ! K-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'K-shell'
   CASE (2)
      UARRAY2(N_needed,run_i) = 2 ! L-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'L-shell'
   CASE (3)
      UARRAY2(N_needed,run_i) = 2 ! L-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'L1-shell'
   CASE (4)
      UARRAY2(N_needed,run_i) = 2 ! L-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'L23-shell'
   CASE (5)
      UARRAY2(N_needed,run_i) = 2 ! L-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'L2-shell'
   CASE (6)
      UARRAY2(N_needed,run_i) = 2 ! L-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'L3-shell'
   CASE (7)
      UARRAY2(N_needed,run_i) = 3 ! M-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'M-shell'
   CASE (8)
      UARRAY2(N_needed,run_i) = 3 ! M-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'M1-shell'
   CASE (9)
      UARRAY2(N_needed,run_i) = 3 ! M-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'M23-shell'
   CASE (10)
      UARRAY2(N_needed,run_i) = 3 ! M-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'M2-shell'
   CASE (11)
      UARRAY2(N_needed,run_i) = 3 ! M-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'M3-shell'
   CASE (12)
      UARRAY2(N_needed,run_i) = 3 ! M-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'M45-shell'
   CASE (13)
      UARRAY2(N_needed,run_i) = 3 ! M-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'M4-shell'
   CASE (14)
      UARRAY2(N_needed,run_i) = 3 ! M-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'M5-shell'
   CASE (15)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N-shell'
   CASE (16)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N1-shell'
   CASE (17)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N23-shell'
   CASE (18)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N2-shell'
   CASE (19)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N3-shell'
   CASE (20)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N45-shell'
   CASE (21)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N4-shell'
   CASE (22)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N5-shell'
   CASE (23)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N67-shell'
   CASE (24)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N6-shell'
   CASE (25)
      UARRAY2(N_needed,run_i) = 4 ! N-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'N7-shell'
   CASE (26)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O-shell'
   CASE (27)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O1-shell'
   CASE (28)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O23-shell'
   CASE (29)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O2-shell'
   CASE (30)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O3-shell'
   CASE (31)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O45-shell'
   CASE (32)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O4-shell'
   CASE (33)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O5-shell'
   CASE (34)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O67-shell'
   CASE (35)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O6-shell'
   CASE (36)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O7-shell'
   CASE (37)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O89-shell'
   CASE (38)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O8-shell'
   CASE (39)
      UARRAY2(N_needed,run_i) = 5 ! O-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'O9-shell'
   CASE (40)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P-shell'
   CASE (41)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P1-shell'
   CASE (42)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P23-shell'
   CASE (43)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P2-shell'
   CASE (44)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P3-shell'
   CASE (45)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P45-shell'
   CASE (46)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P4-shell'
   CASE (47)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P5-shell'
   CASE (48)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P67-shell'
   CASE (49)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P6-shell'
   CASE (50)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P7-shell'
   CASE (51)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P89-shell'
   CASE (52)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P8-shell'
   CASE (53)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P9-shell'
   CASE (54)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P1011-shell'
   CASE (55)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P10-shell'
   CASE (56)
      UARRAY2(N_needed,run_i) = 6 ! P-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'P11-shell'
   CASE (57)
      UARRAY2(N_needed,run_i) = 7 ! Q-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'Q-shell'
   CASE (58)
      UARRAY2(N_needed,run_i) = 7 ! Q-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'Q1-shell'
   CASE (59)
      UARRAY2(N_needed,run_i) = 7 ! Q-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'Q23-shell'
   CASE (60)
      UARRAY2(N_needed,run_i) = 7 ! Q-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'Q2-shell'
   CASE (61)
      UARRAY2(N_needed,run_i) = 7 ! Q-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'Q3-shell'
   CASE (62:)
      UARRAY2(N_needed,run_i) = 7 ! Q-shell
      if (present(Uarray3)) Uarray3(N_needed, run_i) = 'Q...-shell'
   CASE DEFAULT
      UARRAY2(N_needed,run_i) = 1 ! K-shell, if something else happened...
   END SELECT
end subroutine define_PQN

!--------------------------------------------------------------------------------

! Carriagecontrol:
subroutine Printing_persents(i, N)
   class(*), intent(in) :: i, N
   real(8) P
   select type(i)
   type is (integer)
      select type(N)
         type is (integer)
            P = DBLE(i)/DBLE(N)*100.0d0
         type is (real)
            P = i/DBLE(N)*100.0d0
      endselect
   type is (real)
      select type(N)
         type is (integer)
            P = i/DBLE(N)*100.0d0
         type is (real)
            P = i/N*100.0d0
      endselect
   endselect
!    open(unit=6,form='formatted',carriagecontrol='fortran')
!    write(6,'(a)') '+'
   write(6,'(a, f6.2, " %")') '+', P
!    close(6)
end subroutine Printing_persents


subroutine Find_element_name(Z, Element, N, Num_el, Num_VB_el)
   integer, intent (in) :: Z	! atomic number
   type(Atom), DIMENSION(:), ALLOCATABLE, INTENT(inout) :: Element	! data about all elements
   integer, intent(in) :: N 	! number of element
   integer, intent(in), optional :: Num_el ! total number of elements
   real(8), intent(out), optional :: Num_VB_el ! number of VB electrons for this element

   real(8) M	! mass [proton mass]
   real(8) Num_VB ! number of valence electrons in this element
   Character(3) Name ! short name of element
   Character(30) Full_Name ! full name of element

   if (.not. allocated(Element)) then
      if (present(Num_el)) then
         allocate(Element(Num_el))
      else
         print*, 'Number of elements is not present, Element-array can not be alocated!'
      endif
   endif

   select case (Z)
   case(1)
 	Name = 'H'
 	Full_Name = 'Hydrogen'
 	M = 1.0082d0
 	Num_VB = 1.0d0
   case(2) 	
	Name = 'He' 	
	Full_Name = 'Helium'
	M = 4.002602d0
	Num_VB = 2.0d0
   case(3)	
	Name = 'Li' 	
	Full_Name = 'Lithium'
	M = 6.942d0
	Num_VB = 1.0d0
   case(4)	
	Name = 'Be' 	
	Full_Name = 'Beryllium'
	M = 9.012182d0
	Num_VB = 2.0d0
   case(5)
 	Name = 'B' 	
	Full_Name = 'Boron' 
	M = 10.812d0
	Num_VB = 3.0d0
   case(6)	
 	Name = 'C' 	
	Full_Name = 'Carbon' 
	M = 12.0112d0
	Num_VB = 4.0d0
   case(7)
	Name = 'N' 	
	Full_Name = 'Nitrogen'  	
 	M = 14.0072d0
	Num_VB = 5.0d0
   case(8)
	Name = 'O' 	
	Full_Name = 'Oxygen'  	
 	M = 15.9992d0
	Num_VB = 6.0d0
   case(9)
	Name = 'F' 	
	Full_Name = 'Fluorine'  	
 	M = 18.9984032d0
	Num_VB = 7.0d0
   case(10)
	Name = 'Ne' 	
	Full_Name = 'Neon'  	
 	M = 20.1797d0
	Num_VB = 8.0d0
   case(11)
	Name = 'Na' 	
	Full_Name = 'Sodium'  	
 	M = 22.98976928d0
	Num_VB = 1.0d0
   case(12)
	Name = 'Mg' 	
	Full_Name = 'Magnesium'  	
 	M = 24.3059d0
	Num_VB = 2.0d0
   case(13)
	Name = 'Al' 	
	Full_Name = 'Aluminium' 	
 	M = 26.9815386d0
	Num_VB = 3.0d0
   case(14)
	Name = 'Si' 	
	Full_Name = 'Silicon'  	
 	M = 28.0854d0
	Num_VB = 4.0d0
   case(15)
	Name = 'P' 	
	Full_Name = 'Phosphorus'  	
 	M = 30.973762d0
	Num_VB = 5.0d0
   case(16)
	Name = 'S' 	
	Full_Name = 'Sulfur'  	
 	M = 32.062d0
	Num_VB = 6.0d0
   case(17)
	Name = 'Cl' 	
	Full_Name = 'Chlorine' 	
 	M = 35.452d0
	Num_VB = 7.0d0
   case(18)
	Name = 'Ar' 	
	Full_Name = 'Argon'
	M = 39.948d0
	Num_VB = 8.0d0
   case(19)
	Name = 'K' 	
	Full_Name = 'Potassium'  	
 	M = 39.0983d0
	Num_VB = 1.0d0
   case(20)
	Name = 'Ca' 	
	Full_Name = 'Calcium'  	
 	M = 40.078d0
	Num_VB = 2.0d0
   case(21)
	Name = 'Sc' 	
	Full_Name = 'Scandium'  	
 	M = 44.955912d0
	Num_VB = 3.0d0
   case(22)
	Name = 'Ti' 	
	Full_Name = 'Titanium'  	
 	M = 47.867d0
	Num_VB = 4.0d0
   case(23)
	Name = 'V'	
	Full_Name = 'Vanadium'  	
 	M = 50.9415d0
	Num_VB = 5.0d0
   case(24)
	Name = 'Cr' 	
	Full_Name = 'Chromium'  	
 	M = 51.9961d0
	Num_VB = 6.0d0
   case(25)
	Name = 'Mn' 	
	Full_Name = 'Manganese' 	
 	M = 54.938045d0
	Num_VB = 7.0d0
   case(26)
	Name = 'Fe' 	
	Full_Name = 'Iron'  	
 	M = 55.845d0
	Num_VB = 8.0d0
   case(27)
	Name = 'Co' 	
	Full_Name = 'Cobalt' 	
 	M = 58.933195d0
	Num_VB = 9.0d0
   case(28)
	Name = 'Ni' 	
	Full_Name = 'Nickel'  	
 	M = 58.6934d0
	Num_VB = 10.0d0
   case(29)
	Name = 'Cu' 	
	Full_Name = 'Copper'  	
 	M = 63.546d0
	Num_VB = 1.0d0
   case(30)
	Name = 'Zn' 	
	Full_Name = 'Zinc'  	
 	M = 65.38d0
	Num_VB = 2.0d0
   case(31)
	Name = 'Ga' 	
	Full_Name = 'Gallium'  	
 	M = 69.723d0
	Num_VB = 3.0d0
   case(32)
	Name = 'Ge' 	
	Full_Name = 'Germanium'  	
 	M = 72.630d0
	Num_VB = 4.0d0
   case(33)
	Name = 'As' 	
	Full_Name = 'Arsenic' 	
 	M = 74.92160d0
	Num_VB = 5.0d0
   case(34)
	Name = 'Se' 	
	Full_Name = 'Selenium' 	
 	M = 78.96d0
	Num_VB = 6.0d0
   case(35)
	Name = 'Br' 	
	Full_Name = 'Bromine'  	
 	M = 79.9049d0
	Num_VB = 7.0d0
   case(36)
	Name = 'Kr' 	
	Full_Name = 'Krypton'  	
 	M = 83.798d0
	Num_VB = 8.0d0
   case(37)
	Name = 'Rb' 	
	Full_Name = 'Rubidium'  	
 	M = 85.4678d0
	Num_VB = 1.0d0
   case(38)
	Name = 'Sr' 	
	Full_Name = 'Strontium'  	
 	M = 87.62d0
	Num_VB = 2.0d0
   case(39)
	Name = 'Y' 	
	Full_Name = 'Yttrium'  	
 	M = 88.90585d0
	Num_VB = 3.0d0
   case(40)
	Name = 'Zr' 	
	Full_Name = 'Zirconium'  	
 	M = 91.224d0
	Num_VB = 4.0d0
   case(41)
	Name = 'Nb' 	
	Full_Name = 'Niobium'  	
 	M = 92.90638d0
	Num_VB = 5.0d0
   case(42)
	Name = 'Mo' 	
	Full_Name = 'Molybdenum'  	
 	M = 95.96d0
	Num_VB = 6.0d0
   case(43)
	Name = 'Tc' 	
	Full_Name = 'Technetium' 
 	M = 98d0
	Num_VB = 7.0d0
   case(44)
	Name = 'Ru' 	
	Full_Name = 'Ruthenium'  	
 	M = 101.07d0
	Num_VB = 8.0d0
   case(45)
	Name = 'Rh' 	
	Full_Name = 'Rhodium'  	
 	M = 102.90550d0
	Num_VB = 9.0d0
   case(46)
	Name = 'Pd' 	
	Full_Name = 'Palladium'  	
 	M = 106.42d0
	Num_VB = 10.0d0
   case(47)
	Name = 'Ag' 	
	Full_Name = 'Silver'  	
 	M = 107.8682d0
	Num_VB = 1.0d0
   case(48)
	Name = 'Cd' 	
	Full_Name = 'Cadmium'  	
 	M = 112.411d0
	Num_VB = 2.0d0
   case(49)
	Name = 'In' 	
	Full_Name = 'Indium'  	
 	M = 114.818d0
	Num_VB = 3.0d0
   case(50)
	Name = 'Sn' 	
	Full_Name = 'Tin'
 	M = 118.710d0
	Num_VB = 4.0d0
   case(51)
	Name = 'Sb' 	
	Full_Name = 'Antimony'  	
 	M = 121.760d0
	Num_VB = 5.0d0
   case(52)
	Name = 'Te' 	
	Full_Name = 'Tellurium'  	
 	M = 127.60d0
	Num_VB = 6.0d0
   case(53)
	Name = 'I'	
	Full_Name = 'Iodine'  	
 	M = 126.90447d0
	Num_VB = 7.0d0
   case(54)
	Name = 'Xe' 	
	Full_Name = 'Xenon'  	
 	M = 131.293d0
	Num_VB = 8.0d0
   case(55)
	Name = 'Cs' 	
	Full_Name = 'Caesium'  	
 	M = 132.9054519d0
	Num_VB = 1.0d0
   case(56)
	Name = 'Ba' 	
	Full_Name = 'Barium'  	
 	M = 137.327d0
	Num_VB = 2.0d0
   case(57)
	Name = 'La' 	
	Full_Name = 'Lanthanum'  	
 	M = 138.90547d0
	Num_VB = 3.0d0
   case(58)
	Name = 'Ce' 	
	Full_Name = 'Cerium'  	
 	M = 140.116d0
	Num_VB = 4.0d0
   case(59)
	Name = 'Pr' 	
	Full_Name = 'Praseodymium'  	
 	M = 140.90765d0
	Num_VB = 5.0d0
   case(60)
	Name = 'Nd' 	
	Full_Name = 'Neodymium'  	
 	M = 144.242d0
	Num_VB = 6.0d0
   case(61)
	Name = 'Pm' 	
	Full_Name = 'Promethium' 
 	M = 145d0
	Num_VB = 7.0d0
   case(62)
	Name = 'Sm' 	
	Full_Name = 'Samarium'  	
 	M = 150.36d0
	Num_VB = 8.0d0
   case(63)
	Name = 'Eu' 	
	Full_Name = 'Europium' 	
 	M = 151.964d0
	Num_VB = 9.0d0
   case(64)
	Name = 'Gd' 	
	Full_Name = 'Gadolinium'  	
 	M = 157.25d0
	Num_VB = 10.0d0
   case(65)
	Name = 'Tb' 	
	Full_Name = 'Terbium'  	
 	M = 158.92535d0
	Num_VB = 11.0d0
   case(66)
	Name = 'Dy' 	
	Full_Name = 'Dysprosium'  	
 	M = 162.500d0
	Num_VB = 12.0d0
   case(67)
	Name = 'Ho' 	
	Full_Name = 'Holmium'  	
 	M = 164.93032d0
	Num_VB = 13.0d0
   case(68)
	Name = 'Er' 	
	Full_Name = 'Erbium'  	
 	M = 167.259d0
	Num_VB = 14.0d0
   case(69)
	Name = 'Tm' 	
	Full_Name = 'Thulium'  	
 	M = 168.93421d0
	Num_VB = 1.0d0
   case(70)
	Name = 'Yb' 	
	Full_Name = 'Ytterbium'  	
 	M = 173.054d0
	Num_VB = 2.0d0
   case(71)
	Name = 'Lu' 	
	Full_Name = 'Lutetium'  	
 	M = 174.9668d0
	Num_VB = 3.0d0
   case(72)
	Name = 'Hf' 	
	Full_Name = 'Hafnium'  	
 	M = 178.49d0
	Num_VB = 4.0d0
   case(73)
	Name = 'Ta' 	
	Full_Name = 'Tantalum'  	
 	M = 180.94788d0
	Num_VB = 5.0d0
   case(74)
	Name = 'W' 	
	Full_Name = 'Tungsten'  	
 	M = 183.84d0
	Num_VB = 6.0d0
   case(75)
	Name = 'Re' 	
	Full_Name = 'Rhenium'  	
 	M = 186.207d0
	Num_VB = 7.0d0
   case(76)
	Name = 'Os' 	
	Full_Name = 'Osmium' 	
 	M = 190.23d0
	Num_VB = 8.0d0
   case(77)
	Name = 'Ir' 	
	Full_Name = 'Iridium'  	
 	M = 192.217d0
	Num_VB = 9.0d0
   case(78)
	Name = 'Pt' 	
	Full_Name = 'Platinum'  	
 	M = 195.084d0
	Num_VB = 10.0d0
   case(79)
	Name = 'Au' 	
	Full_Name = 'Gold'  	
 	M = 196.966569d0
	Num_VB = 11.0d0
   case(80)
	Name = 'Hg' 	
	Full_Name = 'Mercury'  	
 	M = 200.592d0
	Num_VB = 12.0d0
   case(81)
	Name = 'Tl'	
	Full_Name = 'Thallium' 	
 	M = 204.389d0
	Num_VB = 3.0d0
   case(82)
	Name = 'Pb' 	
	Full_Name = 'Lead' 
	M = 207.2d0
	Num_VB = 4.0d0
   case(83)
	Name = 'Bi' 	
	Full_Name = 'Bismuth'  	
 	M = 208.98040d0
	Num_VB = 5.0d0
   case(84)
	Name = 'Po' 	
	Full_Name = 'Polonium' 
 	M = 209.0d0
	Num_VB = 6.0d0
   case(85)
	Name = 'At' 	
	Full_Name = 'Astatine' 
 	M = 210.0d0
	Num_VB = 7.0d0
   case(86)
	Name = 'Rn' 	
	Full_Name = 'Radon' 	
 	M = 222.0d0
	Num_VB = 8.0d0
   case(87)
	Name = 'Fr' 	
	Full_Name = 'Francium' 	
 	M = 223.0d0
	Num_VB = 1.0d0
   case(88)
	Name = 'Ra' 	
	Full_Name = 'Radium' 
 	M = 226.0d0
	Num_VB = 2.0d0
   case(89)
	Name = 'Ac' 	
	Full_Name = 'Actinium' 
 	M = 227.0d0
	Num_VB = 3.0d0
   case(90)
	Name = 'Th' 	
	Full_Name = 'Thorium'  	
 	M = 232.03806d0
	Num_VB = 4.0d0
   case(91)
	Name = 'Pa' 	
	Full_Name = 'Protactinium'
 	M = 231.035880d0
	Num_VB = 5.0d0
   case(92)
	Name = 'U' 	
	Full_Name = 'Uranium'  	
 	M = 238.02891d0
	Num_VB = 6.0d0
   case(93)
	Name = 'Np' 	
	Full_Name = 'Neptunium'
 	M = 237.0d0
	Num_VB = 7.0d0
   case(94)
	Name = 'Pu' 	
	Full_Name = 'Plutonium'
 	M = 244.0d0
	Num_VB = 8.0d0
   case(95)
	Name = 'Am' 	
	Full_Name = 'Americium' 
	M = 243.0d0
	Num_VB = 9.0d0
   case(96)
	Name = 'Cm' 	
	Full_Name = 'Curium' 
 	M = 247.0d0
	Num_VB = 10.0d0
   case(97)
	Name = 'Bk' 	
	Full_Name = 'Berkelium' 
 	M = 247.0d0
	Num_VB = 11.0d0
   case(98)
	Name = 'Cf' 	
	Full_Name = 'Californium' 
 	M = 251.0d0
	Num_VB = 12.0d0
   case(99)
	Name = 'Es' 	
	Full_Name = 'Einsteinium' 
 	M = 252.0d0
	Num_VB = 13.0d0
   case(100)
	Name = 'Fm' 	
	Full_Name = 'Fermium'
 	M = 257.0d0
	Num_VB = 14.0d0
   case(101)
	Name = 'Md' 	
	Full_Name = 'Mendelevium' 
 	M = 258.0d0
	Num_VB = 15.0d0
   case(102)
	Name = 'No' 	
	Full_Name = 'Nobelium'
 	M = 259.0d0
	Num_VB = 16.0d0
   case(103)
	Name = 'Lr' 	
	Full_Name = 'Lawrencium'
 	M = 262.0d0
	Num_VB = 3.0d0
   case(104)
	Name = 'Rf' 	
	Full_Name = 'Rutherfordium' 
 	M = 267.0d0
	Num_VB = 4.0d0
   case(105)
	Name = 'Db' 	
	Full_Name = 'Dubnium' 
 	M = 268.0d0
	Num_VB = 5.0d0
   case(106)
	Name = 'Sg' 	
	Full_Name = 'Seaborgium' 
 	M = 269.0d0
	Num_VB = 6.0d0
   case(107)
	Name = 'Bh' 	
	Full_Name = 'Bohrium'
 	M = 270.0d0
	Num_VB = 7.0d0
   case(108)
	Name = 'Hs' 	
	Full_Name = 'Hassium' 
 	M = 269.0d0
	Num_VB = 8.0d0
   case(109)
	Name = 'Mt' 	
	Full_Name = 'Meitnerium' 
 	M = 278.0d0
	Num_VB = 9.0d0
   case(110)
	Name = 'Ds' 	
	Full_Name = 'Darmstadtium' 	
 	M = 281.0d0
	Num_VB = 10.0d0
   case(111)
	Name = 'Rg' 	
	Full_Name = 'Roentgenium'
 	M = 281.0d0
	Num_VB = 11.0d0
   case(112)
	Name = 'Cn' 	
	Full_Name = 'Copernicium'
 	M = 285.0d0
	Num_VB = 12.0d0
   case(113)
	Name = 'Uut' 	
	Full_Name = 'Ununtrium'
 	M = 286.0d0
	Num_VB = 13.0d0
   case(114)
	Name = 'Fl' 	
	Full_Name = 'Flerovium' 
 	M = 289.0d0
	Num_VB = 14.0d0
   case(115)
	Name = 'Uup' 	
	Full_Name = 'Ununpentium' 
 	M = 288.0d0
	Num_VB = 15.0d0
   case(116)
	Name = 'Lv' 	
	Full_Name = 'Livermorium'
 	M = 293.0d0
	Num_VB = 16.0d0
   case(117)
	Name = 'Uus' 	
	Full_Name = 'Ununseptium'	
 	M = 294.0d0
	Num_VB = 17.0d0
   case(118)
	Name = 'Uuo' 	
	Full_Name = 'Ununoctium'
 	M = 294.0d0
 	Num_VB = 18.0d0
   case(119:)	
	Name = 'UFO'
	Full_Name = 'Unknown'
	M = 0.0d0
	write(*,'(a,i4,a)') 'The element number #', Z, ' is not in the database,'
	write(*,'(a)') 'please, provide its number of valence electrons'
	write(*,'(a)') '(type real positive number):'
	Num_VB = -1
	do while (Num_VB .LE. 0.0d0)
	   read*, Num_VB
	   if (Num_VB .LE. 0.0d0) print*, 'It is a negative number. Enter a positive one (N > 0)!'
	enddo
end select

Element(N)%Z = Z
Element(N)%M = M
Element(N)%Name = Name
Element(N)%Full_name = Full_name
if (present(Num_VB_el)) Num_VB_el = Num_VB

end subroutine Find_element_name



!--------------------------------------------------------------------------------
! Obsolete subroutines, not used any longer, left from the old input data-format:
subroutine READ_FIRST_LINE(File_name, Nmax)
   character(100), intent(in) :: File_name
   integer Nmax,FN
   FN = 334
   open(unit = FN, FILE = trim(adjustl(File_name)), status = 'old')
   read(FN,*) Nmax ! number of shells in this element
   rewind(334)
   flush(FN)
   close(FN)
end subroutine READ_FIRST_LINE


subroutine READ_ELEMENT_FILES(File_name, N_el, Ip_elements, Nel_elements, PQN_elements, t_Auger)
   character(100), intent(in) :: File_name
   integer, intent(in) :: N_el
   real(8), DIMENSION(:,:), INTENT(inout) :: Ip_elements	! ionization potentials [eV]
   real(8), DIMENSION(:,:), INTENT(inout) :: Nel_elements	! numbers of electrons in shells 
   integer, DIMENSION(:,:), INTENT(inout) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements   
   real(8), DIMENSION(:,:), INTENT(inout) :: t_Auger		! Auger-decays times [fs]
   integer FN, Nsh, i, j, PQN
   real(8) Ip, Auger_t
   FN = 335
   open(unit = FN, FILE = trim(adjustl(File_name)), status = 'old')
   read(FN,*) Nsh ! number of shells in this element
   do i = 1, Nsh
      read(FN,*) Ip, j, PQN, Auger_t
      Ip_elements(N_el, i) = Ip
      Nel_elements(N_el, i) = j
      PQN_elements(N_el, i) = PQN
      t_Auger(N_el, i) = Auger_t
   enddo
   flush(FN)
   close(FN)
end subroutine READ_ELEMENT_FILES

END MODULE Files_dealing
