!    XCASCADE software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains analytical cross-section calculations
MODULE Cross_sections
 use Constants   ! universal constants
 use Objects_and_types	! objects
 implicit none

contains


subroutine Energy_transfer(Noc, T, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, NOEL, shl, dE, Error_message, IMFPs, At_dens, Ritchi)
   integer, DIMENSION(:,:), intent(in) :: Noc	! number of the cross-section used
   real(8), intent(in) :: T 	! [eV] kinetic energy of the electron
   real(8), DIMENSION(:,:), INTENT(in) :: Ip_elements	! ionization potentials [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Ekin_elements	! kinetic energies [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Nel_elements	! numbers of electrons in shells
   integer, DIMENSION(:,:), INTENT(in) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements   
   integer, DIMENSION(:), INTENT(in) :: N_elements	! number of elements in 1 molecule
   integer, DIMENSION(:), INTENT(in) :: Z_elements ! atomic number of this element
   REAL(8), DIMENSION(:), INTENT(in) :: M_elements ! mass of this element
   integer, intent(in) :: NOEL, shl
   real(8), intent(out) :: dE 	! [eV] transferred energy
   type(Error_handling), optional, intent(inout) :: Error_message ! deals with errors, if any
   type(MFP_from_file), dimension(:,:), intent(in), optional :: IMFPs	! inelastic mean free paths from files
   real(8), intent(in) :: At_dens
   type(CDF), dimension(:,:), intent(inout), optional :: Ritchi    ! Ritchi CDF to use for cross-sections

   real(8) RN, Sigma, dSigma_int, Sigma_cur, E_min, E_max, E_cur
   character(100) Writing_var
   integer coun

   call random_number(RN)
   coun = 0
   E_cur = 0.0d0

   if (T .GE. Ip_elements(NOEL, shl)) then
      call Cross_section(Noc, T, NOEL, shl, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, Sigma, IMFPs, At_dens, Ritchi)
      dSigma_int = RN*Sigma
      select case (Noc(NOEL, shl))
         case (2)
            E_max = (T-Ip_elements(NOEL, shl)) !*0.5d0
            Sigma_cur = dSigma_IMFP_int(IMFPs, NOEL, shl, Ip_elements, T, E_max, At_dens, Nel_elements)
         case (3)
!             print*, 'Energy_transfer #1'
            call dSigma_IMFP_CDF(T, IMFPs, Ritchi, Ip_elements, N_elements, NOEL, shl, At_dens, dSigma_int, E_cur)
!             print*, 'Energy_transfer #1.1', E_cur, T
         case default
            E_max = (T-Ip_elements(NOEL, shl))*0.5d0
            Sigma_cur = dSigma_int_BEB(T, E_max, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, NOEL, shl)
      endselect
      E_min = 0.0d0	!Ip_elements(NOEL, shl)

!             call dSigma_IMFP_int(IMFPs, KOA, shl, Ip_elements, T, E_max)
      if (Noc(NOEL, shl) .NE. 3) then
       do while (ABS(Sigma_cur-dSigma_int)/dSigma_int .GT. 0.0001d0) ! search the transferred energy by bisection:
         coun = coun + 1	! just count the loops
         select case (Noc(NOEL, shl))
            case (2)
               Sigma_cur = dSigma_IMFP_int(IMFPs, NOEL, shl, Ip_elements, T, E_cur, At_dens, Nel_elements)
            case (3)
               !call dSigma_IMFP_CDF(IMFPs, NOEL, shl, Ip_elements, T, E_max, At_dens, Nel_elements)
            case default
               Sigma_cur = dSigma_int_BEB(T, E_cur, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, NOEL, shl)
         endselect
         if (Sigma_cur .LT. dSigma_int) then
            E_min = E_cur
         else
            E_max = E_cur
         endif
         E_cur = (E_max+E_min)*0.5d0
!          write(*,'(f,f,e,e,f,f,e)') T, Ip_elements(NOEL, shl), dSigma_int, Sigma_cur, E_max, E_min, Sigma
         if (coun .GE. 1.d3) then 
            Writing_var = 'TOO MANY LOOPS IN BISECTION...'
            write(*,'(a,e25.16,e25.16,e25.16,e25.16,e25.16)') trim(adjustl(Writing_var)), E_max, E_min, dSigma_int, Sigma_cur, Sigma
            call Save_error_details(Error_message, 6, Writing_var)
	    exit
         endif

       enddo
!        stop 'Energy_transfer'
      endif !(Noc(NOEL, shl) .NE. 3) then
      !print*, Sigma_cur, dSigma_int, Sigma
   else
      Writing_var = 'The energy is smaller than Ip'
      write(*,'(a,f25.16,f25.16,i12,i12)') trim(adjustl(Writing_var)), T, Ip_elements(NOEL, shl), NOEL, shl
      call Save_error_details(Error_message, 7, Writing_var)
   endif


   select case (Noc(NOEL, shl))
      case(3)
         dE = min(E_cur, T)
      case default
         dE = Ip_elements(NOEL, shl) + E_cur
   endselect

!    select case (Noc(NOEL, shl))
!       case (2)	! Precalculated IMFPs:
!          if (T .GE. Ip_elements(NOEL, shl)) then
!             call Cross_section(Noc, T, NOEL, shl, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, Sigma, IMFPs, At_dens)
!             dSigma_int = RN*Sigma
!             E_max = (T-Ip_elements(NOEL, shl)) !*0.5d0
!             E_min = 0.0d0	!Ip_elements(NOEL, shl)
!             !Sigma_cur = dSigma_int_BEB(T, E_max, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, NOEL, shl)
!             Sigma_cur = dSigma_IMFP_int(IMFPs, NOEL, shl, Ip_elements, T, E_max, At_dens, Nel_elements)
!             do while (ABS(Sigma_cur-dSigma_int)/dSigma_int .GT. 0.0001d0) ! search the transferred energy by bisection:
!                coun = coun + 1	! just count the loops
!                !Sigma_cur = dSigma_int_BEB(T, E_cur, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, NOEL, shl)
!                Sigma_cur = dSigma_IMFP_int(IMFPs, NOEL, shl, Ip_elements, T, E_cur, At_dens, Nel_elements)
!                if (Sigma_cur .LT. dSigma_int) then
!                   E_min = E_cur
!                else
!                   E_max = E_cur
!                endif
!                E_cur = (E_max+E_min)*0.5d0
! !              write(*,'(f25.16,f25.16,e25.16,e25.16,f25.16,f25.16,e25.16)') T, Ip_elements(NOEL, shl), dSigma_int, Sigma_cur, E_max, E_min, Sigma
!                if (coun .GE. 1.d3) then 
!                   Writing_var = 'TOO MANY LOOPS IN BISECTION...'
!                   write(*,'(a,e25.16,e25.16,e25.16,e25.16,e25.16)') trim(adjustl(Writing_var)), E_max, E_min, dSigma_int, Sigma_cur, Sigma
!                   call Save_error_details(Error_message, 6, Writing_var)
!                endif
!                if (coun .GE. 1.d3) exit
!             enddo
!             !print*, Sigma_cur, dSigma_int, Sigma
!          else
!             Writing_var = 'The energy is smaller than Ip'
!             write(*,'(a,f25.16,f25.16,i25.16,i25.16)') trim(adjustl(Writing_var)), T, Ip_elements(NOEL, shl), NOEL, shl
!             call Save_error_details(Error_message, 7, Writing_var)
!          endif
!       case default ! at the moment, no other cross-sections, only BEB
!          if (T .GE. Ip_elements(NOEL, shl)) then
!             call Cross_section(Noc, T, NOEL, shl, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, Sigma, IMFPs, At_dens)
!             dSigma_int = RN*Sigma
!             E_max = (T-Ip_elements(NOEL, shl))*0.5d0
!             E_min = 0.0d0	!Ip_elements(NOEL, shl)
!             Sigma_cur = dSigma_int_BEB(T, E_max, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, NOEL, shl)
! !             call dSigma_IMFP_int(IMFPs, KOA, shl, Ip_elements, T, E_max)
!             do while (ABS(Sigma_cur-dSigma_int)/dSigma_int .GT. 0.0001d0) ! search the transferred energy by bisection:
!                coun = coun + 1	! just count the loops
!                Sigma_cur = dSigma_int_BEB(T, E_cur, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, NOEL, shl)
!                !call dSigma_IMFP_int(IMFPs, KOA, shl, Ip_elements, T, E_cur)
!                if (Sigma_cur .LT. dSigma_int) then
!                   E_min = E_cur
!                else
!                   E_max = E_cur
!                endif
!                E_cur = (E_max+E_min)*0.5d0
!                !write(*,'(f25.16,f25.16,e25.16,e25.16,f25.16,f25.16,e25.16)') T, Ip_elements(NOEL, shl), dSigma_int, Sigma_cur, E_max, E_min, Sigma
!                if (coun .GE. 1.d3) then 
!                   Writing_var = 'TOO MANY LOOPS IN BISECTION...'
!                   write(*,'(a,e25.16,e25.16,e25.16,e25.16,e25.16)') trim(adjustl(Writing_var)), E_max, E_min, dSigma_int, Sigma_cur, Sigma
!                   call Save_error_details(Error_message, 6, Writing_var)
!                endif
!                if (coun .GE. 1.d3) exit
!             enddo
!             !print*, Sigma_cur, dSigma_int, Sigma
!          else
!             Writing_var = 'The energy is smaller than Ip'
!             write(*,'(a,f25.16,f25.16,i12,i12)') trim(adjustl(Writing_var)), T, Ip_elements(NOEL, shl), NOEL, shl
!             call Save_error_details(Error_message, 7, Writing_var)
!          endif
!    endselect
end subroutine Energy_transfer


pure function dSigma_int_BEB(T, w, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, NOEL, shl)
   real(8) dSigma_int_BEB   ! differential cross-section
   !integer, intent(in) :: Noc	! number of the cross-section used
   real(8), intent(in) :: T 	! [eV] kinetic energy of the electron
   real(8), intent(in) :: w 	! [eV] transferred energy
   real(8), DIMENSION(:,:), INTENT(in) :: Ip_elements	! ionization potentials [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Ekin_elements	! kinetic energies [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Nel_elements	! numbers of electrons in shells
   !integer, DIMENSION(:,:), INTENT(in) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements   
   integer, DIMENSION(:), INTENT(in) :: N_elements	! number of elements in 1 molecule
   integer, DIMENSION(:), INTENT(in) :: Z_elements ! atomic number of this element
   REAL(8), DIMENSION(:), INTENT(in) :: M_elements ! mass of this element
   integer, intent(in) :: NOEL	! number of element which is considered in this collision
   integer, intent(in) :: shl	! number of shell which is being ionized

   real(8) B	! bindning energy of atomic electron [eV]
   real(8) U	! mean kinetic energy of electron in the sub-shell [eV]
   real(8) N	! ocupation number of electrons in this shell
   real(8) t0, u0, w0, S, Sigma, dSigma0
   integer i

   B = Ip_elements(NOEL,shl)
   U = Ekin_elements(NOEL,shl)
   N = DBLE(Nel_elements(NOEL,shl))

   S = 4.0d0*g_Pi*g_a0*g_a0*N*(g_Ry/B)*(g_Ry/B) ! Eq.(4)
   t0 = T/B	! energy normalized to the Rydberg constant ! Eq.(4)
   u0 = U/B  	! kinetic energy normalized
   w0 = w/B	! transferred energy normalized

   dSigma0 = dSigma_dw_int(S, t0, u0, 0.0d0)
   dSigma_int_BEB = dSigma_dw_int(S, t0, u0, w0) - dSigma0
end function dSigma_int_BEB

pure function dSigma_dw_int(S, K, u0, w0)
   real(8) dSigma_dw_int
   real(8), intent(in) :: S, K, u0, w0
   dSigma_dw_int = S/(K+u0+1.0d0)*(-(log(w0+1.0d0)-log(K-w0))/(K+1.0d0) + (1.0d0/(K-w0)-1.0d0/(w0+1.0d0)) + log(K)*0.5d0*(1.0d0/((K-w0)*(K-w0)) - 1.0d0/((w0+1.0d0)*(w0+1.0d0))))
end function dSigma_dw_int


! function dSigma_IMFP(IMFPs, KOA, shl, Ip_elements, Ee, dE)
!    type(MFP_from_file), dimension(:,:), intent(in), optional :: IMFPs	! inelastic mean free paths from files
!    integer, intent(in) :: KOA, shl
!    real(8), DIMENSION(:,:), INTENT(in) :: Ip_elements	! ionization potentials [eV]
!    real(8), intent(in) :: Ee 	! [eV] kinetic energy of the electron
!    real(8), intent(in) :: dE	! [eV] transferred energy
!    real(8) dSigma_IMFP
!    real(8) A, Ee2, Ip2, dEIp, EedE
! 
!    if (Ee .GT. Ip_elements(KOA, shl)) then	! there is cross-section
!       Ee2 = Ee*Ee
!       Ip2 = Ip_elements(KOA,shl)*Ip_elements(KOA,shl)
!       dEIp = (dE + Ip_elements(KOA,shl))
!       EedE = (Ee - dE)
!       A = Ee2*Ip2/(Ee2 - Ip2)
!       dSigma_IMFP = IMFPs(KOA, shl)*A*(1.0d0/(dEIp*dEIp*dEIp) + 1.0d0/(EedE*EedE*EedE))
!    else	! there is no way to ionize this shell
!       dSigma_IMFP = 0.0d0
!    endif
! end function dSigma_IMFP

function dSigma_IMFP_int(IMFPs, KOA, shl, Ip_elements, Ee, dE, At_dens, Nel_elements)
   type(MFP_from_file), dimension(:,:), intent(in), optional :: IMFPs	! inelastic mean free paths from files
   integer, intent(in) :: KOA, shl
   real(8), DIMENSION(:,:), INTENT(in) :: Ip_elements	! ionization potentials [eV]
   real(8), intent(in) :: Ee 	! [eV] kinetic energy of the electron
   real(8), intent(in) :: dE	! [eV] transferred energy
   real(8), intent(in), optional ::  At_dens	! atomic density [1/cm^3]
   real(8), DIMENSION(:,:), INTENT(in) :: Nel_elements	! numbers of electrons in shells
   real(8) dSigma_IMFP_int
   real(8) A, Ee2, Ip2, dEIp, EedE, Sigma

   if (Ee .GT. Ip_elements(KOA, shl)) then	! there is cross-section
      Ee2 = Ee*Ee
      Ip2 = Ip_elements(KOA,shl)*Ip_elements(KOA,shl)
      dEIp = (dE + Ip_elements(KOA,shl))
      EedE = (Ee - dE)
      A = Ee2*Ip2/(Ee2 - Ip2)
      Sigma = Find_in_Array(IMFPs, KOA, shl, Ee, At_dens, Nel_elements)
      dSigma_IMFP_int = 0.5d0*Sigma*(1.0d0 + A*(1.0d0/(EedE*EedE)-1.0d0/(dEIp*dEIp)))
   else	! there is no way to ionize this shell
      dSigma_IMFP_int = 0.0d0
   endif
!    print*, 'S:' ,Ee, dSigma_IMFP_int, Sigma
end function dSigma_IMFP_int


pure subroutine El_free_flight_time(Noc, T, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, Tim, Ritchi, NOI, dEdx)
   integer, DIMENSION(:,:), intent(in) :: Noc	! number of the cross-section used
   real(8), intent(in) :: T 	! [eV] kinetic energy of the electron
   real(8), DIMENSION(:,:), INTENT(in) :: Ip_elements	! ionization potentials [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Ekin_elements	! kinetic energies [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Nel_elements	! numbers of electrons in shells
   integer, DIMENSION(:,:), INTENT(in) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements   
   integer, DIMENSION(:), INTENT(in) :: N_elements	! number of elements in 1 molecule
   integer, DIMENSION(:), INTENT(in) :: Z_elements ! atomic number of this element
   REAL(8), DIMENSION(:), INTENT(in) :: M_elements ! mass of this element
   real(8), intent(in), optional ::  At_dens	! atomic density [1/cm^3]
   real(8), intent(out) :: Sigma_tot 	! [A^2] cross-section for this energy
   type(MFP_from_file), dimension(:,:), intent(inout), optional :: IMFPs	! inelastic mean free paths from files
   real(8), intent(out) :: IMFP		! [A] inelastic mean free path
   real(8), intent(out) :: Tim		! [fs] free flight time
   type(CDF), dimension(:,:), intent(inout), optional :: Ritchi    ! Ritchi CDF to use for cross-sections
   integer, intent(in), optional :: NOI		! number of iteration
   real(8), intent(out), optional :: dEdx 	! electron energy loss [eV/A]

   if (present(NOI)) then
      if (present(dEdx)) then
         call Total_cross_section(Noc, T, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, Ritchi, NOI, dEdx)
      else
         call Total_cross_section(Noc, T, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, Ritchi, NOI)
      endif
   else
      call Total_cross_section(Noc, T, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, Ritchi)
   endif

   if (ANY((Ip_elements .LT. T) .AND. Ip_elements .GT. 0.0d0)) then
      Tim = IMFP/dSQRT(T*2.0d0*g_e/(g_me))*1.d5 ! [fs] free-flight time of an electron
   else
      Tim = 1.d21 ! [fs] too slow electron never makes impact ionization
   endif
!     print*, 'FFT', T, Tim, IMFP
end subroutine El_free_flight_time


pure subroutine Total_cross_section(Noc, T, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, Ritchi, NOI, dEdx)
   integer, DIMENSION(:,:), intent(in) :: Noc	! number of the cross-section used
   real(8), intent(in) :: T 	! [eV] kinetic energy of the electron
   real(8), DIMENSION(:,:), INTENT(in) :: Ip_elements	! ionization potentials [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Ekin_elements	! kinetic energies [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Nel_elements	! numbers of electrons in shells
   integer, DIMENSION(:,:), INTENT(in) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements   
   integer, DIMENSION(:), INTENT(in) :: N_elements	! number of elements in 1 molecule
   integer, DIMENSION(:), INTENT(in) :: Z_elements ! atomic number of this element
   REAL(8), DIMENSION(:), INTENT(in) :: M_elements ! mass of this element
   real(8), intent(out) :: Sigma_tot 	! [A^2] cross-section for this energy
   type(MFP_from_file), dimension(:,:), intent(inout), optional :: IMFPs	! inelastic mean free paths from files
   real(8), optional, intent(out) :: IMFP	! [A] inelastic mean free path
   type(CDF), dimension(:,:), intent(inout), optional :: Ritchi    ! Ritchi CDF to use for cross-sections
   integer, intent(in), optional :: NOI	! number of iteration
   real(8), intent(out), optional :: dEdx 	! electron energy loss [eV/A]
   real(8), intent(in) :: At_dens	! [1/cm^3] atomic density
   real(8) Sigma, dEdx_sh
   integer NoEiC, j, k, M
   NoEiC = size(Ip_elements,1) ! elements/spieces
   M = size(Ip_elements,2) ! shells
   Sigma_tot = 0.0d0
   if (present(dEdx)) dEdx = 0.0d0

   do j = 1, NoEiC
      do k = 1, M
         if ((Nel_elements(j,k) .GT. 0.0d0) .AND. (T .GT. Ip_elements(j,k))) then
            call Cross_section(Noc, T, j, k, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, Sigma, IMFPs, At_dens, Ritchi)
            select case(Noc(j, k))
              case (2)
                Sigma_tot = Sigma_tot + Sigma !*N_elements(j)/SUM(N_elements) ! [A^2] cross-section
              case (3)
                Sigma_tot = Sigma_tot + Sigma !*N_elements(j)/SUM(N_elements) ! [A^2] cross-section
                if ((present(NOI)) .AND. (IMFPs(j,k)%E(NOI) .LE. 0.0d0)) then
                  IMFPs(j,k)%E(NOI) = T
                  IMFPs(j,k)%MFP(NOI) = 1.d8/((At_dens*N_elements(j))*Sigma*1.d-16) ! [A] mean free path
                endif
              case default ! beb:
                Sigma_tot = Sigma_tot + Sigma*N_elements(j)/SUM(N_elements) ! [A^2] cross-section
                if (present(NOI)) then
                  if (IMFPs(j,k)%E(NOI) .LE. 0.d0) then
                     IMFPs(j,k)%E(NOI) = T
                     !IMFPs(j,k)%MFP(NOI) = 1.0d8/((At_dens*N_elements(j))*Sigma*1d-16) ! [A] mean free path
                     IMFPs(j,k)%MFP(NOI) = Sigma ! [A^2] cross-section
                  endif
                endif
                if (present(dEdx)) then ! calculate electron energy loss
                   call get_dEdx(T, Ip_elements, j, k, Nel_elements, Ekin_elements, At_dens, dEdx_sh)
                   dEdx = dEdx + dEdx_sh*N_elements(j)/SUM(N_elements) ! [A^2] cross-section
                endif
            endselect
!             write(*,'(a2, i2, i2, 1f8.1, e25.16, e25.16, i12)') 'T', j, k, T, IMFP, Sigma, Noc(j, k)
         endif ! (Nel_elements(j,k) .GT. 0)
      enddo ! k
   enddo ! j

   if (present(IMFP)) then
      if (Sigma_tot .GT. 0.0d0) then 
         select case(Noc(1, 1))
            case (3)
               IMFP = 1.d8/((At_dens*SUM(N_elements))*Sigma_tot*1.d-16) ! [A] mean free path
            case default
               IMFP = 1.d8/((At_dens)*Sigma_tot*1.d-16) ! [A] mean free path
            endselect
      else
         IMFP = 1.d22	! infinity
      endif
   endif
end subroutine Total_cross_section


pure subroutine get_dEdx(T, Ip_elements, NOEL,shl, Nel_elements, Ekin_elements, At_dens, dEdx)
   real(8), intent(in) :: T 	! [eV] kinetic energy of the electron
   integer, intent(in) :: NOEL	! number of element which is considered in this collision
   integer, intent(in) :: shl	! number of shell which is being ionized
   real(8), DIMENSION(:,:), INTENT(in) :: Ip_elements	! ionization potentials [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Ekin_elements	! kinetic energies [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Nel_elements	! numbers of electrons in shells
   real(8), intent(in) :: At_dens	! [1/cm^3] atomic density
   real(8), intent(out) :: dEdx ! energy loss [eV/A]
   real(8) B	! bindning energy of atomic electron [eV]
   real(8) U	! mean kinetic energy of electron in the sub-shell [eV]
   real(8) N	! ocupation number of electrons in this shell
   B = Ip_elements(NOEL,shl)
   N = DBLE(Nel_elements(NOEL,shl))
   U = Ekin_elements(NOEL,shl)
   dEdx = (At_dens*1.d-24)*B*dEdx_BEB(T,B,U,N) ! [eV/A]
end subroutine get_dEdx 


pure subroutine Cross_section(Noc, T, NOEL, shl, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, Sigma, IMFPs, At_dens, Ritchi)
   integer, DIMENSION(:,:), intent(in) :: Noc	! number of the cross-section used
   real(8), intent(in) :: T 	! [eV] kinetic energy of the electron
   integer, intent(in) :: NOEL	! number of element which is considered in this collision
   integer, intent(in) :: shl	! number of shell which is being ionized
   real(8), DIMENSION(:,:), INTENT(in) :: Ip_elements	! ionization potentials [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Ekin_elements	! kinetic energies [eV]
   real(8), DIMENSION(:,:), INTENT(in) :: Nel_elements	! numbers of electrons in shells
   integer, DIMENSION(:,:), INTENT(in) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements   
   integer, DIMENSION(:), INTENT(in) :: Z_elements ! atomic number of this element
   REAL(8), DIMENSION(:), INTENT(in) :: M_elements ! mass of this element
   real(8), intent(out) :: Sigma 	! [A^2] cross-section for this energy
   type(MFP_from_file), dimension(:,:), intent(in), optional :: IMFPs	! inelastic mean free paths from files
   real(8), intent(in) :: At_dens	! [1/cm^3] atomic density
   type(CDF), dimension(:,:), intent(inout), optional :: Ritchi    ! Ritchi CDF to use for cross-sections

   real(8) B	! bindning energy of atomic electron [eV]
   real(8) U	! mean kinetic energy of electron in the sub-shell [eV]
   real(8) N	! ocupation number of electrons in this shell
   real(8) Z	! atomic number
   real(8) M	! mass of the element (for elastic cross-section, if needed)
   real(8) n_p	! principal quantum number
   real(8) N_eff, N_eff2 ! effective number of electrons for effective charge

   B = Ip_elements(NOEL,shl)
   N = DBLE(Nel_elements(NOEL,shl))

   SELECT CASE (Noc(NOEL,shl))
      CASE(:1)	! Original BEB
         U = Ekin_elements(NOEL,shl)
         Sigma = Sigma_BEB(T,B,U,N)
      CASE(2)	! IMFP read from a file
         Sigma = Find_in_Array(IMFPs, NOEL, shl, T, At_dens, Nel_elements)
      CASE(3)	! CDF calculation
         !call CDF_IMFP(T, Ritchi, Ip_elements, NOEL, shl, Sigma, dEdx)
         call CDF_IMFP(T, Ritchi, Ip_elements, NOEL, shl, Sigma)
      CASE(100)	! MBEB
         Z = Z_elements(NOEL)
         n_p = DBLE(PQN_elements(NOEL,shl))
         N_eff = SUM(Nel_elements(NOEL,shl:))
         if (shl .LT. size(Nel_elements,2)) then
            N_eff2 = SUM(Nel_elements(NOEL,(shl+1):))
         else
            N_eff2 = 0.0d0
         endif
         Sigma = Sigma_MBEB(shl,T,B,N,Z,n_p,N_eff,N_eff2)
      CASE DEFAULT ! use BEB by default
         U = Ekin_elements(NOEL,shl)
         Sigma = Sigma_BEB(T,B,U,N)
   END SELECT
!    print*, 'CS:', T, Sigma
end subroutine Cross_section

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! CDF:

subroutine dSigma_IMFP_CDF(Ele, IMFPs, Ritchi, Ip_elements, N_elements, NOEL, shl, At_dens, dSigma_int, dE_out, dEdx)
    real(8), intent(in) :: Ele  ! electron energy [eV]
    type(MFP_from_file), dimension(:,:), intent(in) :: IMFPs	! inelastic mean free paths from files
    type(CDF), dimension(:,:), intent(inout) :: Ritchi    ! Ritchi CDF to use for cross-sections
    real(8), dimension(:,:), intent(in) :: Ip_elements  ! ionization potentials all shells of all elements [eV]
    integer, DIMENSION(:), INTENT(in) :: N_elements	! numbers of electrons in shells -no!
    integer, intent(in) :: NOEL	! number of element which is considered in this collision
    integer, intent(in) :: shl	! number of shell which is being ionized
    real(8), intent(in) :: At_dens	! atomic density
    real(8), intent(in) :: dSigma_int	! integrated cross-section
    real(8), intent(out) :: dE_out	! calculated energy loss out of cross-section
    real(8), intent(inout), optional :: dEdx   ! calculated inverse mean free path[1/A], and energy losses [eV/A]

    integer i, j, n, int_meth
    real(8) Emin, Emax, E, dE, dL, Ltot1, Ltot0, ddEdx, a, b, temp1, temp2

    Emin = Ip_elements(NOEL, shl)  ! min transferred energy [eV]
    !if (Emin .LE. 1d-6) Emin = 1.d-6 ! for metals there is no band gap
    if (Emin .LE. 1.0d-3) Emin = 1.0d-3 ! for metals there is no band gap
    Emax = (Ele + Emin)*0.5d0 ! [eV] maximum energy transfer, accounting for equality of electrons
    !n = 100    ! number of integration steps
    n = 10*(MAX(INT(Emin),10))    ! number of integration steps
    dE = (Emax - Emin)/(DBLE(n)) ! differential of transferred momentum [kg*m/s]
    i = 1       ! to start integration
    E = Emin    ! to start integration
    Ltot1 = 0.0d0
    Ltot0 = 0.0d0
    !call Diff_cross_section(Ele, E, A0, d0, Gamma0, Nff_esh, Nosh, Ltot0)
    call Diff_cross_section(Ele, E, Ritchi, NOEL, shl, Ltot0)
    ddEdx = 0.0d0
    int_meth = 1	! integration method

    do while (Ltot1 .LT. dSigma_int) ! integration

        !dE = E/DBLE(n)
        dE = (1.0d0/(E+1.0d0) + E)/DBLE(n)
        ! i = i + 1
        SELECT CASE (int_meth)
        CASE(:1)
           ! If it's Simpson integration:
           a =  E + dE*0.5d0
           !call Diff_cross_section(Ele, a, A0, d0, Gamma0, Nff_esh, Nosh, dL)
           call Diff_cross_section(Ele, a, Ritchi, NOEL, shl, dL)
           temp1 = dL
           b = E + dE
           !call Diff_cross_section(Ele, b, A0, d0, Gamma0, Nff_esh, Nosh, dL)
           call Diff_cross_section(Ele, b, Ritchi, NOEL, shl, dL)
           temp2 = dE/6.0d0*(Ltot0 + 4.0d0*temp1 + dL)
           Ltot1 = Ltot1 + temp2
           ddEdx = ddEdx + E*temp2
        CASE(2)        ! Method trapeziod
           !call Diff_cross_section(Ele, E, A0, d0, Gamma0, Nff_esh, Nosh, dL)
           call Diff_cross_section(Ele, E, Ritchi, NOEL, shl, dL)
           !write(*,'(f25.16,f25.16,f25.16,f25.16,e25.16)') dE, Emin, Emax, E, dL
           if (i .EQ. 1) then 
              Ltot1 = Ltot1 + dL
              ddEdx = ddEdx + E*dL
           else
              Ltot1 = Ltot1 + (dL + Ltot0)*0.5d0
              ddEdx = ddEdx + E*(dL + Ltot0)*0.5d0
           endif
        END SELECT
        Ltot0 = dL

        E = E + dE  ! [eV]
    enddo

    dE_out = E

end subroutine dSigma_IMFP_CDF



pure subroutine CDF_IMFP(Ele, Ritchi, Ip_elements, NOEL, shl, Sigma, dEdx)
    real(8), intent(in) :: Ele  ! electron energy [eV]
    type(CDF), dimension(:,:), intent(inout) :: Ritchi    ! Ritchi CDF to use for cross-sections
    real(8), dimension(:,:), intent(in) :: Ip_elements  ! ionization potentials all shells of all elements [eV]
    integer, intent(in) :: NOEL	! number of element which is considered in this collision
    integer, intent(in) :: shl	! number of shell which is being ionized
    real(8), intent(out) :: Sigma
    real(8), intent(out), optional :: dEdx   ! calculated inverse mean free path[1/A], and energy losses [eV/A]

    integer i, j, n, int_meth
    real(8) Emin, Emax, E, dE, dL, Ltot1, Ltot0, ddEdx, a, b, temp1, temp2

    Emin = Ip_elements(NOEL, shl)  ! min transferred energy [eV]
    !if (Emin .LE. 1d-6) Emin = 1.d-6 ! for metals there is no band gap
    if (Emin .LE. 1.0d-3) Emin = 1.0d-3 ! for metals there is no band gap
    Emax = (Ele + Emin)*0.5d0 ! [eV] maximum energy transfer, accounting for equality of electrons
    !n = 100    ! number of integration steps
    n = 10*(MAX(INT(Emin),10))    ! number of integration steps
    dE = (Emax - Emin)/(DBLE(n)) ! differential of transferred momentum [kg*m/s]
    i = 1       ! to start integration
    E = Emin    ! to start integration
    Ltot1 = 0.0d0
    Ltot0 = 0.0d0
    !call Diff_cross_section(Ele, E, A0, d0, Gamma0, Nff_esh, Nosh, Ltot0)
    call Diff_cross_section(Ele, E, Ritchi, NOEL, shl, Ltot0)
    ddEdx = 0.0d0
    int_meth = 1	! integration method
    do while (E .LE. Emax) ! integration
        !dE = E/DBLE(n)
        dE = (1.0d0/(E+1.0d0) + E)/DBLE(n)
        ! i = i + 1
        SELECT CASE (int_meth)
        CASE(:1)
           ! If it's Simpson integration:
           a =  E + dE*0.5d0
           !call Diff_cross_section(Ele, a, A0, d0, Gamma0, Nff_esh, Nosh, dL)
           call Diff_cross_section(Ele, a, Ritchi, NOEL, shl, dL)
           temp1 = dL
           b = E + dE
           !call Diff_cross_section(Ele, b, A0, d0, Gamma0, Nff_esh, Nosh, dL)
           call Diff_cross_section(Ele, b, Ritchi, NOEL, shl, dL)
           temp2 = dE/6.0d0*(Ltot0 + 4.0d0*temp1 + dL)
           Ltot1 = Ltot1 + temp2
           ddEdx = ddEdx + E*temp2
        CASE(2)        ! Method trapeziod
           !call Diff_cross_section(Ele, E, A0, d0, Gamma0, Nff_esh, Nosh, dL)
           call Diff_cross_section(Ele, E, Ritchi, NOEL, shl, dL)
           !write(*,'(f,f,f,f,e)') dE, Emin, Emax, E, dL
           if (i .EQ. 1) then 
              Ltot1 = Ltot1 + dL
              ddEdx = ddEdx + E*dL
           else
              Ltot1 = Ltot1 + (dL + Ltot0)*0.5d0
              ddEdx = ddEdx + E*(dL + Ltot0)*0.5d0
           endif
        END SELECT
        Ltot0 = dL
        E = E + dE  ! [eV]
    enddo
    Sigma = Ltot1 !*dE ! [1/A]
    if (present(dEdx)) dEdx = ddEdx !*dE ! energy losses [eV/A]
end subroutine CDF_IMFP


pure subroutine Diff_cross_section(Ele, hw, Ritchi, NOEL, shl, Diff_IMFP)
    real(8), intent(in) :: Ele  ! electron energy [eV]
    REAL(8), INTENT(in) :: hw   ! transferred energy [eV]
    type(CDF), dimension(:,:), intent(inout) :: Ritchi    ! Ritchi CDF to use for cross-sections
    integer, intent(in) :: NOEL	! number of element which is considered in this collision
    integer, intent(in) :: shl	! number of shell which is being ionized
    real(8), intent(out) :: Diff_IMFP   ! differential inverse mean free path 1/lambda(Ele,dE)

    integer i, n, int_meth
    real(8) dLs, qmin, qmax, hq, ddq, dq, Ee, dE, Ime, dLs0, dL, hq0, dq_save
    real(8) a, b, x, temp1, temp2
    !Ee = Ele/g_e    ! energy [J]
    Ee = Ele        ! energy [eV]
    !dE = hw/g_e     ! transferred energy [J]
    dE = hw         ! transferred energy [eV]
    qmin = sqrt(2.0d0*g_me)/g_h*(sqrt(Ee) - sqrt((Ee - dE))) ! min transferred momentum [kg*m/s]
    qmax = sqrt(2.0d0*g_me)/g_h*(sqrt(Ee) + sqrt((Ee - dE))) ! max transferred momentum [kg*m/s]
    dLs = 0.0d0 ! starting integration, mean free path per energy [A/eV]^(-1)
    hq = qmin    ! transient transferred momentum for integration [kg*m/s]
    n = 100
    dq = (qmax - qmin)/DBLE(n) ! differential of transferred momentum [kg*m/s]
    !dq = qmin/DBLE(n)
    dLs0 = 0.0d0
    int_meth = 1	! integration method
    do while (hq .LT. qmax) ! no matter how many points, go till the end
        dq = hq/DBLE(n)
        SELECT CASE (int_meth)
        CASE (:1)           ! If it's Simpson integration:
           a = hq + dq*0.5d0
           !call Imewq(hw, a, ImE, A0, d0, Gamma0, Nff_esh, Nosh)
           call Imewq(hw, a, ImE, Ritchi, NOEL, shl)
           temp1 = ImE
           b = hq + dq
           !call Imewq(hw, b, ImE, A0, d0, Gamma0, Nff_esh, Nosh)
           call Imewq(hw, b, ImE, Ritchi, NOEL, shl)
           dL = ImE
           dLs = dLs + dq/6.0d0*(dLs0 + 4.0d0*temp1 + dL)/hq
        CASE(2)           ! It's Gaussian integration:
           a =  hq + dq*0.5d0
           b = dq/(2.0d0*sqrt(3.0d0))
           x = a + b
           !call Imewq(hw, x, ImE, A0, d0, Gamma0, Nff_esh, Nosh)
           call Imewq(hw, x, ImE, Ritchi, NOEL, shl)
           temp1 = ImE
           x = a - b
           !call Imewq(hw, x, ImE, A0, d0, Gamma0, Nff_esh, Nosh)
           call Imewq(hw, x, ImE, Ritchi, NOEL, shl)
           temp2 = ImE
           dLs = dLs + dq*0.5d0*(temp1 + temp2)/hq
        CASE(3)           ! It's trapeziod method of integration:
           !call Imewq(hw, hq, ImE, A0, d0, Gamma0, Nff_esh, Nosh)
           call Imewq(hw, hq, ImE, Ritchi, NOEL, shl)
           dL = ImE*dq/hq
           !call Square_int(dL, dLs)
           call Trapez_int(dLs0, dL, dLs)
        ENDSELECT
        dLs0 = dL
        hq = hq + dq
    enddo
    Diff_IMFP = 1.0d0/(g_Pi*g_a0*Ele)*dLs
end subroutine Diff_cross_section

pure subroutine Square_int(y0, int_y)
    real(8), intent(in) :: y0
    real(8), intent(inout) :: int_y
    int_y = int_y + y0
end subroutine Square_int

pure subroutine Trapez_int(y0, y1, int_y)
    real(8), intent(in) :: y0, y1
    real(8), intent(inout) :: int_y
    int_y = int_y + (y0 + y1)*0.5d0
end subroutine Trapez_int

pure subroutine Imewq(hw, hq, ImE, Ritchi, NOEL, shl) ! constructs Im(-1/e(w,q)) as a sum of Drude-like functions
    REAL(8), INTENT(in) ::  hw    ! transferred energy [eV]
    REAL(8), INTENT(in) ::  hq    ! transferred momentum [kg*m/s]
    REAL(8), INTENT(out) :: ImE
    type(CDF), dimension(:,:), intent(in) :: Ritchi    ! Ritchi CDF to use for cross-sections
    integer, intent(in) :: NOEL	! number of element which is considered in this collision
    integer, intent(in) :: shl	! number of shell which is being ionized
    !integer, intent(in) :: Index    ! number of shell
    real(8) sumf, A, Gamma, E
    integer i, N
    ImE = 0.0d0
    N = size(Ritchi(NOEL,shl)%A)
    do i = 1, N !Nff_esh(Nosh)
        A = Ritchi(NOEL,shl)%A(i)
        E = Ritchi(NOEL,shl)%E0(i)
        Gamma = Ritchi(NOEL,shl)%Gamma(i)
        sumf = Diel_func(A,E,Gamma, hw, hq) ! Diel_func - function, see below
        ImE = ImE + sumf
    enddo
end subroutine Imewq

pure function Diel_func(A,E,Gamma,dE,dq) ! fit functions in Ritchi algorithm
    real(8) Diel_func ! function itself
    real(8), intent(in) :: A, E, Gamma, dE, dq  ! parameters and variable
    real(8) d0  ! temporary parameter
    d0 = E + g_h*g_h*dq*dq/(2.0d0*g_me)
    Diel_func = A*Gamma*dE/((dE*dE - d0*d0)*(dE*dE - d0*d0) + Gamma*Gamma*dE*dE)
end function Diel_func
! <-CDF
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!



pure function Find_in_Array(IMFPs, NOEL, shl, T, At_dens, Nel_elements)
   real(8) Find_in_Array  !  [A^2] cross-section for this energy
   type(MFP_from_file), dimension(:,:), intent(in) :: IMFPs	! inelastic mean free paths from files
   integer, intent(in) :: NOEL	! number of element which is considered in this collision
   integer, intent(in) :: shl	! number of shell which is being ionized
   real(8), intent(in) :: T 	! [eV] kinetic energy of the electron
   real(8), intent(in) :: At_dens	! [1/cm^3] atomic density
   real(8), DIMENSION(:,:), INTENT(in) :: Nel_elements	! numbers of electrons in shells
   integer i
   real(8) E, IMFP, Norm
   real(8) Sigma ! this will be the result

   Norm = 1.0d0
!    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!    Norm = 2.0d0
!    if ((NOEL .EQ. 1) .AND. (shl .EQ. 2)) Norm = 8.0d0
!    if ((NOEL .EQ. 1) .AND. (shl .EQ. 3)) Norm = 4.0d0
!    if ((NOEL .EQ. 2) .AND. (shl .EQ. 2)) Norm = 6.0d0
!    !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!    print*, 'TEST 0', T
   if (IMFPs(NOEL,shl)%E(1) .GT. T) then	! T>d0
      Sigma = 0.0d0
   else
      if (T .GE. IMFPs(NOEL,shl)%E(size(IMFPs(NOEL,shl)%E))) then
         Sigma = 1.0d24/(At_dens*IMFPs(NOEL,shl)%MFP(size(IMFPs(NOEL,shl)%E)))	! [A^2]
      else
         i = 1
         do while (IMFPs(NOEL,shl)%E(i) .LT. T)
            i = i + 1
         enddo
         !i = i - 1
         !IMFP = IMFPs(NOEL,shl)%MFP(i-1) + (IMFPs(NOEL,shl)%MFP(i)-IMFPs(NOEL,shl)%MFP(i-1)) / (IMFPs(NOEL,shl)%E(i)-IMFPs(NOEL,shl)%E(i-1))*(T-IMFPs(NOEL,shl)%E(i-1))
!          print*, 'i', i, IMFPs(NOEL,shl)%MFP(i)
         if (i .EQ. 1) then
            IMFP = IMFPs(NOEL,shl)%MFP(i)	! [A]
!             print*, 'i = 1', IMFP 
         else
!             print*, 'i =/= 1'
            IMFP = Linear_interpolation(IMFPs(NOEL,shl)%MFP(i-1), IMFPs(NOEL,shl)%MFP(i), IMFPs(NOEL,shl)%E(i-1), IMFPs(NOEL,shl)%E(i), T)
         endif
!          print*, 'TEST 1'
!          print*, At_dens
!          print*, IMFP
         Sigma = 1.0d24/(At_dens*IMFP)	! [A^2]
!          print*, 'TEST 1.1'
      endif
   endif
!    print*, 'TEST', T, Sigma, IMFPs(NOEL,shl)%E(i), IMFPs(NOEL,shl)%MFP(i)
   Find_in_Array = Sigma/Norm	! normalize it according to IMFP file -- different for each file!!!
!    write(*,'(a6, i2, i2, f7.2, e, e)') 'Sigma', NOEL, shl, T, Sigma, IMFPs(NOEL,shl)%E(1)
end function Find_in_Array

pure function Linear_interpolation(y1, y2, x1, x2, x)
   real(8) Linear_interpolation
   real(8), intent(in) :: y1, y2, x1, x2, x
   Linear_interpolation = y1 + (y2-y1)/(x2-x1)*(x-x1)
end function Linear_interpolation


! Original BEB cross-section, Eq.(57) [Y.K.Kim, M.E.Rudd, Phys.Rev.A 50 (1994) 3954]
pure function Sigma_BEB(T,B,U,N)
   real(8) Sigma_BEB ! cross-section [A^2]
   real(8), intent(in) :: T	! kinetic energy of incident electron [eV]
   real(8), intent(in) :: B	! bindning energy of atomic electron [eV]
   real(8), intent(in) :: U	! mean kinetic energy of electron in sub-shell [eV]
   real(8), intent(in) :: N	! ocupation number of electrons in this shell
   real(8) K, u0, S
   S = 4.0d0*g_Pi*g_a0*g_a0*N*(g_Ry/B)*(g_Ry/B) ! Eq.(4)
   K = T/B	! energy normalized to the Rydberg constant ! Eq.(4)
   u0 = U/B
   Sigma_BEB = S/(K+u0+1.0d0)*(log(K)*0.5d0*(1.0d0-1.0d0/(K*K)) + (1.0d0-1.0d0/K) -log(K)/(K+1.0d0))
end function Sigma_BEB


! Energy loss corresponding to BEB cross-section, Eq.(57) [Y.K.Kim, M.E.Rudd, Phys.Rev.A 50 (1994) 3954]
pure function dEdx_BEB(T,B,U,N)
   !real(8) Sigma_BEB ! cross-section [A^2]
   real(8) dEdx_BEB   ! Energy loss [A^2], not normalized yet: [na*Ip*dEdx_BEB] = [eV/A]
   real(8), intent(in) :: T	! kinetic energy of incident electron [eV]
   real(8), intent(in) :: B	! bindning energy of atomic electron [eV]
   real(8), intent(in) :: U	! mean kinetic energy of electron in sub-shell [eV]
   real(8), intent(in) :: N	! ocupation number of electrons in this shell
   real(8) t0, u0, S, t01, t0m1, logt, log2
   S = 4.0d0*g_Pi*g_a0*g_a0*N*(g_Ry/B)*(g_Ry/B) ! Eq.(4)
   t0 = T/B	! energy normalized to the Rydberg constant ! Eq.(4)
   u0 = U/B
   !Sigma_BEB = S/(t0+u0+1.0d0)*(log(t0)*0.5d0*(1.0d0-1.0d0/(t0*t0)) + (1.0d0-1.0d0/t0) -log(t0)/(t0+1.0d0))
   logt = log(t0)
   t01 = t0 + 1.0d0
   t0m1 = t0 - 1.0d0
   log2 = 2.0d0*log(2.0d0)
   dEdx_BEB = S/(t0+u0+1.0d0)*( (-logt-log(t0m1))/(t01) + log2/(t01*t01) + 2.0d0*log(t01) - logt - log2 + logt*0.5d0*t0m1*t0m1/(t0*t01)) ! [A^2]
end function dEdx_BEB



! Modified BEB cross-section, Eq.(3) [M. Guerra et al. Int. J. Mass Spectr 313 (2012) 1-7]
pure function Sigma_MBEB(shl,T,B,N,Z,n_p,N_eff,N_eff2)
   real(8) Sigma_MBEB ! cross-section [A^2]
   integer, intent(in) :: shl	! number of shell which is being ionized
   real(8), intent(in) :: T	! kinetic energy of incident electron [eV]
   real(8), intent(in) :: B	! bindning energy of atomic electron [eV]
   real(8), intent(in) :: N	! ocupation number of electrons in this shell
   real(8), intent(in) :: Z	! atmoic number
   real(8), intent(in) :: n_p	! principal quantum number
   real(8), intent(in) :: N_eff, N_eff2 ! effective number of electrons for effective charge

   real(8) t0, c0, C, S, a0, b0, Zeff, Zeff2, n2
   a0 = 0.3d0
   b0 = 0.7d0

   call Z_eff_all_shells(shl, Z, N_eff, N_eff2, Zeff, Zeff2)
   n2 = n_p + 1.0d0

   C = a0*Zeff*Zeff/(2.0d0*n_p*n_p) + b0*Zeff2*Zeff2/(2.0d0*n2*n2) ! Eq.(1)
   t0 = T/B	! energy normalized to the Rydberg constant ! Eq.(4)
   c0 = C/B*2.0d0*g_Ry ! Eq.(4)
   S = 4.0d0*g_Pi*g_a0*g_a0*N*(g_Ry/B)*(g_Ry/B) ! Eq.(4)
   Sigma_MBEB = S/(t0+c0)*(0.5d0*(1.0d0-1.0d0/(t0*t0))*log(t0) + (1.0d0-1.0d0/t0) - log(t0)/(t0+1.0d0)) ! Eq.(3)
end function Sigma_MBEB


pure subroutine Z_eff_all_shells(shl, Z, N_eff, N_eff2, Zeff1, Zeff2)
   integer, intent(in) :: shl	! shell number
   real(8), intent(in) :: Z	! atomic number
   real(8), intent(in) :: N_eff, N_eff2 ! effective number of electrons for effective charge
   real(8), intent(out) :: Zeff1, Zeff2
   if (shl .EQ. 1) then	! K-shell, use Eq.(2)
      Zeff1 = 0.9834d0*Z - 0.1947d0
      Zeff2 = 0.7558d0*Z - 1.1724d0
   else
      Zeff1 = N_eff
      Zeff2 = N_eff2
   endif
end subroutine Z_eff_all_shells


pure function Total_elastic_cross_section(N_elements, Z_elements, kin_nrg, meff)

    REAL(8) Total_elastic_cross_section
 
    integer, DIMENSION(:), INTENT(in) :: N_elements	! how many each of them presents there, i.e. Si02 means 1 and 2
    integer, DIMENSION(:), INTENT(in) :: Z_elements ! atomic number of each element
    real(8), intent(in) :: kin_nrg 	! [eV] kinetic energy of the electron
    real(8), intent(in) :: meff 	! [kg] effective mass

    integer coun_atoms, NoEiC	! number of different elements
    real(8) vel_mod		! modulus of the electron velocity
    real(8) sigma_el		! collecting the result in this variable 

    NoEiC = size(N_elements,1) ! elements/spieces

    vel_mod = dsqrt(2.d0/meff*kin_nrg*g_e)

    sigma_el = 0.d0
    do coun_atoms = 1, NoEiC
        sigma_el = sigma_el + Elastic_cross_section(kin_nrg,Z_elements(coun_atoms),meff) *dble(N_elements(coun_atoms))
    enddo

    Total_elastic_cross_section = sigma_el / sum(N_elements(:))	! average per-atom cross section; to be used with atomic density

end function Total_elastic_cross_section



pure function Elastic_cross_section(nrg_ev,Z,meff)
!  Elastic electron-atom scattering
!  [Plante and Cucinotta; Jenkins et al.]

	real(8) Elastic_cross_section	! [A^2]

	real(8), intent(in) :: nrg_ev  ! kinetic energy of the electron in eV
	integer, intent(in) :: Z    !  atomic number
	real(8), intent(in) :: meff  ! effective mass in the units of electron mass
	REAL(8) Pi, clight, alpha, me, el, re
	real(8) E_ev

	real(8) tau, eta_prime, beta_sq
	real(8) Z_real, mass

	! vel = vel_SI*1.d5		! m/s -> A/fs; internal units of this function
	! E_ev   = vel*vel/36.d0	!  [eV]
	E_ev   = nrg_ev
	z_real = dble(Z)

       if (E_ev.gt.0.d0) then

		Pi = 3.1415926535897932d0
		clight = 299792458.d0		! speed of light (m/s)
		alpha = 0.0072973525664d0	! fine-structure constant, ~1/137
		me = 9.10938356d-31		! electron mass
		el = 1.6021766208d-19		! electron charge
		re = 2.8179403267d-15		! classical electron radius
		
		mass = meff*me

		beta_sq = 2.0d0*E_ev*el/(mass*clight**2)	! speed of the electron in units of c, or simply v/c,  squared
		tau = E_ev*el/(mass*clight**2)			! kinetic energy of the electron in units of mc^2

		! T.M. Jenkins, W.R. Nelson, A. Rindi. Monte Carlo Transport of Electrons and Photons. Springer Science & Business Media, 2012. ISBN: 1461310598, 9781461310594. Formula 7.8.
		eta_prime = 0.25d0*(alpha/(0.885d0))**2 /beta_sq * (z_real)**(2.d0/3.d0) * ( 1.13d0 + 3.76d0*(alpha*z_real)**2/beta_sq * dsqrt(tau/(tau+1.d0)) )	! Epmirically modified Moliere's screening
		! (see also unmodified screening: A.F. Akkerman. Modelirovanie traektoriy zaryazhennyh chastits v veschestve. M.: Energoatomizdat, 1991. ISBN: 5-283-02924-7. Formula 1.17.)
		! eta_M = 1.7d-5 * (z_real)**(2.d0/3.d0) * (1.d0-beta_sq)/beta_sq * (1.13d0+3.76d0*alpha_sq/beta_sq)	! unmodified Moliere's screening

		! I. Plante and F. A. Cucinotta, 
		! Cross sections for the interactions of 1 eV-100 MeV electrons in liquid water and application to Monte-Carlo simulation of HZE radiation tracks
		! New Journal of Physics 11.6, 063047 (2009)
		! http://dx.doi.org/10.1088/1367-2630/11/6/063047
		! Formula (41) (we use a different screening parameter eta_prime)
		Elastic_cross_section = Pi*z_real*(z_real+1.d0)*re**2 *(1.d0-beta_sq)/ ( eta_prime*(eta_prime+1.d0) * beta_sq**2 )  * 1.d20	! m^2 -> A^2

       else
		Elastic_cross_section=0.d0
       endif
       
      !~ return

end function Elastic_cross_section


pure function Old_Elastic_cross_section(nrg_ev,Z)
!  Elastic electron-atom scattering
! [Frignani,Grasso]

    real(8) Old_Elastic_cross_section	! [A^2]

    real(8), intent(in) :: nrg_ev  ! kinetic energy of the electron in eV
    integer, intent(in) :: Z    !  atomic number
    REAL(8) nc, nc2, nc3, betta, a0, Ry, me, el, ee,  cvel, PPi

    !~ vel = vel_SI*1.d5		! m/s -> A/fs; internal units of this function
    !~ ee   = vel*vel/36.d0	!  [eV]
    ee   = nrg_ev


! CORRECTION
       if (ee.gt.0.d0) then

       me = 9.1093821545d-31
       cvel = 299792458.0d0
       el = 1.602176487d-19
       Ry = 13.606d0
       PPi = 3.14159265d0
       a0 = 0.53d0

       betta = dsqrt(ee/(ee+me*cvel*cvel/el))
       nc = 1.7d0*(1d-5)*(dble(Z)**(2.0d0/3.0d0))
       nc2 = nc*(1.0d0-2.0d0*ee/(me*cvel*cvel/el))
       nc = nc2/(2.0d0*ee/(me*cvel*cvel/el))
   ! screening parameter
   ! book: Akkerman A.F. Modelirovanie traektoriy zaryazhennyh chastits v veschestve. M.: Energoatomizdat, 1991. ISBN 5-283-02924-7. Formula 1.17.
   ! book: Glauber R.J. Lectures in theoretical physics. N.Y.: Interscience, 1959. Vol 1. P. 315-361
       nc3=nc*(1.13d0+3.76d0*(dble(Z)/137.0d0)**2/(2.0d0*ee/(me*cvel**2/el))*betta)
      
       Old_Elastic_cross_section = PPi*a0*a0*dble(Z)*(dble(Z)+1.0d0)/(nc3*(nc3+1.0d0))*(Ry/ee)**2

       else
      
       Old_Elastic_cross_section=0.d0
      
       endif
      return

end function Old_Elastic_cross_section


! the function from Nikita Medvedev - just for testing
pure function Atomic_elastic_sigma(Z, Ee) ! total cross-section of elastic scattering on an atom:
! New J. Phys. 11, 063047 (2009), formula (41)
   REAL(8) :: Atomic_elastic_sigma   ! cross-section of elastic scteering [cm^2]
   real(8), intent(in) :: Z ! atomic number
   real(8), intent(in) :: Ee    ! [eV] incident electron energy
   real(8) nc, pc, mec2e, Zat137, RyEe
   
   real(8) :: Zat ! atomic number
   real(8) :: nat ! atomic density of the target [1/cm^3]
   
real(8) g_Pi, g_e, g_me, g_lc, g_clight, g_h, g_Ry, g_a0   ! universal constants 
!UUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUU
! Universal constants:
parameter (g_Pi = 3.1415926535897932384626433832795d0)            ! Pi
parameter (g_e = 1.602176487d-19)         ! Electron charge       [Coulomb]
parameter (g_me = 9.1093821545d-31)       ! Electron mass         [kg]
parameter (g_clight = 299792458.0d0)        ! Light velosity        [m/sec]
parameter (g_lc = 2.4263102389d-2)        ! Compton wavelength [A]
parameter (g_h = 1.05457162853d-34)       ! Plank constant        [J*sec]
parameter (g_Ry = 13.606d0)               ! Rydberg constant      [eV]
parameter (g_a0 = 0.5291772085936d0)      ! Bohr radius           [A]
   
   Zat = Z ! atom atomic number, point onto it to use below
   mec2e = g_me*g_clight*g_clight/g_e   ! a parameter enterring eq. below:
   Zat137 = Zat/137.0d0   ! a parameter enterring eq. below:
   RyEe = g_Ry/Ee   ! a parameter enterring eq. below:

   pc = 1.7d-5*(Zat**(2.0d0/3.0d0))*(1.0d0-2.0d0*Ee/mec2e)/(2.0d0*Ee/mec2e)
   nc = pc*(            1.13d0 + 3.76d0*(Zat137*Zat137)/(2.0d0*Ee/mec2e)*dsqrt(Ee/(Ee+mec2e))                       )
   Atomic_elastic_sigma = g_Pi*g_a0*g_a0*Zat*(Zat+1.0d0)/(nc*(nc+1.0d0))*RyEe*RyEe*1d-16  ! Mott cross-section [cm^2]

end function Atomic_elastic_sigma



! the function for Beata - just for testing
pure function sigmae(Zat, vel)
! [Plante and Cucinotta; Jenkins et al.]

       ! elastic cross section of electrons on atom with number Z [A^2]
       real(8) sigmae

       ! atomic number
       real(8), intent(in) :: Zat
       ! velocity of the electron [A/fs]
       real(8), intent(in) :: vel 
	
       REAL(8) Pi, clight, alpha, me, el, re
       real(8) Ekin

       real(8) tau, eta_prime, beta_sq

       ! Pi
       Pi = 3.1415926535897932d0
       ! speed of light [m/s]
       clight = 299792458.d0
       ! fine-structure constant  ~1/137
       alpha = 0.0072973525664d0
       ! electron mass [kg]
       me = 9.10938356d-31
       ! electron charge  [C]
       el = 1.6021766208d-19
       ! classical electron radius [m]
       re = 2.8179403267d-15

       ! kinetic energy [J]; non-relativistic approximation
       Ekin   = 0.5d-10*me*vel*vel
       ! relativistic kinetic energy [J]
       ! Ekin   = me*clight*clight/dsqrt(1-vel*vel*1.d10/clight**2)  - me*clight*clight

       if (Ekin.gt.0.d0) then

          ! speed of the electron in units of c, or simply v/c,  squared
          beta_sq = 2.0d0*Ekin/(me*clight**2)
          ! kinetic energy of the electron in units of mc^2
          tau = Ekin/(me*clight**2)

          ! T.M. Jenkins, W.R. Nelson, A. Rindi. Monte Carlo Transport of Electrons and Photons. Springer Science & Business Media, 2012. ISBN: 1461310598, 9781461310594. Formula 7.8.
          ! Empirically modified Moliere's screening:
          eta_prime = 0.25d0*(alpha/(0.885d0))**2 * (1-beta_sq)/beta_sq * (Zat)**(2.d0/3.d0) * ( 1.13d0 + 3.76d0*(alpha*Zat)**2/beta_sq * dsqrt(tau/(tau+1.d0)) )
          ! (see also unmodified Moliere's screening: A.F. Akkerman. Modelirovanie traektoriy zaryazhennyh chastits v veschestve. M.: Energoatomizdat, 1991. ISBN: 5-283-02924-7. Formula 1.17.)
          ! eta_M = 1.7d-5 * (Zat)**(2.d0/3.d0) * (1.d0-beta_sq)/beta_sq * (1.13d0+3.76d0*alpha_sq/beta_sq)

          ! I. Plante and F. A. Cucinotta, 
          ! Cross sections for the interactions of 1 eV-100 MeV electrons in liquid water and application to Monte-Carlo simulation of HZE radiation tracks
          ! New Journal of Physics 11.6, 063047 (2009)
          ! http://dx.doi.org/10.1088/1367-2630/11/6/063047
          ! Formula (41) (we use a different screening parameter eta_prime)
          sigmae = Pi*Zat*(Zat+1.d0)*re**2 *(1.d0-beta_sq) / ( eta_prime*(eta_prime+1.d0) * beta_sq**2 ) * 1.d20	! m^2 -> A^2
		
       else
          sigmae=0.d0
       endif
       
       return
       end function sigmae


END MODULE Cross_sections
