!    XCASCADE software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains subroutines necessary for generating output files
MODULE Creating_output_files
use Constants   ! universal constants 
use Objects_and_types	! object oriented stuff
implicit none

 contains

!CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
subroutine make_output_folder(Output_file_name, INPUT_FILE_NAME, Fluence, Eph, Total_t, NMC, KOP)
character(*), intent(in) :: Output_file_name, INPUT_FILE_NAME ! part of the filename for output; input file name
real(8), intent(in) :: Fluence ! fluence [J/cm^2]
real(8), intent(in) :: Eph     ! photon/electron energy [eV]
real(8), intent(in) :: Total_t ! total time of simulation [fs]
integer, intent(in) :: NMC ! number of MC iterations
integer, intent(in) :: KOP ! electron vs photon impact
!---------------------------------------------------
character(1) :: path_sep
character(200) :: command, Folder, Folder2, File_name, part_name, ch_temp, ch1
logical :: file_exist
integer :: i

part_name = ''
! Files where we save the outputs:
if (Fluence > 0.0d0) then
   write(ch_temp, '(f6.3)') Fluence
   write(part_name, '(a)') 'F_'//trim(adjustl(ch_temp))
else
   write(part_name, '(a)') 'single_cascade'
endif

select case (KOP)
case (:1, 3:) ! photon impact
   write(part_name, '(a)') trim(adjustl(part_name))//'_photon'
case (2)    ! electron impact
   write(part_name, '(a)') trim(adjustl(part_name))//'_electron'
endselect

write(ch_temp, '(f7.1)') Eph
write(part_name, '(a)') trim(adjustl(part_name))//'_hw_'//trim(adjustl(ch_temp))
write(ch_temp, '(f7.1)') Total_t
write(part_name, '(a)') trim(adjustl(part_name))//'_t_'//trim(adjustl(ch_temp))
write(ch_temp, '(i6)') NMC
write(part_name, '(a)') trim(adjustl(part_name))//'_MC_'//trim(adjustl(ch_temp))

! Directory name:
Folder = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_'//trim(adjustl(part_name))
Folder2 = Folder

! Create the directory for output files:
i = 0
!~ inquire(DIRECTORY=trim(adjustl(Folder)),exist=file_exist)    ! check if input file excists; doesn't work with gfortran :(
do while (file_exist)
   i = i + 1
   write(ch1,'(i6)') i
   write(Folder2,'(a,a,a)') trim(adjustl(Folder)), '_v', trim(adjustl(ch1))
   !~ inquire(DIRECTORY=trim(adjustl(Folder2)),exist=file_exist)    ! check if input file excists; doesn't work with gfortran :(
enddo
command='mkdir '//trim(adjustl(Folder2)) ! to create a folder use this command
CALL system(command)  ! create the folder

! Find out which OS it is:
call Path_separator(path_sep) ! see below

! Copy files into this folder:
!File_name = 'INPUT_material.txt'
File_name = trim(adjustl(INPUT_FILE_NAME))
call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
 
File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Atomic_parameters.dat'
call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
call delete_file(File_name)

File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Densities.dat'
call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
call delete_file(File_name)

File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Density_derivative.dat'
call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
call delete_file(File_name)

File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_electron_distribution.dat'
inquire(file=trim(adjustl(File_name)),exist=file_exist)
if (file_exist) call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
call delete_file(File_name)

File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Energies.dat'
call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
call delete_file(File_name)

File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Gnuplot_eps.sh'
call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
call delete_file(File_name)

File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Gnuplot_x11.sh'
call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
call delete_file(File_name)

File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Densities.eps'
call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
call delete_file(File_name)

File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Energies.eps'
call copy_file(trim(adjustl(File_name)), trim(adjustl(Folder2)), path_sep)
call delete_file(File_name)

end subroutine make_output_folder 


subroutine delete_file(File_name)
character(*), intent(in) :: File_name ! file to be deleted
logical :: file_exist, file_opened
integer :: stat
inquire(file=trim(adjustl(File_name)),exist=file_exist,opened=file_opened)
if (file_exist) then
   if (.not.file_opened) then
      open(unit=1234, iostat=stat, file=trim(adjustl(File_name)), status='old')
   endif
   ! now file is opened, so close and delete it:
   if (stat == 0) close(1234, status='delete')
endif
end subroutine delete_file


subroutine delete_old_cascades()
integer fn, fnm
logical file_exist
character(100) file_name,file_number

! removing existing old files with cascades
    fn = 0
    file_exist = .true.
    
    do while (file_exist)
        fn = fn+1

        write (file_number, "(I6)") fn
        File_name = 'cascades/OUTPUT_cascade'//trim(adjustl(file_number))//'.dat'
	
        inquire(file=trim(adjustl(File_name)),exist=file_exist) ! check if the file is there

        if (file_exist) then
		call delete_file(File_name)
        endif

    enddo

end subroutine delete_old_cascades


subroutine copy_file(file_to_copy, folder_copy_to, path_sep)
character(len=*), intent(in) :: file_to_copy, folder_copy_to
character(*), intent(in) :: path_sep ! allows to define whether it is windows or linux
character(200) command
select case (path_sep)
case ('/')  ! Lunux or unix based OS
   command='scp '//trim(adjustl(file_to_copy))//' '//trim(adjustl(folder_copy_to))   
case default        ! windows
   command='xcopy '//trim(adjustl(file_to_copy))//' '//trim(adjustl(folder_copy_to))
end select
CALL system(command)
end subroutine copy_file


subroutine Path_separator(path_sep)
CHARACTER(len=1), intent(out) :: path_sep
CHARACTER(len = 100) :: path
CALL get_environment_variable("PATH",path)
if (path(1:1) .EQ. '/') then        !unix based OS
    path_sep = '/'
else if (path(3:3) .EQ. '\') then   !Windows OS
    path_sep = '\'
else
    path_sep = ''
    print*, 'Path separator is not defined'    !Unknown OS
endif 
end subroutine Path_separator

subroutine inout_xcascade3d(fluence, mheff, tfp_h, min_Ip_elements)
real(8), intent(in) :: fluence
real(8), intent(in) :: mheff	! effective mass of a valence hole
real(8), intent(in) :: tfp_h	! time of mean free path for holes
real(8), intent(in) :: min_Ip_elements	! minimal ionization potential

   open(201, FILE = 'INOUT_xcascade3d.iout', action="write", status="replace")
	write(201,*) fluence, mheff, tfp_h, min_Ip_elements
   close (unit=201)

end subroutine inout_xcascade3d

!OOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOOO
!Output files:
subroutine Printing_output_MC_densities(Output_file_name, NMC, Ne_max, NoEiC, N_shells, Fluence, N_abs, Eph, Ip_elements, El_distr, Hole_distr, print_distr)
   character(100), intent(in) :: Output_file_name ! part of the filename for output
   integer, intent(in) :: NMC, NoEiC, N_shells, Ne_max	! dimensions
   real(8), intent(in) :: Fluence, N_abs	! fluence [J/cm^2] and [1/cm^3] density of absorbed photons
   real(8), dimension(:,:), intent(in) :: Ip_elements	! ionization potentials [eV]
   class(Electron_density), intent(inout) :: El_distr ! electron density in time
   class(Hole_density), intent(inout) :: Hole_distr   ! holes densities in time
   real(8), intent(in) :: Eph
   logical, intent(in) :: print_distr

   character(100) File_name, File_name2, File_name3
   integer FN, FN2, FN3, i, j, k, l
   real(8) FWHM
   real(8), dimension(:), allocatable :: Deriv_dens

   ! Calculate derivative of the electron density:
   call Derivative_of_density(El_distr, Deriv_dens, FWHM)
   if (MAXVAL(Deriv_dens).gt.0.d0) Deriv_dens = Deriv_dens/(MAXVAL(Deriv_dens))

   FN = 102
   FN2 = 103
   FN3 = 104
!    print*, 'starting Printing_output_MC_densities'
   !FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
   ! Files where we save the outputs:
   File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Densities.dat'
   open(FN, FILE = trim(adjustl(File_name)), action="write", status="replace")
   File_name2 = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Density_derivative.dat'
   open(FN2, FILE = trim(adjustl(File_name2)), action="write", status="replace")
   if (print_distr) then
      File_name3 = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_electron_distribution.dat'
!       print*, 'SAVING: ', trim(adjustl(File_name3))
      open(FN3, FILE = trim(adjustl(File_name3)), action="write", status="replace")
!       print*, 'OPENED: ', trim(adjustl(File_name3))
   endif
   !FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
   write(FN, '(a,i9,a)') '# The output of the ', NMC, ' MC iterations'
   write(FN, '(a,f7.1,a)') '#for the ', Eph, ' [eV] photon energy,'
   write(FN, '(a,es13.5,a,es13.5)') '# the fluence ', Fluence, ' [J/cm^2]. Absorbed photons ', N_abs
   write(FN, '(a)', advance='NO') '# Time [fs],	Electron density [1/cm^3], Holes densities [1/cm^3] in different shells:'
   do k = 1, NoEiC
      do j = 1, N_shells
         if ((j .EQ. N_shells) .AND. (k .EQ. NoEiC)) then
            if (Ip_elements(k,j) .GT. 0.0d0) then
               write(FN, '(f25.16)') Ip_elements(k,j)
            else
               write(FN, '(a)') ''
            endif
         else
            if (Ip_elements(k,j) .GT. 0.0d0) then
               write(FN, '(f25.16)', advance='NO') Ip_elements(k,j)
            endif
         endif
      enddo ! j
   enddo ! i
   write(FN2, '(a,i9,a)') '# The output of the ', NMC, ' MC iterations'
   write(FN2, '(a,f7.1,a)') '#for the ', Eph, ' [eV] photon energy,'
   write(FN2, '(a,es13.5,a,es13.5)') '# the fluence ', Fluence, ' [J/cm^2]. Absorbed photons ', N_abs
   write(FN2, '(a,es13.5)') '# Time [fs],	Electron density [1/cm^3], 	Density derivative [1/cm^3/fs],	FWHM=', FWHM

   if (print_distr) then
!       print*, size(El_distr%ne), size(El_distr%Ee_vs_t,1), size(El_distr%Ee_vs_t,2), size(El_distr%Ee_vs_t_grid)
      do j = 1, size(El_distr%Ee_vs_t_grid) ! all grid-points:
         write(FN3, '(f25.16)', advance='NO') El_distr%Ee_vs_t_grid(j)
         do i = 1, Ne_max
            write(FN3, '(es25.16)', advance='NO') El_distr%Ee_vs_t(i,j)
         enddo
         write(FN3, '(a)') ''
      enddo
   endif

   do i = 1, Ne_max
      write(FN, '(f25.16,es25.16)', advance='NO') El_distr%t(i), El_distr%ne(i)
      write(FN2, '(f25.16,es25.16,es25.16)') El_distr%t(i), El_distr%ne(i), Deriv_dens(i)
      do k = 1, NoEiC
         do j = 1, N_shells
            l = (k-1)*N_shells + j
            if ((j .EQ. N_shells) .AND. (k .EQ. NoEiC)) then
               if (Ip_elements(k,j) .GT. 0.0d0) then
                  write(FN, '(es25.16)') Hole_distr%nh(l,i)
!                   write(*, '(i12,i12,i12,f25.16)') k,j,l, Hole_distr%nh(l,i)
               else
                  write(FN, '(a)') ''
!                   write(*, '(a)') ''
               endif
            else
               if (Ip_elements(k,j) .GT. 0.0d0) then
                  write(FN, '(es25.16)', advance='NO') Hole_distr%nh(l,i)
!                   write(*, '(i12,i12,i12,f25.16)') k,j,l, Hole_distr%nh(l,i)
               endif
            endif
         enddo ! j
      enddo ! k
   enddo ! i
   print*, 'The calculated densities are saved in file: ', trim(adjustl(File_name))
   print*, 'The density derivatives are saved in file: ', trim(adjustl(File_name2))
   if (print_distr) then
      print*, 'The electron distribution is saved in file: ', trim(adjustl(File_name3))
   endif

   print*, ''
   write(*,'(a,1x,f12.4,1x,a)') ' FWHM of the time derivative of electron density is: ', FWHM, '[fs]'

   close(FN)
   close(FN2)
   close(FN3)
!    stop 'stop'
end subroutine Printing_output_MC_densities


subroutine Derivative_of_density(El_distr, Deriv_dens, FWHM)
   class(Electron_density), intent(in) :: El_distr ! electron density in time
   real(8), dimension(:), allocatable, intent(inout) :: Deriv_dens
   real(8), intent(out) :: FWHM	! full width at half maximum of the density derivative
   integer i, N, k(1), i_max, i_left, i_right
   N = size(El_distr%ne)
   allocate(Deriv_dens(N))
   Deriv_dens(1) = 0.0d0
   do i = 2, N-1
      Deriv_dens(i) = (El_distr%ne(i)-El_distr%ne(i-1))/(El_distr%t(i)-El_distr%t(i-1))      
!       print*, i, Deriv_dens(i)
   enddo
   Deriv_dens(N) = Deriv_dens(N-1) 
   k = MAXLOC(Deriv_dens)
   i_max = k(1)
!    print*, 'MAX:', i_max, Deriv_dens(i_max)
   call Find_value(Deriv_dens,Deriv_dens(i_max)*0.5d0,1,i_max,i_left)
   call Find_value(Deriv_dens,Deriv_dens(i_max)*0.5d0,i_max,N,i_right)
   FWHM = El_distr%t(i_right) - El_distr%t(i_left)
!    print*, 'FWHM:', i_left, i_right, FWHM
end subroutine Derivative_of_density

subroutine Find_value(array,valu,lb,ub,ind)
   real(8), dimension(:), intent(in) :: array ! array to look into
   real(8), intent(in) :: valu ! value to find the closest from the arrray
   integer, intent(in), optional :: lb, ub ! lower boundary, upper boundary
   integer, intent(out) :: ind  ! index of the value in the array that is the closest to "valu"
   integer ia, ib
   real(8) a,b,cur
   if (present(lb)) then
      ia = lb
   else
      ia = 1
   endif
   !a = array(ia)
   if (present(ub)) then
      ib = ub
   else
      ib = size(array)
   endif
   !b = array(ib)
   ind = FLOOR((ia+ib)*0.5d0)
   cur = array(ind)
!    print*, 'START:', cur, valu
   do while (abs(ia-ib).gt.1)
      ind = FLOOR((ia+ib)*0.5d0)
      cur = array(ind)
      if (array(ia) .LE. array(ib)) then	! increasing array
         if (cur .LT. valu) then
            ia = FLOOR((ia+ib)*0.5d0)
         else
            ib = FLOOR((ia+ib)*0.5d0)
         endif
      else	! decreasing array
         if (cur .GT. valu) then
            ia = FLOOR((ia+ib)*0.5d0)
         else
            ib = FLOOR((ia+ib)*0.5d0)
         endif
      endif
!       write(*,'(es,es,i,i,i)') cur,abs(valu-cur), ia, ib, ind
      if (abs(ia - ib) .LE. 1) exit
   enddo
   !ind = ia
!    stop 'Find_value is done'
end subroutine Find_value


subroutine Printing_output_MC_energies(Output_file_name, NMC, Ne_max, NoEiC, N_shells, Fluence, N_abs, Eph, Ip_elements, El_distr, Hole_distr)
   character(100), intent(in) :: Output_file_name ! part of the filename for output
   integer, intent(in) :: NMC, NoEiC, N_shells, Ne_max	! dimensions
   real(8), intent(in) :: Fluence, N_abs	! fluence [J/cm^2] and [1/cm^3] density of absorbed photons
   real(8), dimension(:,:), intent(in) :: Ip_elements	! ionization potentials [eV]
   class(Electron_density), intent(inout) :: El_distr ! electron density in time
   class(Hole_density), intent(inout) :: Hole_distr   ! holes densities in time
   real(8), intent(in) :: Eph

   character(100) File_name
   integer FN, i, j, k, l
   real(8) Tot_E

   FN = 102
   !FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
   ! Files where we save the outputs:
   File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Energies.dat'
   open(FN, FILE = trim(adjustl(File_name)), action="write", status="replace")
   !FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
   write(FN, '(a,i9,a)') '# The output of the ', NMC, ' MC iterations'
   write(FN, '(a,f7.1,a)') '# for the ', Eph, ' [eV] photon energy,'
   write(FN, '(a,e13.5,a,e13.5)') '# the fluence ', Fluence, ' [J/cm^2]. Absorbed photons ', N_abs
   write(FN, '(a)', advance='NO') '# Time [fs],	Electrons energy density [eV/cm^3], Holes energy densities [eV/cm^3] in different shells:'

   do k = 1, NoEiC
      do j = 1, N_shells
         if ((j .EQ. N_shells) .AND. (k .EQ. NoEiC)) then
            if (Ip_elements(k,j) .GT. 0.0d0) then
               !write(FN, '(f25.16)') Ip_elements(k,j)
               write(FN, '(f25.16)', advance='NO') Ip_elements(k,j)
            else
               !write(FN, '(a)') ''
            endif
         else
            if (Ip_elements(k,j) .GT. 0.0d0) then
               write(FN, '(f25.16)', advance='NO') Ip_elements(k,j)
            endif
         endif
      enddo ! j
   enddo ! i
   write(FN, '(a)') ',	Total energy'

   do i = 1, Ne_max
      Tot_E = 0.0d0
      write(FN, '(f25.16,e25.16)', advance='NO') El_distr%t(i), El_distr%Ee(i)
      do k = 1, NoEiC
         do j = 1, N_shells
            l = (k-1)*N_shells + j
            if ((j .EQ. N_shells) .AND. (k .EQ. NoEiC)) then
               if (Ip_elements(k,j) .GT. 0.0d0) then
                  !write(FN, '(e25.16)') Hole_distr%nh(l,i)*Ip_elements(k,j)
                  write(FN, '(e25.16)', advance='NO') Hole_distr%nh(l,i)*Ip_elements(k,j)
               else
                  !write(FN, '(a)') ''
               endif
            else
               if (Ip_elements(k,j) .GT. 0.0d0) then
                  write(FN, '(e25.16)', advance='NO') Hole_distr%nh(l,i)*Ip_elements(k,j)
               endif
            endif
            Tot_E = Tot_E + (Hole_distr%nh(l,i)*Ip_elements(k,j))
         enddo ! j
      enddo ! k
      write(FN, '(e25.16)') El_distr%Ee(i) + Tot_E 
   enddo ! i
   print*, 'The calculated energies are saved in file: ', trim(adjustl(File_name))
   close(FN)
end subroutine Printing_output_MC_energies


subroutine Gnuplot_plotting(KOF, Output_file_name, Eph, Fluence, Max_pulse, FWHM, Total_t, Global_dt, PQN_elements, Ip_elements, N_shells, NoEiC, Element, Shell_elements, N_abs)
   integer, intent (in) :: KOF	! kind of plot
   character(100), intent(in) :: Output_file_name ! part of the filename for output
   real(8), intent(in) :: Eph, Fluence, Max_pulse, FWHM, Total_t, Global_dt
   integer, DIMENSION(:,:), intent(in) :: PQN_elements 	! principal quantum numbers 4 all shells, all elements
   real(8), dimension(:,:), intent(in) :: Ip_elements	! ionization potentials [eV]
   integer, intent(in) :: NoEiC, N_shells	! dimensions
   type(Atom), DIMENSION(:), INTENT(in) :: Element	! data about all elements
   character(11), DIMENSION(:,:), ALLOCATABLE, intent(inout) :: Shell_elements 	! names of all shells, all elements
   real(8), intent(in), optional :: N_abs	! number of absorbed photons/electrons
!    class(Electron_density), intent(inout) :: El_distr ! electron density in time
!    class(Hole_density), intent(inout) :: Hole_distr   ! holes densities in time

   character(100) :: File_name, File_name_dens, File_name_NRGs
   integer FN, FN2, FN3, i, k, j
   real(8) RN

   FN = 200
   !FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
   ! Files where we save the outputs:
   select case (KOF)
      case (:1)
      File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Gnuplot_eps.sh'
      case (2:)
      File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Gnuplot_x11.sh'
   endselect
   open(FN, FILE = trim(adjustl(File_name)), action="write", status="replace")
   File_name_dens = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Densities.dat'
   File_name_NRGs = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_Energies.dat'
   !FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
   select case (KOF)
      case (:1)
      write(FN, '(a)') '#!/bin/bash'
      write(FN, '(a)') ''
      write(FN, '(a)') 'NAME='//'OUTPUT_'//trim(adjustl(Output_file_name))//'_Densities.eps'
      write(FN, '(a)') 'NAME2='//'OUTPUT_'//trim(adjustl(Output_file_name))//'_Energies.eps'
      write(FN, '(a)') 'LW=5'
      case(2:)
      write(FN, '(a)') 'LW=2'
   endselect
   write(FN, '(a,f8.2, a, f8.2, a, f8.2, a)') 'LABL="', Eph, ' eV', Fluence ,' J/cm2', FWHM, 'fs " '
   write(FN, '(a)') 'TICSIZ=50'
   write(FN, '(a)') 'echo " '
   select case (KOF)
      case (:1)
      write(FN, '(a)') 'set terminal postscript enhanced \"Helvetica\" 16 color '
      write(FN, '(a)') 'set output \"$NAME\"'
      case (2:)
      write(FN, '(a)') 'set terminal x11 persist'
      write(FN, '(a)') 'set multiplot'
      write(FN, '(a)') 'set size 0.5,1'
      write(FN, '(a)') 'set origin 0,0'
      write(FN, '(a)') 'unset label'
   endselect
   if (present(N_abs)) then
      write(FN, '(a)') 'set ylabel \" Electrons per cascade \"      font \"Helvetica,20\" '
   else
      write(FN, '(a)') 'set ylabel \" Density, (1/cm^3) \"      font \"Helvetica,20\" '
   endif
   write(FN, '(a)') 'set xlabel \" Time, (fs) \"        font \"Helvetica,20\" '
   !write(FN, '(a)') 'set label \"$LABL\" at 150,-8 font \"Helvetica,22\" '
   write(FN, '(a)') 'set key '
   write(FN, '(a)') 'set xtics \"$TICSIZ\" '
   write(FN, '(a,f8.2,a,f10.2,a,a,a)') 'p[', Max_pulse-3.0d0*FWHM, ':', Total_t, '] \"', trim(adjustl(File_name_dens)), '\" u 1:2 w l lw \"$LW\" title \"Electron density\" ,\'
   i = 0
   do k = 1, NoEiC
      do j = 1, N_shells
         if (Ip_elements(k,j) .GT. 0.0d0) i = i + 1
         !if ((k .GT. 1) .AND. (Ip_elements(k,j+1) .LE. 0.0d0)) then
         if ( (k .EQ. NoEiC)) then
             !write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_dens)), '\" u 1:', 2+j+(k-1)*N_shells, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)), ' holes\"'
             if (j .EQ. N_shells) then
                write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_dens)), '\" u 1:', 2+i, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)), ' holes\"'
             else if ((Ip_elements(k,j+1) .LE. 0.0d0) ) then
                write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_dens)), '\" u 1:', 2+i, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)), ' holes\"'
                if (Ip_elements(k,j+1) .LE. 0.0d0) exit ! no more shells for this element
             else
                write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_dens)), '\" u 1:', 2+i, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)), ' holes\" ,\'
             endif
         else
            !write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_dens)), '\" u 1:', 2+j+(k-1)*N_shells, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)), ' holes\" ,\'
            write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_dens)), '\" u 1:', 2+i, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)), ' holes\" ,\'
         endif
         !if (Ip_elements(k,j) .LE. 0.0d0) exit ! no more shells for this element
         !print*, 'k', k, j, i, trim(adjustl(Shell_elements(k,j)))
      enddo ! j
   enddo ! i

   write(FN, '(a)') ''
   select case (KOF)
      case (:1)
      write(FN, '(a)') 'set output \"$NAME2\"'
      case (2:)
         write(FN, '(a)') 'set size 0.5,1'
         write(FN, '(a)') 'set origin 0.5,0'
         write(FN, '(a)') 'unset label'
   endselect
   if (present(N_abs)) then
      write(FN, '(a)') 'set ylabel \" Energy, (eV) \"      font \"Helvetica,20\" '
   else
      write(FN, '(a)') 'set ylabel \" Energy density, (eV/cm^3) \"      font \"Helvetica,20\" '
   endif
   write(FN, '(a)') 'set xlabel \" Time, (fs) \"        font \"Helvetica,20\" '
!    write(FN, '(a)') 'set label \"$LABL\"  at 150,5000 font \"Helvetica,22\" '
   write(FN, '(a)') 'set key right top '
   write(FN, '(a,f8.2,a,f10.2,a,a,a)') 'p[', Max_pulse-3.0d0*FWHM, ':', Total_t, '] \"', trim(adjustl(File_name_NRGs)), '\" u 1:2 w l lw \"$LW\" title \"Electrons\" ,\'
   i = 0
   do k = 1, NoEiC
      do j = 1, N_shells
         if (Ip_elements(k,j) .GT. 0.0d0) i = i + 1
         !if (Ip_elements(k,j+1) .LE. 0.0d0) then
         if ( (k .EQ. NoEiC) ) then
            if (j .EQ. N_shells) then
               write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_NRGs)), '\" u 1:',2+i, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)),' holes\",\'
            else if (Ip_elements(k,j+1) .LE. 0.0d0) then
               !write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_NRGs)), '\" u 1:',2+i, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)),' holes\" '
               write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_NRGs)), '\" u 1:',2+i, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)),' holes\",\'
               if (Ip_elements(k,j+1) .LE. 0.0d0) exit ! no more shells for this element
            else
               write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_NRGs)), '\" u 1:',2+i, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)),' holes\" ,\'
            endif
         else
            write(FN, '(a,a,a,i2,a,a,a)') '\"', trim(adjustl(File_name_NRGs)), '\" u 1:',2+i, ' w l lw \"$LW\" title \"', trim(adjustl(Shell_elements(k,j)))//' of '//trim(adjustl(Element(k)%Name)),' holes\" ,\'
         endif
      enddo ! j
   enddo ! i
   i = i+1
   write(FN, '(a,a,a,i2,a)') '\"', trim(adjustl(File_name_NRGs)), '\" u 1:',2+i, ' w l lw \"$LW\" title \"Total energy\" '


   select case (KOF)
      case (:1)
         write(FN, '(a)') 'reset'
         write(FN, '(a)') '" | gnuplot '
         close (FN)
         print*, 'Gnuplot script for eps-files is prepared and executed: ', trim(adjustl(File_name))
         call system('chmod +x '//trim(adjustl(File_name))) ! make the output-script executable
         call system(trim(adjustl('./' // File_name))) ! execute the prepared script
         print*, 'Plots are saved under names: '
         write(*, '(a)') '1) OUTPUT_'//trim(adjustl(Output_file_name))//'_Densities.eps'
         write(*, '(a)') '2) OUTPUT_'//trim(adjustl(Output_file_name))//'_Energies.eps'
      case (2:)
         write(FN, '(a)') 'unset multiplot'
         write(FN, '(a)') 'reset'
         write(FN, '(a)') '" | gnuplot '
         close (FN)
         print*, 'Gnuplot script for plotting is prepared: ', trim(adjustl(File_name))
         call system('chmod +x '//trim(adjustl(File_name))) ! make the output-script executable
         !~ call system(trim(adjustl('./' // File_name))) ! execute the prepared script
   endselect
  close(FN)
end subroutine Gnuplot_plotting


end MODULE Creating_output_files