!    XCASCADE software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This is the main program file

!#define OMP_inside
PROGRAM Cascading_code
!MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM
! Initiate modules:
! module-name:             | file where it's storred:
use Constants              ! Constants.f90
use Cross_sections         ! Cross-sections.f90
use Files_dealing	   		! Dealing_with_input_files.f90
use Creating_output_files   ! Creating_output_files.f90
use Objects_and_types	   ! Objects_subroutine.f90
use Laser_pulse_parameters ! Laser_pulse.f90
use Monte_Carlo		   		! Monte_Carlo.f90
implicit none
!MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM
! Type variables:
type(Error_handling) Error_occured	! error messages
type(Electron_density) El_distr		! electron density in time
type(Hole_density) Hole_distr		! holes density in time
!type(Photon), DIMENSION(:), ALLOCATABLE :: Ph_array 	! photons as objects in array
! type(Electron) El	! electrons as objects
! type(Hole) Hol		! holes as objects

! Variables:
integer, DIMENSION(:), ALLOCATABLE :: Z_elements	! atomic numbers of elements
REAL(8), DIMENSION(:), ALLOCATABLE :: M_elements	! masses of all elements in molecule
integer, DIMENSION(:), ALLOCATABLE :: N_elements	! numbers of elements in 1 molecule
real(8), DIMENSION(:,:), ALLOCATABLE :: Ip_elements ! [eV] ionization potentials of all shells, all elements
logical, DIMENSION(:,:), ALLOCATABLE :: whether_valence ! [eV] ionization potentials of all shells, all elements
real(8), DIMENSION(:,:), ALLOCATABLE :: Ekin_elements ! [eV] kinetic energies of all shells, all elements
real(8), DIMENSION(:,:), ALLOCATABLE :: Nel_elements 	! number of electrons in all shells, all elements
real(8), DIMENSION(:), ALLOCATABLE :: NelVB_elements	! numbers of VALENCE electrons in shells
integer, DIMENSION(:,:), ALLOCATABLE :: PQN_elements 	! principal quantum numbers 4 all shells, all elements
character(11), DIMENSION(:,:), ALLOCATABLE :: Shell_elements 	! names of all shells, all elements
real(8), DIMENSION(:,:), ALLOCATABLE :: t_Auger	! Auger-decays times [fs]
real(8), DIMENSION(:,:), ALLOCATABLE :: Photoabs_sigma ! [A^2] photoionization cross-sections for all shells
type(Atom), DIMENSION(:), ALLOCATABLE :: Element

real(8), dimension(:), allocatable :: Temp_El_distr	! electron distribution to work with openmp
real(8), dimension(:), allocatable :: Temp_El_NRG_distr	! electron distribution to work with openmp
real(8), dimension(:,:), allocatable :: Temp_El_distr_E ! electron deistribution in energy space to work with openmp
real(8), dimension(:), allocatable :: Temp_El_distr_E_grid ! grid for electron deistribution in energy space to work with openmp
real(8), dimension(:,:), allocatable :: Temp_Hol_distr	! holes distributions to work with openmp

type(CDF), dimension(:,:), allocatable :: Ritchi    ! Ritchi CDF to use for cross-sections
type(MFP_from_file), dimension(:,:), allocatable :: IMFPs	! inelastic mean free paths from files

real(8) Effgap	! [eV] effective material bandgap
real(8) Egap	! [eV] real material bandgap
real(8) mheff	! [m_e] mass of valence holes
real(8) velh	! velocity of a hole
real(8) tfp_h	! time of free flight of a valence hole in collisions with atoms
real(8) sigma_vhole	! cross section of a valence hole scattering on atoms
real(8) Ecut 	! [eV] only print out cascades with electron energy larger than Ecut
real(8) Eph	! [eV] photon energy
real(8) Fluence	! fluence of the pulse [J/cm^2]
integer koe	! Kind of event that's happening: photon absorption, electron collision (elastic or inelastic), or deep hole motion
integer KOPulse	! shape of pulse: 0=flat-top, 1=Gaussian
real(8) Pulse_max ! position of the maximum of the mulse (mu-gauss)
real(8) Pulse_duration ! FWHM of the mulse (sigma-gauss)
real(8) Total_t		! Time of analysis [fs]
real(8)	Global_dt	! Timestep [fs], with which you will to have the time resolution
real(8)	IMFP, EMFP	! inelastic/elastic mean free path [A]
real(8) TIM	! electron free flight time [fs]
real(8) dEdx, dEdx0	! electron energy loss [eV/A]
real(8) e_range	! electron range [A]
real(8)	Ph_MFP	! photon attenuation length [cm]
real(8) N_abs	! density of absorbed photons according to fluence [1/cm^3]
real(8) as1, Dens, E, E0, Sigma, Sigma_tot, At_dens, Photoabs_sigma_tot, t, dE
logical incl_elastic ! include the effect of elastic collisions? otherwise only inelastic are included
logical print_distr ! print out the transient electron distribution?
logical print_casc ! print out the cascades?
integer ctim(8), c1(8), NMC, i, NoEiC, j, k, l, stat, Ne_stat, Ne_max, N_shells, N_tim 
integer, DIMENSION(:,:), allocatable :: CSN	! kinds of cross-sections
real(8) :: min_Ip_elements

logical want_to_do, file_exists
character(100) Output_file_name, File_name, File_name2, File_name3, Writing_var, chtest, chtest2, INPUT_FILE_NAME
integer :: which_input

! For OPENMP:
integer Num_th, my_id, OMP_GET_THREAD_NUM, OMP_GET_NUM_THREADS
!MMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMMM

which_input = 0 ! start counting how many input-data variations we have made
INPUT_FILE_NAME = 'INPUT_material.txt' ! first input file

want_to_do = .true. ! by default we want to do anything!..
Error_occured%Err = .false. ! no errors at the beginning, turn it into 'true' if occurs
call random_seed() ! standard FORTRAN seeding of random numbers
call date_and_time(values=c1) ! standard FORTRAN time and date
 ctim=c1
write(*, 1005) ctim(5), ctim(6), ctim(7), ctim(3), ctim(2), ctim(1)
print*, '   '

!SSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSSS
!Start the program itself
!~ #ifdef OMP_inside
   !~ print*, 'The code is compiled with openmp'
!~ #else
   !~ print*, 'The code is compiled without openmp'
!~ #endif


!RRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRR
! Reading data from files:
Output_file_name = ''
call READING_INPUT_FILES(NoEiC, Z_elements, M_elements, N_elements, Ip_elements, whether_valence, Ekin_elements, Nel_elements, NelVB_elements, PQN_elements, Shell_elements, t_Auger, Photoabs_sigma, Dens, Effgap, Egap, mheff, Eph, Fluence, koe, KOPulse, Pulse_max, Pulse_duration, Output_file_name, Error_occured, incl_elastic, print_distr, print_casc, Ecut, NMC, Num_th, Total_t, Global_dt, Element, Ritchi, CSN, IMFPs)
At_dens = Dens/(SUM(M_elements(:)*N_elements(:))/(SUM(N_elements))*g_Mp*1.d3)   ! atomic density [1/cm^3]
Photoabs_sigma_tot = SUM(Photoabs_sigma) ! [A^2] total photoabsorbtion cross-section


select case (koe)
	case (:1, 3:) ! photon impact
		Ph_MFP = 1.0d16/(Photoabs_sigma_tot*(At_dens/SUM(N_elements)))	! [cm] photon attenuation length
		if (Fluence .gt. 0.d0) then	! pulse
			N_abs = (Fluence/(g_e*Eph))*(1.0d0 - dexp(-Ph_MFP/Ph_MFP))/(Ph_MFP)	! [1/cm^3] density of absorbed photons
		else	! single photon
			N_abs = 1.d0
		endif
	case (2)	! electron impact
		call El_free_flight_time(CSN, Eph, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, TIM)
		Ph_MFP = 1.d-8*IMFP	! electron IMFP
		if (Fluence .GT. 0.d0) then ! electron bunch:
			N_abs = (Fluence/(g_e*Eph))*(1.0d0 - dexp(-Ph_MFP/Ph_MFP))/(Ph_MFP) ! [1/cm^3] density of incomming electrons
		else ! single electron:
			N_abs = 1.d0
		endif
endselect


#ifdef OMP_inside
   call OMP_SET_DYNAMIC(0) 	 ! standard openmp subroutine
   call OMP_SET_NUM_THREADS(Num_th) ! start using threads with openmp: Num_th is the number of threads, defined in the input file
!#else
   !print*, 'No openmp to deal with...'
#endif

!RRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRR

!FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
! Files where we save the outputs:
File_name = 'OUTPUT_'//trim(adjustl(Output_file_name))//'_MFP.dat'
open(101, FILE = trim(adjustl(File_name)), action="write", status="replace")
!FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
! Writing_var = 'Test error message'
! call Save_error_details(Error_occured, 2, Writing_var)

!CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
!Cross-section and inelastic mean free path saved to file:
!do j = 1,2
write(101, '(a)') '# Electron energy [eV],	Cross-section [A^2],	IMFP [A],	EMFP [A],	Mean scattering time [fs],	Energy loss [eV/A],	Electron range [nm]'
E = Ip_elements(1,size(Ip_elements,2))
!do i = 1,Eph
i = 0
do while (E .LT. max(Eph,1.d5))
   i = i + 1
   !E = E + 0.1d0*E ! [eV]
   if (E .LT. 10.0d0) then
      E = E + 0.1d0
   else if (E .LT. 100.0d0) then
      E = E + 1.0d0
   else if (E .LT. 1000.0d0) then
      E = E + 10.0d0
   else if (E .LT. 10000.0d0) then
      E = E + 100.0d0
   else if (E .LT. 100000.0d0) then
      E = E + 1.0d3
   else
      E = E + 1.0d4
   endif
enddo

if (.not. allocated(IMFPs)) allocate(IMFPs(size(Ip_elements,1),size(Ip_elements,2)))
if ((CSN(1,1) .EQ. 3) .or. (CSN(1,1) .EQ. 1)) then ! in this case we want to get IMFP:
    want_to_do = .true. ! by default we want to do anything!..
else ! or not:
    want_to_do = .false. ! or not...
endif

do j = 1, size(Ip_elements,1)
   do k = 1, size(Ip_elements,2)
      !if ((CSN(1,1) .EQ. 3) .or. (CSN(1,1) .EQ. 1)) then
      if (want_to_do) then ! do:
         if (.not. allocated(IMFPs(j,k)%E)) allocate(IMFPs(j,k)%E(i))
         if (.not. allocated(IMFPs(j,k)%MFP)) allocate(IMFPs(j,k)%MFP(i))
         IMFPs(j,k)%E = 0.0d0
         IMFPs(j,k)%MFP = 1.0d24
      endif
   enddo
enddo

if (CSN(1,1) .NE. 0) then
 do j = 1, size(Ip_elements,1)
   do k = 1, size(Ip_elements,2)
      if (Ip_elements(j,k) .GT. 0.0d0) then
         !trim(adjustl(Shell_elements(j,k)))//' of '//trim(adjustl(Element(k)%Name))
         open(399+(j-1)*size(Ip_elements,2)+k, FILE='OUTPUT_'//trim(adjustl(Output_file_name))//'_'//trim(adjustl(Element(j)%Name))//'_'//trim(adjustl(Shell_elements(j,k)))//'_MFP.dat', action="write", status="replace")
      endif
   enddo
 enddo

 e_range = 0.0d0
 dEdx = 0.0d0
 dEdx0 = 1.0d20
 E = Ip_elements(1,size(Ip_elements,2))
 E0 = E
 i = 0
 do while (E .LT. max(Eph,1.d5))
   i = i + 1 
   Sigma_tot = 0.0d0
   !E = E + 0.1d0*E ! [eV]
   if (E .LT. 10.0d0) then
      E = E + 0.1d0
   else if (E .LT. 100.0d0) then
      E = E + 1.0d0
   else if (E .LT. 1000.0d0) then
      E = E + 10.0d0
   else if (E .LT. 10000.0d0) then
      E = E + 100.0d0
   else if (E .LT. 100000.0d0) then
      E = E + 1.0d3
   else
      E = E + 1.0d4
   endif

   !call Total_cross_section(1, E, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFP)
   call El_free_flight_time(CSN, E, N_elements, Z_elements, M_elements, Ip_elements, Ekin_elements, Nel_elements, PQN_elements, At_dens, Sigma_tot, IMFPs, IMFP, TIM, Ritchi, i, dEdx)

    if (Total_elastic_cross_section(N_elements, Z_elements,E,1.d0) .gt. 1.d-40) then
       EMFP = 1.d0/(at_dens*Total_elastic_cross_section(N_elements, Z_elements,E,1.d0))*1.d24 		! n*lambda*sigma_tot = 1
    else 
       EMFP = 1.d99		! the elastic mean free path is infinite when the cross section is zero		
    endif

!    write(*, '(e25.16,e25.16,e25.16,e25.16,e25.16,e25.16)') E, Sigma_tot, IMFP, TIM
   !if ((CSN(1,1) .EQ. 3) .or. (CSN(1,1) .EQ. 1)) then
   if (want_to_do) then ! do:
     e_range = e_range + 0.5d0*(1.0d0/dEdx + 1.0d0/dEdx0)*(E - E0)
     write(101, '(f25.16,e25.16,f25.16,f25.16,f25.16,f25.16,f25.16)') E, Sigma_tot, IMFP, EMFP, TIM, dEdx, e_range/10.0d0
     !~ write(*, '(f25.16,e25.16,f25.16,f25.16,f25.16,f25.16)') E, Sigma_tot, IMFP, TIM, dEdx, e_range/10.0d0
     dEdx0 = dEdx ! save for integration
     E0 = E ! save for integration
     !write(*, '(f25.16,e25.16,f25.16,f25.16,f25.16)') E, Sigma_tot, IMFP, TIM, dEdx
     do j = 1, size(Ip_elements,1)
        do k = 1, size(Ip_elements,2)
           if (Ip_elements(j,k) .GT. 0.0d0) then
              write(399+(j-1)*size(Ip_elements,2)+k, '(f25.16,e25.16)') IMFPs(1,size(Ip_elements,2))%E(i), IMFPs(j,k)%MFP(i)
!              print*, 'file ', 399+(j-1)*size(Ip_elements,2)+k, IMFPs(1,size(Ip_elements,2))%E(i), IMFPs(j,k)%MFP(i)
           endif
        enddo
     enddo
   endif
 enddo
 print*, 'The calculated IMFP are saved in file: ', trim(adjustl(File_name))
 ! enddo
 ! stop 'MFP stop'
 flush(101)
 CLOSE(101)
 do j = 1, size(Ip_elements,1)
   do k = 1, size(Ip_elements,2)
      if (Ip_elements(j,k) .GT. 0.0d0) then
         !if ((CSN(1,1) .EQ. 3) .or. (CSN(1,1) .EQ. 1)) then
         if (want_to_do) then ! do:
            close(399+(j-1)*size(Ip_elements,2)+k)
         else
            close(399+(j-1)*size(Ip_elements,2)+k, status='delete')
         endif
         IMFPs(j,k)%E = 0.0d0
         IMFPs(j,k)%MFP = 1.0d24
      endif
   enddo
 enddo
endif
! print*, 'Everything is fine'
!CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

!MCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMC
! Setting up the initial distributions:
Ne_stat = 0.0d0 ! number of electrons for statistics
Ne_max = CEILING(Eph/MINVAL(MINVAL(Ip_elements, 2, (Ip_elements .GT. 0.0d0)))) ! max number of electrons
if (NE_max .gt. 2147483647) stop "NE_max must have type integer*8, otherwise it's too large"

N_shells = size(Ip_elements,2) ! number of shells
l = N_shells*NoEiC ! number of shells for each element together

 ! start of the time
 if (Fluence .GT. 0.0d0) then
   t = Pulse_max - 3.0d0*Pulse_duration  ! start of the time
else
   t = 0.0d0
endif

N_tim = CEILING((Total_t - t)/Global_dt) ! number of grid-points in time
! print*, 'N_tim',N_tim
call Distribbutions_update(N_tim,l,El_distr,Hole_distr) ! MC

if (print_distr) call Energy_distributions_update(N_tim, Eph, El_distr)  ! MC


! just set up the time steps once
do i = 1,N_tim
   El_distr%t(i) = t	! set time steps [fs]
   Hole_distr%t(i) = t	! set time steps [fs]
   t = t + Global_dt	! [fs]
enddo


allocate(Temp_El_distr(N_tim))	! electron distribution to work with openmp
Temp_El_distr = 0.0d0
allocate(Temp_El_NRG_distr(N_tim))	! electron energy distribution to work with openmp
Temp_El_NRG_distr = 0.0d0
if (print_distr) then
   allocate(Temp_El_distr_E(size(El_distr%Ee_vs_t,1),size(El_distr%Ee_vs_t,2)))
   Temp_El_distr_E = 0.0d0
   allocate(Temp_El_distr_E_grid(size(El_distr%Ee_vs_t_grid)))
   Temp_El_distr_E_grid(:) = El_distr%Ee_vs_t_grid(:)
endif
allocate(Temp_Hol_distr(l,N_tim))
Temp_Hol_distr(:,:) = 0.0d0


call delete_old_cascades()


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!print*, "MC  start  "
if (NMC .LE. 0) goto 2012 ! if there is no MC, don't plot stuff either
! The iteration in MC are largely independent, so they can be parallelized with openmp:
!$omp parallel &
!$omp private (stat)
!$omp do reduction( + :Temp_El_distr, Temp_Hol_distr, Temp_El_NRG_distr)
do stat = 1,NMC
#ifdef OMP_inside
   my_id = OMP_GET_THREAD_NUM() ! identify which thread it is
#else
   !print*, 'No openmp to deal with...'
#endif
   call MC(CSN, N_tim, Ne_max, Fluence, Eph, Z_elements, Element, M_elements, N_elements, Ip_elements, whether_valence, Effgap, Egap, Ekin_elements, Nel_elements, PQN_elements, t_Auger, Photoabs_sigma, At_dens, Photoabs_sigma_tot, koe, KOPulse, Pulse_max, Pulse_duration, Total_t, Global_dt, Temp_El_distr, Temp_El_NRG_distr, Temp_El_distr_E, Temp_El_distr_E_grid, Temp_Hol_distr, incl_elastic, print_distr, print_casc, Ecut, stat, Error_occured, IMFPs, Ritchi)
!    do i = 1,size(Temp_Hol_distr,2)
!       print*, i, Temp_Hol_distr(:,i)
!    enddo
enddo
!$omp end do
!$omp end parallel


if (print_casc) then	! preparing data for scattering of valence holes; this process will take place in XCascade-3D code
	if (mheff.gt.0.d0) then
	
	min_Ip_elements = MINVAL(Ip_elements, MASK=(Ip_elements > 1.0d-10)) ! find the minimal ionization potential among all involved atoms:
	
		if (Effgap .gt. 0.d0) then
			sigma_vhole = Total_elastic_cross_section(N_elements, Z_elements,Effgap-Egap,mheff)
			velh = dsqrt(2.d0*(Effgap-Egap)/(mheff*g_me))
		elseif (Effgap.le.0.d0) then
			sigma_vhole = Total_elastic_cross_section(N_elements, Z_elements,min_Ip_elements,mheff)	
			velh = dsqrt(2.d0*(5.39d0)/(mheff*g_me))
		endif
	
		if (sigma_vhole .gt. 1.d-40) then
			tfp_h = 1.d0/(at_dens*sigma_vhole)*1.d24 	/ velh	! n*lambda*sigma_tot = 1
		else 
			tfp_h = 1.d99		! the elastic mean free motion time is infinite when the cross section is zero		
		endif

		call inout_xcascade3d(fluence, mheff, tfp_h, min_Ip_elements)
	endif
endif


El_distr%ne(:) = Temp_El_distr(:)/DBLE(NMC)*N_abs ! [electron/cm^3]
El_distr%Ee(:) = Temp_El_NRG_distr(:)/DBLE(NMC)*N_abs	! [eV/cm^3]
if (print_distr) then
!    print*, size(El_distr%Ee_vs_t,2), size(Temp_El_distr_E,2)
!    print*, size(El_distr%Ee_vs_t,1), size(Temp_El_distr_E,1)
   do j = 1,size(El_distr%Ee_vs_t,1) ! time
      do k = 1,size(El_distr%Ee_vs_t,2) ! energy grid points
         if (El_distr%ne(j) .GT. 0.0d0) then
            El_distr%Ee_vs_t(j,k) = Temp_El_distr_E(j,k)/DBLE(NMC)*N_abs !/El_distr%ne(j)	! [electron/eV] vs time
         else
            El_distr%Ee_vs_t(j,k) = 0.0d0
         endif
      enddo
   enddo
endif
Hole_distr%nh = Temp_Hol_distr/DBLE(NMC)*N_abs	! [hole/cm^3]

! PRINT WARNING OF TOO HIGH FLUENCE:
!print*, 'Els:', maxval(El_distr%ne), 0.1d0*At_dens
if (maxval(El_distr%ne) .GT. 0.1d0*At_dens) then
   write(*,'(a)') '_______________________________________________________________________'
   write(*,'(a)') '                              ATTENTION!'
   write(*,'(a)') 'WARNING:'
   write(*,'(a, e10.3, a)') 'The produced electron density reaches ', maxval(El_distr%ne), ' [1/cm^3]'
   write(*,'(a, e10.3, a)') 'it is higher than 10% of the atomic density (Nat=', At_dens, ' [1/cm^3])'
   write(*,'(a)') 'At such conditions the results might be NOT reliable'
   write(*,'(a)') 'since no Pauli blocking effects are incorporated into the model'
   write(*,'(a)') 'and no scattering on ionized atoms is taken into account!'
   write(*,'(a)') '_______________________________________________________________________'
endif

! do i = 1,size(Temp_Hol_distr,2)
!    print*, i, Hole_distr%nh(:,i)
! enddo
! stop 'CHECK'
!Hole_distr%Eh = (Temp_Hol_distr*Ip_elements)/DBLE(NMC)
deallocate(Temp_El_distr)
deallocate(Temp_El_NRG_distr)
deallocate(Temp_Hol_distr)
if (print_distr) then
!    print*, 'deallocate(Temp_El_distr_E)', allocated(Temp_El_distr_E)
   deallocate(Temp_El_distr_E)
!    print*, 'deallocate(Temp_El_distr_E_grid)', allocated(Temp_El_distr_E_grid)
   deallocate(Temp_El_distr_E_grid)
endif
!  print*, 'L', stat, my_id, Ne_stat

!MCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMCMC
! Saving the resulting destributions into output file:
call Printing_output_MC_densities(Output_file_name, NMC, N_tim, NoEiC, N_shells, Fluence, N_abs, Eph, Ip_elements, El_distr, Hole_distr, print_distr) ! module "Creating_output_files"
do i=1,N_tim
	if (El_distr%ne(i) .ge. 0.90d0*maxval(El_distr%ne)) then
		write(*,"(a,1x,E13.5,1x,a,1x,f12.4,1x,a)") " 90% of max density is", 0.90d0*maxval(El_distr%ne), "[1/cm^3]. It was reached at", El_distr%t(i), "fs"
		print*, ''
		exit	
	endif
enddo

call Printing_output_MC_energies(Output_file_name, NMC, N_tim, NoEiC, N_shells, Fluence, N_abs, Eph, Ip_elements, El_distr, Hole_distr) ! module "Creating_output_files"
! And preparing the gnuplot script to plot the output:
if (Fluence .GT. 0) then ! plots for density:
   call Gnuplot_plotting(1, Output_file_name, Eph, Fluence, Pulse_max, Pulse_duration, Total_t, Global_dt, PQN_elements, Ip_elements, N_shells, NoEiC, Element, Shell_elements) ! module "Creating_output_files"
   call Gnuplot_plotting(2, Output_file_name, Eph, Fluence, Pulse_max, Pulse_duration, Total_t, Global_dt, PQN_elements, Ip_elements, N_shells, NoEiC, Element, Shell_elements) ! module "Creating_output_files"
else ! plots for electrons per cascade:
   call Gnuplot_plotting(1, Output_file_name, Eph, Fluence, Pulse_max, Pulse_duration, Total_t, Global_dt, PQN_elements, Ip_elements, N_shells, NoEiC, Element, Shell_elements, N_abs) ! module "Creating_output_files"
   call Gnuplot_plotting(2, Output_file_name, Eph, Fluence, Pulse_max, Pulse_duration, Total_t, Global_dt, PQN_elements, Ip_elements, N_shells, NoEiC, Element, Shell_elements, N_abs) ! module "Creating_output_files"
endif

! Copy the files into one folder:
!~ call make_output_folder(trim(adjustl(Output_file_name)), trim(adjustl(INPUT_FILE_NAME)), Fluence, Eph, Pulse_duration*2.355d0, NMC, koe) ! module "Creating_output_files"


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! Printing out the duration of the program, starting and ending time and date:
2012 continue
call date_and_time(values=c1)	    ! For calculation of the time of execution of the program
as1=DBLE(24*60*60*(c1(3)-ctim(3))+3600*(c1(5)-ctim(5))+60*(c1(6)-ctim(6))+(c1(7)-ctim(7))+(c1(8)-ctim(8))*0.001d0)	! sec
print*, '   '
if (as1 .lt. 60.d0) then
	write(*,'(a, f10.2, a)') 'Duration of execution of program: ', as1, ' [sec]'
else if (as1 .lt. 3600.d0) then
        write(*,'(a, f12.4, a)') 'Duration of execution of program: ', as1/60.d0, ' [min]'
else if (as1 .lt. 86400.d0) then
      write(*,'(a, f14.6, a)') 'Duration of execution of program: ', as1/3600.d0, ' [hours]'
else
      write(*,'(a, f16.8, a)') 'Duration of execution of program: ', as1/86400.d0, '[days]' 
endif
print*, '   '
write(*, 1001) ctim(5),ctim(6),ctim(7), ctim(3), ctim(2), ctim(1)

write(*, 1002) c1(5), c1(6), c1(7), c1(3),c1(2), c1(1)		

print*, '   '

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	
1001 format ('Begining: ', i2.2, ':', i2.2, ':', i2.2, '  ', i2.2, '/', i2.2, '/', i4.4)
1002 format ('The end:  ', i2.2, ':', i2.2, ':', i2.2, '  ', i2.2, '/', i2.2, '/', i4.4)
1005 format ('Start at: ', i2.2, ':', i2.2, ':', i2.2, '  ', i2.2, '/', i2.2, '/', i4.4)

 
!~ subroutine deallocate_all()
   !~ if (allocated(Z_elements)) deallocate(Z_elements)   ! atomic numbers of elements
   !~ if (allocated(M_elements)) deallocate(M_elements)   ! masses of all elements in molecule
   !~ if (allocated(N_elements)) deallocate(N_elements)   ! numbers of elements in 1 molecule
   !~ if (allocated(Ip_elements)) deallocate(Ip_elements) ! [eV] ionization potentials of all shells, all elements
   !~ if (allocated(Ekin_elements)) deallocate(Ekin_elements) ! [eV] kinetic energies of all shells, all elements
   !~ if (allocated(Nel_elements)) deallocate(Nel_elements)       ! number of electrons in all shells, all elements
   !~ if (allocated(NelVB_elements)) deallocate(NelVB_elements)   ! numbers of VALENCE electrons in shells
   !~ if (allocated(PQN_elements)) deallocate(PQN_elements)       ! principal quantum numbers 4 all shells, all elements
   !~ if (allocated(Shell_elements)) deallocate(Shell_elements)   ! names of all shells, all elements
   !~ if (allocated(t_Auger)) deallocate(t_Auger) ! Auger-decays times [fs]
   !~ if (allocated(Photoabs_sigma)) deallocate(Photoabs_sigma) ! [A^2] photoionization cross-sections for all shells
   !~ if (allocated(Element)) deallocate(Element)
   !~ if (allocated(Temp_El_distr)) deallocate(Temp_El_distr)     ! electron distribution to work with openmp
   !~ if (allocated(Temp_El_NRG_distr)) deallocate(Temp_El_NRG_distr)     ! electron distribution to work with openmp
   !~ if (allocated(Temp_El_distr_E)) deallocate(Temp_El_distr_E) ! electron deistribution in energy space to work with openmp
   !~ if (allocated(Temp_El_distr_E_grid)) deallocate(Temp_El_distr_E_grid) ! grid for electron deistribution in energy space to work with openmp
   !~ if (allocated(Temp_Hol_distr)) deallocate(Temp_Hol_distr)   ! holes distributions to work with openmp
   !~ if (allocated(Ritchi)) deallocate(Ritchi)    ! Ritchi CDF to use for cross-sections
   !~ if (allocated(IMFPs)) deallocate(IMFPs)     ! inelastic mean free paths from files
   !~ if (allocated(CSN)) deallocate(CSN) ! kinds of cross-sections
   !~ if (allocated(El_distr%t)) deallocate(El_distr%t) ! time instances
   !~ if (allocated(El_distr%ne)) deallocate(El_distr%ne) ! electron density
   !~ if (allocated(El_distr%Ee)) deallocate(El_distr%Ee) ! electron energy density
   !~ if (allocated(El_distr%Ee_vs_t)) deallocate(El_distr%Ee_vs_t)       ! transient electron distribution (1/E vs t)
   !~ if (allocated(El_distr%Ee_vs_t_grid)) deallocate(El_distr%Ee_vs_t_grid)     ! grid for transient electron distribution (
   !~ if (allocated(Hole_distr%t)) deallocate(Hole_distr%t) ! time instances
   !~ if (allocated(Hole_distr%nh)) deallocate(Hole_distr%nh) ! holes in different shells
   !~ if (allocated(Hole_distr%Eh)) deallocate(Hole_distr%Eh) ! holes energy density
!~ end subroutine 



END PROGRAM Cascading_code