!    XCASCADE-3D software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE-3D software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE-3D is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE-3D is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE-3D. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

MODULE Write_output
use DefaultParameters	! object oriented stuff 8-)
use Constants   ! universal constants 
use Monte_Carlo

implicit none

 contains

! This module does not follow the "Do Not Repeat Yourself" rule. However it allows to separate the different outputs and make them independently switchable and readable.
! It contains the algorithms to write various output files and to construct transient distributions of physical parameters
!=====================Write cascades==================================================

subroutine print_simulation_conditions(nmc, physical, switch, beam)
use DefaultParameters

	integer, intent(in) :: nmc
	type(physical_parameters_struct) , intent(in) ::physical
	type(io_switches_struct), intent(in) :: switch
	type(beam_struct), intent(in) :: beam
	
	if (nmc .le. 0) then
		print*, 'No cascades found in /cascades. Program stops.'
		stop
	else
		if (nmc.gt.1) then
			print*, 'Augmenting', nmc, 'cascades with spatial resolution'
		else
			print*, 'Augmenting', nmc, 'cascade with spatial resolution'
		endif
	endif

	if (physical%photoel_direction%direction .eq. 0) then
		if (physical%photoel_direction%axis .eq. 1) print *, "All photoelectrons fly in +X direction"
		if (physical%photoel_direction%axis .eq. 2) print *, "All photoelectrons fly in +Y direction"
		if (physical%photoel_direction%axis .eq. 3) print *, "All photoelectrons fly in +Z direction"
	else if (physical%photoel_direction%direction .eq. 1) then
		print *, "Photoelectrons fly isotropically (no polarization)"
	else if (physical%photoel_direction%direction .eq. 2 .or. physical%photoel_direction%direction .eq. 3) then
		if (physical%photoel_direction%axis .eq. 1) print *, "Photoelectrons fly in +/-X directions (linear polarization)"
		if (physical%photoel_direction%axis .eq. 2) print *, "Photoelectrons fly in +/-Y directions (linear polarization)"
		if (physical%photoel_direction%axis .eq. 3) print *, "Photoelectrons fly in +/-Z directions (linear polarization)"
	else if (physical%photoel_direction%direction .eq. 3) then
		print *, "Photoelectrons fly anisotropically with cos^2 distribution"
	endif

	if (physical%scattering_angles.eq.0) then
		print*, "Forward scattering is assumed"
	else if (physical%scattering_angles.eq.1) then
		print*, "Isotropic scattering is assumed"
	else if (physical%scattering_angles.eq.2) then
		!~ print*, "Anisotropic scattering is assumed"
	else 
		Stop "The parameter physical%scattering_angles is defined incorrectly"
	endif

	if (physical%inclHighEnEl) then
		print*, "High-energy electrons are included into the energy plots"
	else 
		print*, "High-energy electrons are excluded from the energy plots"
	endif
		print*, ""
		
	
	if (switch%electrons%make_3d.gt.0 .or. switch%events%make_3d.gt.0) write(*,'(A,1x,F9.2,1x,A)') " 3D distributions will be calculated at",switch%make3d_time,"fs"
		print*, ""
	

end subroutine print_simulation_conditions

!==========================================================================

subroutine WRITING_cascade_old(curr_iter,n_events,koe,nop,time_of_event,E_i, E_f, E2, Eh,Ne_CB, r_event,Etot,Ech)

integer, intent (IN) :: curr_iter			! current MC iteration = current cascade
integer, intent (IN) :: n_events			! number of events in this cascade
integer, intent (IN) :: koe(n_events)		! kind of event = photon absorption (1)/electron scattering (2=inelastic,4=elastic)/hole decay (3)
integer, intent (IN) :: nop(n_events)		! number of acting particle
real(8), intent (IN) :: time_of_event(n_events)			! time of a particular event
real(8), intent (IN) :: E_i(n_events), E_f(n_events), E2(n_events), Eh(n_events)		! [eV] initial energy of particle, final energy of particle after an event, energy of a secondary excited particle (ionization potential was subtracted)
Integer, intent (IN) :: Ne_CB(n_events)		! number of the excited particle during certain event = number of carriers in conduction band after the event
real(8), intent (IN) :: r_event(n_events,3)	! coordinates of each event
real(8), intent(in), optional :: Etot(n_events)
real(8), intent(in), optional :: Ech(n_events)

integer fn5	! file number
character(100) File_name, File_number, Error_descript
logical file_exist
integer end_file
integer curr_event				! each line corresponds to an event
real(8) tim

	fn5 = 10+curr_iter	! could it be dangerous?
	write(file_number,"(i7)") curr_iter

	! Write the cascades:
	file_name = 'cascades/OUTPUT_cascade'//trim(adjustl(file_number))//'.dat'
	inquire(file=file_name,exist=file_exist)
	if (file_exist) then
	open(unit = fn5, FILE = file_name, action="write", status = 'replace')
	else
	Error_descript = 'File '//file_name//' is not found!'
	print*, trim(adjustl(Error_descript))
	stop 'Stopping due to unrecoverable error'
	endif

	rewind(fn5)	! back to the beginning of the file


	if (present(Etot)) then

		if (present(Ech)) then

			write (fn5, *) '#KOE		', 'NOP		', 'Time		', 'Ei		', 'Ef		', 'E2		', 'Eh		', 'NeCB		', 'x		','y		','z		','Etot		','Ech		'
			
			do curr_event=1,n_events
				write (fn5, "(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6,4x,F18.10,4x,F18.10,4x,F18.10,4x,F10.4,4x,F10.4)")  & 
						koe(curr_event), nop(curr_event), time_of_event(curr_event), E_i(curr_event),E_f(curr_event), E2(curr_event), Eh(curr_event), Ne_CB(curr_event),  r_event(curr_event,:)*1.d9, Etot(curr_event)	, Ech(curr_event)			
			enddo
			
		else
		
			write (fn5, *) '#KOE		', 'NOP		', 'Time		', 'Ei		', 'Ef		', 'E2		', 'Eh		', 'NeCB		', 'x		','y		','z		','Etot		'
			
			do curr_event=1,n_events
				write (fn5, "(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6,4x,F18.10,4x,F18.10,4x,F18.10,4x,F10.4)")  & 
						koe(curr_event), nop(curr_event), time_of_event(curr_event), E_i(curr_event),E_f(curr_event), E2(curr_event), Eh(curr_event), Ne_CB(curr_event),  r_event(curr_event,:)*1.d9, Etot(curr_event)			
			enddo
		endif
		
	else

			write (fn5, *) '#KOE		', 'NOP		', 'Time		', 'Ei		', 'Ef		', 'E2		', 'Eh		', 'NeCB		', 'x		','y		','z		'

		do curr_event=1,n_events
			write (fn5, "(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6,4x,F18.10,4x,F18.10,4x,F18.10)")  & 
					koe(curr_event), nop(curr_event), time_of_event(curr_event), E_i(curr_event),E_f(curr_event), E2(curr_event), Eh(curr_event), Ne_CB(curr_event),  r_event(curr_event,:)*1.d9		
		enddo	

	endif

	close(fn5)
   
   
END subroutine WRITING_cascade_old



subroutine WRITING_cascade(curr_iter,n_events,event)
use Objects

integer, intent (IN) :: curr_iter			! current MC iteration = current cascade
integer, intent (IN) :: n_events			! number of events in this cascade
type(event_type), intent(in) :: event(n_events)

integer fn5	! file number
character(100) File_name, File_number, Error_descript
logical file_exist
integer end_file
integer curr_event				! each line corresponds to an event
real(8) tim

	fn5 = 10+curr_iter	! could it be dangerous?
	write(file_number,"(i7)") curr_iter

	! Write the cascades:
	file_name = 'cascades/OUTPUT_cascade'//trim(adjustl(file_number))//'.dat'
	inquire(file=file_name,exist=file_exist)
	if (file_exist) then
	open(unit = fn5, FILE = file_name, action="write", status = 'replace')
	else
	Error_descript = 'File '//file_name//' is not found!'
	print*, trim(adjustl(Error_descript))
	stop 'Stopping due to unrecoverable error'
	endif

	rewind(fn5)	! back to the beginning of the file


	write (fn5, *) '#KOE		', 'NOP		', 'Time		', 'Ei		', 'Ef		', 'E2		', 'Eh		', 'NeCB		', 'x		','y		','z		','Etot		','Ech		'
	
	do curr_event=1,n_events
		write (fn5, "(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6,4x,F18.10,4x,F18.10,4x,F18.10,4x,F10.4,4x,F10.4)")  & 
				event(curr_event)%koe, event(curr_event)%nop, event(curr_event)%time, event(curr_event)%E_i,event(curr_event)%E_f, event(curr_event)%E2,event(curr_event)%Eh, event(curr_event)%Ne_CB,  event(curr_event)%r(:)*1.d9, event(curr_event)%Etot	, event(curr_event)%Ech			
	enddo
			

	close(fn5)
   
   
END subroutine WRITING_cascade


!===================3d electron density======================================================

pure subroutine Forming_3d_electron_distribution(time_index,Nparticles,space, time,  r_timegrid, cur_3d_Ne)
use DefaultParameters	! object oriented stuff 8-)
integer, intent (IN) :: time_index					! at which moment we prepare the distribution
integer(8), intent (IN) :: Nparticles					! total number of electrons	
type(spatial_rgrid_struct), intent (IN) :: space			! contains information about space grid
type(temporal_tgrid_struct), intent (IN) :: time			! contains information about time grid
real(8), intent (IN) :: r_timegrid(Nparticles,time%Ntime,3)	! coordinates of each particle at each time moment 
real(8), intent(inout) :: cur_3d_Ne(-space%NX:space%NX,-space%NY:space%NY,-space%NZ:space%NZ)

real(8) real_space_index_x,real_space_index_y,real_space_index_z
integer(8) space_index_x,space_index_y,space_index_z
integer :: curr_particle

	cur_3d_Ne(:,:,:) = 0.d0

	particles_loop: do curr_particle=1, Nparticles

			real_space_index_x = r_timegrid(curr_particle,time_index,1)/space%dx
			real_space_index_y = r_timegrid(curr_particle,time_index,2)/space%dy
			real_space_index_z = r_timegrid(curr_particle,time_index,3)/space%dz

			! avoiding possible integer overflow:
			if (   abs(real_space_index_x).gt.dble(space%NX+space%dx) .or. &
				abs(real_space_index_y).gt.dble(space%NY+space%dy) .or. &
				abs(real_space_index_z).gt.dble(space%NZ+space%dz) ) cycle particles_loop
				
			space_index_x = nint(real_space_index_x,8)
			space_index_y = nint(real_space_index_y,8)
			space_index_z = nint(real_space_index_z,8)
			
			! avoiding array out of boundaries // ignoring particles outside of the computational box:
			if (   abs(space_index_x) .gt. space%NX .or. &
				abs(space_index_y) .gt. space%NY .or. &
				abs(space_index_z) .gt. space%NZ   ) cycle particles_loop
				
			cur_3d_Ne(space_index_x,space_index_y,space_index_z) = &
			cur_3d_Ne(space_index_x,space_index_y,space_index_z) + 1.d0
				
	enddo particles_loop

end subroutine Forming_3d_electron_distribution



subroutine writing_electrons_3d_distribution(curr_time,Nparticles,space, time, avg_3d_Ne)	
use DefaultParameters	! object oriented stuff 8-)
real(8) :: curr_time
integer(8), intent (IN) :: Nparticles				! total number of electrons	
type(spatial_rgrid_struct), intent (IN) :: space			! contains information about space grid
type(temporal_tgrid_struct), intent (IN) :: time		! contains information about time grid
real(8), intent(in) :: avg_3d_Ne(-space%NX:space%NX,-space%NY:space%NY,-space%NZ:space%NZ)

integer(8) space_index_x,space_index_y,space_index_z
real(8) volume

open(unit=155, file='OUTPUTs/OUTPUT_electrons3d.dat', action="write", status = 'replace')
print*, "Generating output file OUTPUT_electrons3d.dat"
print*, "  The grid covered", sum(avg_3d_Ne), "electrons"

write (155,'(A,1x,F16.2,A)') "# time = ", curr_time, " fs"
write (155,'(4(A))') "# Coordinate 1 (nm), Coordinate 2 (nm), Coordinate 3 (nm), Electron density (nm^-3)"

	volume = space%dx*1.d9 * space%dy*1.d9 * space%dz*1.d9

	do space_index_x = -space%NX, space%NX
	do space_index_y = -space%NY, space%NY
	do space_index_z = -space%NZ, space%NZ

		write (155,'(3(F9.2,1x),3(E12.5,1x))') dble(space_index_x)*space%dx*1.d9,dble(space_index_y)*space%dy*1.d9,dble(space_index_z)*space%dz*1.d9,avg_3d_Ne(space_index_x,space_index_y,space_index_z)/volume	! per dx*dy*dz

	enddo
	!~ write (155,*) ''	! for gnuplot contour style
	enddo
	write (155,*) ''	! for gnuplot contour style
	enddo

close (unit=155)

end subroutine writing_electrons_3d_distribution

!=================  / 3d electron density======================================================






!===================Write contour plots======================================================

pure subroutine Forming_spherical_holes(n_events, koe, r_event,time_of_event,Ntime,Nspace,min_time,dt,dr, Eh, E_f, out_spherical_Eh)
integer, intent(in) :: N_events
integer, intent(in) :: koe(n_events) 
real(8), intent(in) :: r_event(n_events,3) 
real(8), intent(in) :: time_of_event(n_events) 
integer(8), intent (IN) :: Ntime	! number of time steps
integer(8), intent (IN) :: Nspace	! number of space points
real(8), intent(in) :: min_time	! beginning of time 
real(8), intent(in) :: dt, dr 	! time and space steps
!~ real(8), intent (IN) :: r_timegrid(Nparticles,Ntime,3)	! coordinates of each particle at each time moment 
real(8), intent(in) :: Eh(N_events)
real(8), intent(in) :: E_f(n_events)
real(8), intent(inout) :: out_spherical_Eh(Ntime,Nspace)

real(8) radius
integer(8) time_index, curr_particle, space_index, curr_event

out_spherical_Eh(:,:) = 0.d0

events_loop: do curr_event = 1, N_events	

			radius = dsqrt( dot_product(r_event(curr_event,:),r_event(curr_event,:)) )
			
			if ( radius/dr+1.d0 .gt. Nspace+dr ) cycle events_loop
			space_index = nint( radius/dr, 8 )+1	
			
			if (koe(curr_event).eq.3 .and. E_f(curr_event).gt.0.d0) then
				time_index = nint((time_of_event(curr_event)-min_time)/dt,8)
				if ( abs(time_of_event(curr_event)/dt)+1.d0.gt.Ntime ) cycle events_loop	
				out_spherical_Eh(time_index, space_index) = out_spherical_Eh(time_index, space_index) + E_f(curr_event)
			endif

			if (abs(space_index).le.Nspace .and. Eh(curr_event).gt.0.d0) then
				time_index = nint((time_of_event(curr_event)-min_time)/dt,8)
				if ( abs(time_of_event(curr_event)/dt)+1.d0.gt.Ntime ) cycle events_loop	
				out_spherical_Eh(time_index, space_index) = out_spherical_Eh(time_index, space_index) + Eh(curr_event)
			endif

enddo events_loop

end subroutine Forming_spherical_holes



subroutine writing_spherical_holes(n_events,Ntime,Nspace,min_time,dt,dr, spherical_Eh)
integer, intent(in) :: N_events
integer(8), intent (IN) :: Ntime	! number of time steps
integer(8), intent (IN) :: Nspace	! number of space points
real(8), intent(in) :: min_time	! beginning of time 
real(8), intent(in) :: dt, dr 	! time and space steps
!~ real(8), intent (IN) :: r_timegrid(Nparticles,Ntime,3)	! coordinates of each particle at each time moment 
real(8), intent(in) :: spherical_Eh(Ntime,Nspace)

real(8) radius, volume
integer(8) time_index, curr_particle, space_index, curr_event

	open(unit=166, file='OUTPUTs/OUTPUT_spherical_holes.dat', action="write", status = 'replace')
	print*, "Generating output file OUTPUT_spherical_holes.dat"
	print*, "  The grid covered", sum(spherical_Eh(:,:)), " eV energy"
	print*, ''

	write (166,'(4(A))') "# Distance (nm), Average number of holes, Average energy (eV), Volume (nm^3)"

		do space_index = 1, Nspace
			
			volume = 4.d0/3.d0*Pi * ((dble(space_index)*dr*1.d9)**3 - (dble(space_index-1)*dr*1.d9)**3)
		
			write (166,'(5(E13.5))') dble(space_index)*dr*1.d9 &
				,sum(spherical_Eh(:,space_index))/5.39d0, sum(spherical_Eh(:,space_index)),volume	! energy per volume (eV/nm^3); testing; the energy is not correct now!
		enddo
		
		write (166,*) " " 	! needed for gnuplot style

	close(unit=166)

end subroutine writing_spherical_holes



! forming the space indexes for the variables on a contour grid; the time grid has already been constructed
pure subroutine Forming_spherical(Nparticles,Ntime,Nspace, min_time,dt,dr, r_timegrid, vel_timegrid, E_timegrid, out_spherical_Ne,out_spherical_Ee,out_spherical_Ee2)

integer(8), intent (IN) :: Nparticles	! number of particles
integer(8), intent (IN) :: Ntime	! number of time steps
integer(8), intent (IN) :: Nspace	! number of space points
real(8), intent(in) :: min_time	! beginning of time 
real(8), intent(in) :: dt, dr 	! time and space steps
real(8), intent (IN) :: r_timegrid(Nparticles,Ntime,3)	! coordinates of each particle at each time moment 
real(8), intent (IN) :: vel_timegrid(Nparticles,Ntime,3)	! velocities of each particle at each time moment 
real(8), intent (IN) :: E_timegrid(Nparticles,Ntime)	! energies of each particle at each time moment 

real(8), intent(out) :: out_spherical_Ne(Ntime,Nspace)
real(8), intent(out) :: out_spherical_Ee(Ntime,Nspace)
real(8), intent(out) :: out_spherical_Ee2(Ntime,Nspace)

real(8) radius
integer(8) curr_particle
integer(8) space_index, time_index

Real(8) Ne_temp(Ntime,Nspace), Ee_temp(Ntime,Nspace), Ee2_temp(Ntime,Nspace)!, Ve_temp(Ntime,Nspace,3)

Ne_temp(:,:) = 0.d0
Ee_temp(:,:) = 0.d0
Ee2_temp(:,:) = 0.d0

time_loop: do time_index = 1, Ntime	
	particle_loop: do curr_particle = 1, Nparticles
				
			radius = dsqrt( dot_product(r_timegrid(curr_particle,time_index,:),r_timegrid(curr_particle,time_index,:)) )
			if ( radius/dr.ge.dble(huge(space_index)) ) cycle particle_loop
			
			space_index = nint( radius/dr, 8 )+1	
			if (space_index .le. Nspace) then
				Ne_temp(time_index,space_index) = Ne_temp(time_index,space_index) + 1.d0
				Ee_temp(time_index,space_index) = Ee_temp(time_index,space_index) + 0.5d0*me/e*dot_product(vel_timegrid(curr_particle,time_index,:),vel_timegrid(curr_particle,time_index,:))
				Ee2_temp(time_index,space_index) = Ee2_temp(time_index,space_index) + E_timegrid(curr_particle,time_index)
			endif

	enddo particle_loop
enddo time_loop


do time_index = 1, Ntime
do space_index = 1, Nspace
	!~ if (Ne_temp(time_index,space_index) .lt. 1.d-50) Ne_temp(time_index,space_index)=0.d0
	!~ if (Ee_temp(time_index,space_index).lt. 1.d-50) Ee_temp(time_index,space_index)=0.d0
	!~ if (Ee2_temp(time_index,space_index).lt. 1.d-50) Ee2_temp(time_index,space_index)=0.d0
	
	out_spherical_Ne(time_index,space_index) = Ne_temp(time_index,space_index)
	out_spherical_Ee(time_index,space_index) = Ee_temp(time_index,space_index)
	out_spherical_Ee2(time_index,space_index) = Ee2_temp(time_index,space_index)
enddo
enddo

END subroutine Forming_spherical


subroutine writing_spherical(min_time,dt,Ntime,dr,Nspace,spherical_Ne,spherical_Ee,spherical_Ee2)
real(8), intent(in) :: min_time
real(8), intent(in) :: dt, dr	! temporal and spatial time steps
integer(8), intent(in) :: Ntime, Nspace
real(8), intent(in) :: spherical_Ne(Ntime,Nspace)	! number of electrons depending on time and distance to the first photon
real(8), intent(in) :: spherical_Ee(Ntime,Nspace)	! energy of electrons depending on time and distance to the first photon
real(8), intent(in) :: spherical_Ee2(Ntime,Nspace)	! energy of electrons depending on time and distance to the first photon; calculated from velocities

integer(8) time_index, space_index
real(8) volume

open(unit=177, file='OUTPUTs/OUTPUT_spherical.dat', action="write", status = 'replace')
print*, "Generating output file OUTPUT_spherical.dat"
write(*,'(A,1x,f19.12,1x,A,1x,f19.12,1x,f19.12,A)') "  The grid covered", sum(spherical_Ne(Ntime,:)), "electrons and", sum(spherical_Ee(Ntime,:)), sum(spherical_Ee2(Ntime,:)), ' eV energy'

write (177,'(4(A))') "# Time (fs), Spherical radius (nm), Average number of electrons, Average energy (eV)"


do time_index = 1, Ntime

	do space_index = 1, Nspace
		
		volume = 4.d0/3.d0*Pi * ((dble(space_index)*dr*1.d9)**3 - (dble(space_index-1)*dr*1.d9)**3)
	
		write (177,'(6(E13.5))') dble(time_index)*dt+min_time,dble(space_index)*dr*1.d9,spherical_Ne(time_index,space_index) &
			, spherical_Ee(time_index,space_index), spherical_Ee2(time_index,space_index),volume	! energy per volume (eV/nm^3)
	enddo
	
	write (177,*) " " 	! needed for gnuplot style
enddo

close (unit=177)

end subroutine writing_spherical



!=================CYLINDRICAL===========================


! forming the space indexes for the variables on a contour grid; the time grid has already been constructed
pure subroutine Forming_cylindrical(Nparticles,Ntime,Nspace, min_time,dt,drho, r_timegrid, vel_timegrid, E_timegrid, out_cylind_Ne,out_cylind_Ee,out_cylind_Ee2)

integer(8), intent (IN) :: Nparticles	! number of particles
integer(8), intent (IN) :: Ntime	! number of time steps
integer(8), intent (IN) :: Nspace	! number of space points
real(8), intent(in) :: min_time	! beginning of time 
real(8), intent(in) :: dt, drho 	! time and space steps
real(8), intent (IN) :: r_timegrid(Nparticles,Ntime,3)	! coordinates of each particle at each time moment 
real(8), intent (IN) :: vel_timegrid(Nparticles,Ntime,3)	! velocities of each particle at each time moment 
real(8), intent (IN) :: E_timegrid(Nparticles,Ntime)	! energies of each particle at each time moment 

real(8), intent(out) :: out_cylind_Ne(Ntime,Nspace)
real(8), intent(out) :: out_cylind_Ee(Ntime,Nspace)
real(8), intent(out) :: out_cylind_Ee2(Ntime,Nspace)

real(8) rho	! cylindrhoical radius
integer(8) curr_particle
integer(8) space_index, time_index

out_cylind_Ne(:,:) = 0.d0
out_cylind_Ee(:,:) = 0.d0
out_cylind_Ee2(:,:) = 0.d0

time_loop: do time_index = 1, Ntime	
	particle_loop: do curr_particle = 1, Nparticles
				
			rho = dsqrt( dot_product(r_timegrid(curr_particle,time_index,1:2),r_timegrid(curr_particle,time_index,1:2)) )
			if ( rho/drho.ge.dble(huge(space_index)) ) cycle particle_loop
			
			space_index = nint( rho/drho, 8 )+1	
			if (space_index .le. Nspace) then
				out_cylind_Ne(time_index,space_index) = out_cylind_Ne(time_index,space_index) + 1.d0
				out_cylind_Ee(time_index,space_index) = out_cylind_Ee(time_index,space_index) + E_timegrid(curr_particle,time_index)
				out_cylind_Ee2(time_index,space_index) = out_cylind_Ee2(time_index,space_index) + 0.5d0*me/e*dot_product(vel_timegrid(curr_particle,time_index,:),vel_timegrid(curr_particle,time_index,:))
			endif

	enddo particle_loop
enddo time_loop


END subroutine Forming_cylindrical


subroutine writing_cylindrical(min_time,dt,Ntime,drho,Nspace,cylind_Ne,cylind_Ee,cylind_Ee2)
real(8), intent(in) :: min_time
real(8), intent(in) :: dt, drho	! temporal and spatial time steps
integer(8), intent(in) :: Ntime, Nspace
real(8), intent(in) :: cylind_Ne(Ntime,Nspace)	! number of electrons depending on time and distance to the first photon
real(8), intent(in) :: cylind_Ee(Ntime,Nspace)	! energy of electrons depending on time and distance to the first photon
real(8), intent(in) :: cylind_Ee2(Ntime,Nspace)	! energy of electrons depending on time and distance to the first photon; calculated from velocities

integer(8) time_index, space_index
real(8) area

open(unit=177, file='OUTPUTs/OUTPUT_cylindrical.dat', action="write", status = 'replace')
print*, "Generating output file OUTPUT_cylindrical.dat"
write(*,'(A,1x,f19.12,1x,A,1x,f19.12,A)') "  The grid covered", sum(cylind_Ne(Ntime,:)), "electrons and", sum(cylind_Ee(Ntime,:)), ' eV energy'
print*, ''

write (177,'(4(A))') "# Time (fs), Cylindrical radius (nm), Average number of electrons per dz, Average energy (eV/dz), ..., area (nm^2)"


do time_index = 1, Ntime

	do space_index = 1, Nspace
		
		area = Pi * ((dble(space_index)*drho*1.d9)**2 - (dble(space_index-1)*drho*1.d9)**2)
	
		write (177,'(3(E18.5),1x,a,1x,3(E18.5))') dble(time_index)*dt+min_time,dble(space_index)*drho*1.d9,cylind_Ne(time_index,space_index) &
			, "N/A", cylind_Ee(time_index,space_index),area	! energy per volume (eV/nm^2)
	enddo
	
	write (177,*) " " 	! needed for gnuplot style
enddo

close (unit=177)

end subroutine writing_cylindrical



!!!!!!!!!!!!!!!!!!!!!!!!   Z - DEPENDENCE  !!!!!!!!!!!!!!!!!!!!!!!!!!!!


! forming the space indexes for the variables on a contour grid; the time grid has already been constructed
pure subroutine Forming_Z_dependence(Nparticles,Ntime,Nspace, min_time,dt,dz, r_timegrid, vel_timegrid, E_timegrid, out_Z_Ne,out_Z_Ee,out_Z_Ee2)

integer(8), intent (IN) :: Nparticles	! number of particles
integer(8), intent (IN) :: Ntime	! number of time steps in the Z plot
integer(8), intent (IN) :: Nspace	! number of space points in the Z plot
real(8), intent(in) :: min_time	! beginning of time for the Z plot
real(8), intent(in) :: dt, dz 	! time and space steps in the Z plot
real(8), intent (IN) :: r_timegrid(Nparticles,Ntime,3)	! coordinates of each particle at each time moment on the Z grid
real(8), intent (IN) :: vel_timegrid(Nparticles,Ntime,3)	! velocities of each particle at each time moment on the Z grid
real(8), intent (IN) :: E_timegrid(Nparticles,Ntime)	! energies of each particle at each time moment on the Z grid

real(8), intent(out) :: out_Z_Ne(Ntime,-Nspace:Nspace)
real(8), intent(out) :: out_Z_Ee(Ntime,-Nspace:Nspace)
real(8), intent(out) :: out_Z_Ee2(Ntime,-Nspace:Nspace)
!~ real(8), intent(out) :: out_Z_EeTot(Ntime,-Nspace:Nspace)

real(8) Z_coordinate
integer curr_particle
integer(8) space_index, time_index

out_Z_Ne(:,:) = 0.d0
out_Z_Ee(:,:) = 0.d0
out_Z_Ee2(:,:) = 0.d0
!~ out_Z_EeTot(:,:) = 0.d0

time_loop: do time_index = 1, Ntime
	particle_loop: do curr_particle = 1, Nparticles
				
			Z_coordinate = r_timegrid(curr_particle,time_index,3)
			if ( abs(Z_coordinate/dz).ge.dble(huge(space_index)) ) cycle particle_loop
	
			space_index = nint( Z_coordinate/dz, 8 )	
			if (abs(space_index) .le. Nspace) then
				out_Z_Ne(time_index,space_index) = out_Z_Ne(time_index,space_index) + 1
				out_Z_Ee(time_index,space_index) = out_Z_Ee(time_index,space_index) + 0.5d0*me/e*dot_product(vel_timegrid(curr_particle,time_index,:),vel_timegrid(curr_particle,time_index,:))
				out_Z_Ee2(time_index,space_index) = out_Z_Ee2(time_index,space_index) + E_timegrid(curr_particle,time_index)
				!~ out_Z_EeTot(time_index,space_index) = out_Z_EeTot(time_index,space_index) + sum E_timegrid(curr_particle,time_index)
			endif

	enddo particle_loop
enddo time_loop


END subroutine Forming_Z_dependence



subroutine writing_Z_dependence(min_time,dt,Ntime,dz,Nspace,Z_Ne,Z_Ee,Z_Ee2,zt_output)
real(8), intent(in) :: min_time
real(8), intent(in) :: dt, dz	! temporal and spatial time steps
integer(8), intent(in) :: Ntime, Nspace
real(8), intent(in) :: Z_Ne(Ntime,-Nspace:Nspace)	! number of electrons depending on time and distance to the first photon
real(8), intent(in) :: Z_Ee(Ntime,-Nspace:Nspace)	! energy of electrons depending on time and distance to the first photon
real(8), intent(in) :: Z_Ee2(Ntime,-Nspace:Nspace)	! energy of electrons depending on time and distance to the first photon; calculated from velocities
integer, intent(in) :: zt_output						! output format: swap time and space

real(8) Z_Ne_diff(Ntime,-Nspace:Nspace)	! added number of electrons per time step depending on time and distance to the first photon
real(8) Z_Ee_diff(Ntime,-Nspace:Nspace)	! added  energy of electrons per time step depending on time and distance to the first photon
real(8) Z_Ee2_diff(Ntime,-Nspace:Nspace)	! added energy of electrons per time step depending on time and distance to the first photon; calculated from velocities


integer(8) time_index, space_index

!~ area = 2.d0*Nspace*2.d0*Nspace*dz*dz*1.d9*1.d9
!~ volume = dx*dy*dz*1.d9*1.d9*1.d9


if (zt_output .eq. 2 .or. zt_output .eq. 3) then


	open(unit=135, file='OUTPUTs/OUTPUT_Z_dependence_zt.dat', action="write", status = 'replace')
	print*, "Generating output file OUTPUT_Z_dependence_zt.dat"
	
	write (135,'(4(A))') "# Coordinate (nm), Time (fs), Average number of electrons per dz, Average energy density (eV/dz), Velocity-based average energy density (eV/dz)"

	do space_index = -Nspace,Nspace
		do time_index = 1, Ntime
			write (135,'(5(E13.5))') dble(space_index)*dz*1.d9,dble(time_index)*dt+min_time,Z_Ne(time_index,space_index) &
				, Z_Ee2(time_index,space_index), Z_Ee(time_index,space_index)	! energy per length (eV/dz)
		enddo
		
		write (135,*) " " 	! needed for gnuplot style
	enddo
	
	close (unit=135)

endif

if (zt_output .eq. 1 .or. zt_output .eq. 3) then


	open(unit=136, file='OUTPUTs/OUTPUT_Z_dependence.dat', action="write", status = 'replace')
	print*, "Generating output file OUTPUT_Z_dependence.dat"
	
	write (136,'(4(A))') "# Time (fs), Coordinate (nm), Average number of electrons per dz, Average energy density (eV/dz)"
	
	do time_index = 1, Ntime
		do space_index = -Nspace,Nspace
		
			write (136,'(5(E13.5))') dble(time_index)*dt+min_time,dble(space_index)*dz*1.d9,Z_Ne(time_index,space_index) &
				, Z_Ee2(time_index,space_index), Z_Ee(time_index,space_index)	! energy per length (eV/dz)
		
		enddo
		
		write (136,*) " " 	! needed for gnuplot style
	enddo
	
	close (unit=136)

endif
	
	Z_Ne_diff(1,:) = Z_Ne(1,:)
	Z_Ee_diff(1,:) = Z_Ee(1,:)
	Z_Ee2_diff(1,:) = Z_Ee2(1,:)
	
		do time_index = 2, Ntime
			Z_Ne_diff(time_index,:) = Z_Ne(time_index,:) - Z_Ne(time_index-1,:)
			Z_Ee_diff(time_index,:) = Z_Ee(time_index,:) - Z_Ee(time_index-1,:)
			Z_Ee2_diff(time_index,:) = Z_Ee2(time_index,:) - Z_Ee2(time_index-1,:)
		enddo
	
	write(*,'(A,1x,f19.12,1x,A,1x,f19.12,1x,f19.12,A)') "  The grid covered", sum(Z_Ne_diff(:,:)), "electrons and", sum(Z_Ee_diff(:,:)), sum(Z_Ee2_diff(:,:)), ' eV energy'
	!~ print*, ''

! checking consevation laws
	!~ open(unit=323, file='OUTPUTs/OUTPUT_conservation_laws.dat', action="write", status = 'replace')
		!~ do time_index = 1, Ntime
		!~ write (323,'(5(E13.5))') dble(time_index)*dt+min_time, sum(Z_Ne(time_index,:)), sum(Z_Ee(time_index,:))
		!~ enddo
	!~ close (unit=323)

end subroutine writing_Z_dependence




!!!========================HOLES=================================

subroutine writing_holes_3d(dx,dy,dz,Nspace,holes_3d_in)	! writing 3d spatial distribution of the holes

real(8), intent (IN) :: dx,dy,dz	! number of space points
integer(8), intent (IN) :: Nspace	! number of space points
real(8), intent(in) :: holes_3d_in(1:3,-Nspace:Nspace,-Nspace:Nspace,-Nspace:Nspace)

real(8) :: holes_3d(1:3,-Nspace:Nspace,-Nspace:Nspace,-Nspace:Nspace), volume
integer(8) space_index_x, space_index_y, space_index_z

holes_3d(1:3,:,:,:) = holes_3d_in(1:3,:,:,:)

open(unit=52, file='OUTPUTs/OUTPUT_holes3d.dat', action="write", status = 'replace')
print*, "Generating output file OUTPUT_holes3d.dat"

write (52,'(4(A))') "# Coordinate 1 (nm), Coordinate 2 (nm), Coordinate 3 (nm), Average number of single holes per nm^3 (x,y,z)"

	volume = dx*1.d9 * dx*1.d9 * dx*1.d9

	do space_index_x = -Nspace, Nspace
	do space_index_y = -Nspace, Nspace
	do space_index_z = -Nspace, Nspace
		
		write (52,'(3(F9.2,1x),3(F11.4,1x))') dble(space_index_x)*dx*1.d9, &
									   dble(space_index_y)*dx*1.d9, &
									   dble(space_index_z)*dx*1.d9, &
																holes_3d(1, space_index_x,space_index_y,space_index_z) / volume !	! per nm^3

	enddo
	write (52,*) ''	! for gnuplot contour style
	enddo
	write (52,*) ''	! for gnuplot contour style
	enddo

close (unit=52)

end subroutine writing_holes_3d



! forming the space indexes for the variables on a contour grid; the time grid has already been constructed
pure subroutine Forming_Z_dependence_holes(n_events, koe, E_f, r_event,time_of_event,Ntime,Nspace,min_time,dt,dz, Eh, out_Z_Nh,out_Z_Eh)

integer, intent(in) :: N_events
integer, intent(in) :: koe(n_events)
real(8), intent(in) :: E_f(n_events)
real(8), intent(in) :: r_event(n_events,3) 
real(8), intent(in) :: time_of_event(n_events) 
integer(8), intent (IN) :: Ntime	! number of time steps
integer(8), intent (IN) :: Nspace	! number of space points
real(8), intent(in) :: min_time	! beginning of time
real(8), intent(in) :: dt, dz 	! time and space steps
real(8), intent(in) :: Eh(N_events)

real(8), intent(out) :: out_Z_Nh(Ntime,-Nspace:Nspace)
real(8), intent(out) :: out_Z_Eh(Ntime,-Nspace:Nspace)

real(8) :: tmp_Z_Nh(Ntime,-Nspace:Nspace)
real(8) :: tmp_Z_Eh(Ntime,-Nspace:Nspace)

real(8) Z_coordinate
integer curr_event
integer(8) space_index, time_index

tmp_Z_Nh(:,:) = 0.d0
tmp_Z_Eh(:,:) = 0.d0

! counting the valence holes created during all events
time_index = Ntime

events_loop: do curr_event = 1, N_events	

			Z_coordinate = r_event(curr_event,3)
			if ( abs(Z_coordinate/dz).ge.dble(huge(space_index)) ) cycle events_loop
			space_index = nint( Z_coordinate/dz, 8 )	
			if (abs(space_index) .gt. Nspace) cycle events_loop
			
                        time_index = nint((time_of_event(curr_event)-min_time)/dt,8)
                        if ( abs(time_of_event(curr_event)/dt).ge.dble(Ntime) ) cycle events_loop ! we might loose half of the statistics in the last cell due to how nint works here...

			if (koe(curr_event).eq.3 .and. E_f(curr_event).gt.0.d0) then
				tmp_Z_Nh(time_index, space_index) = tmp_Z_Nh(time_index, space_index) + 1.d0	! core hole also became a valence hole
				tmp_Z_Eh(time_index, space_index) = tmp_Z_Eh(time_index, space_index) + E_f(curr_event)
			endif
			
			if (abs(space_index) .gt. Nspace .or. Eh(curr_event).le.0.d0) cycle events_loop
			
			tmp_Z_Nh(time_index, space_index) = tmp_Z_Nh(time_index, space_index) + 1.d0	! valence hole created
			tmp_Z_Eh(time_index, space_index) = tmp_Z_Eh(time_index, space_index) + Eh(curr_event)

enddo events_loop


! these holes should stay there forever
do space_index=-Nspace,Nspace
	tmp_Z_Nh(1,space_index) = tmp_Z_Nh(1,space_index)
	tmp_Z_Eh(1,space_index) = tmp_Z_Eh(1,space_index)
	do time_index = 2, Ntime
		!~ if (tmp_Z_Nh(time_index,space_index) .lt. 1.d-50) tmp_Z_Nh(time_index,space_index)=0.d0
		!~ if (tmp_Z_Eh(time_index,space_index).lt. 1.d-50) tmp_Z_Eh(time_index,space_index)=0.d0
		
		tmp_Z_Nh(time_index,space_index) = tmp_Z_Nh(time_index,space_index)+tmp_Z_Nh(time_index-1,space_index)
		tmp_Z_Eh(time_index,space_index) = tmp_Z_Eh(time_index,space_index)+tmp_Z_Eh(time_index-1,space_index)
	enddo
enddo

out_Z_Nh(:,:) = tmp_Z_Nh(:,:)
out_Z_Eh(:,:) = tmp_Z_Eh(:,:)


END subroutine Forming_Z_dependence_holes


subroutine writing_Z_dependence_holes(min_time,dt,Ntime,dz,Nspace,Z_Nh,Z_Eh,zt_output)
real(8), intent(in) :: min_time
real(8), intent(in) :: dt, dz	! temporal and spatial time steps
integer(8), intent(in) :: Ntime, Nspace
real(8), intent(in) :: Z_Nh(Ntime,-Nspace:Nspace)	! number of electrons depending on time and distance to the first photon
real(8), intent(in) :: Z_Eh(Ntime,-Nspace:Nspace)	! energy of electrons depending on time and distance to the first photon
integer, intent(in) :: zt_output						! output format: swap time and space

integer(8) time_index, space_index

real(8) :: Z_Nh_diff(Ntime,-Nspace:Nspace)	! number of electrons depending on time and distance to the first photon
real(8) :: Z_Eh_diff(Ntime,-Nspace:Nspace)	! energy of electrons depending on time and distance to the first photon

!~ area = 2.d0*Nspace*2.d0*Nspace*dz*dz*1.d9*1.d9
!~ volume = dx*dy*dz*1.d9*1.d9*1.d9

Z_Nh_diff(1,:) = Z_Nh(1,:)
Z_Eh_diff(1,:) = Z_Eh(1,:)
	do time_index = 2, Ntime
		Z_Nh_diff(time_index,:) = Z_Nh(time_index,:) -Z_Nh(time_index-1,:) 
		Z_Eh_diff(time_index,:) = Z_Eh(time_index,:) - Z_Eh(time_index-1,:) 
	enddo

if (zt_output .eq. 2 .or. zt_output .eq. 3) then


	open(unit=135, file='OUTPUTs/OUTPUT_Z_dependence_holes_zt.dat', action="write", status = 'replace')
	print*, "Generating output file OUTPUT_Z_dependence_holes_zt.dat"
	
	write (135,'(4(A))') "# Coordinate (nm), Time (fs), Average number of holes per dz, Average energy density (eV/dz)"

	do space_index = -Nspace,Nspace
		do time_index = 1, Ntime
			write (135,'(5(E13.5))') dble(space_index)*dz*1.d9,dble(time_index)*dt+min_time &
			, Z_Nh(time_index,space_index), Z_Eh(time_index,space_index)	! energy per length (eV/dz)
		enddo
		
		write (135,*) " " 	! needed for gnuplot style
	enddo
	
	close (unit=135)

endif

if (zt_output .eq. 1 .or. zt_output .eq. 3) then

	open(unit=136, file='OUTPUTs/OUTPUT_Z_dependence_holes.dat', action="write", status = 'replace')
	print*, "Generating output file OUTPUT_Z_dependence_holes.dat"
	write(*,'(A,1x,f19.12,1x,A,1x,f19.12,A)') "  The grid covered", sum(Z_Nh_diff(:,:)), "holes and", sum(Z_Eh_diff(:,:)), ' eV energy'
	print*, ''
	
	write (136,'(4(A))') "# Time (fs), Coordinate (nm), Average number of electrons per dz, Average energy density (eV/dz)"
	
	do time_index = 1, Ntime
		do space_index = -Nspace,Nspace
		
			write (136,'(5(E13.5))') dble(time_index)*dt+min_time,dble(space_index)*dz*1.d9 &
			, Z_Nh(time_index,space_index), Z_Eh(time_index,space_index)	! energy per length (eV/dz)
		
		enddo
		
		write (136,*) " " 	! needed for gnuplot style
	enddo
	
	close (unit=136)

endif

end subroutine writing_Z_dependence_holes



!!============================================
!!==============EVENTS========================
!!============================================

subroutine writing_NEevents(min_time,dt,Ntime,dz,Nspace,Nevents,Eevents,XTANT_OUTPUT_events)
real(8), intent(in) :: min_time
real(8), intent(in) :: dt, dz	! temporal and spatial time steps
integer(8), intent(in) :: Ntime, Nspace
real(8), intent(in) :: Nevents(Ntime,-Nspace:Nspace) 	! space-time distribution of the occured events
real(8), intent(in) :: Eevents(Ntime,-Nspace:Nspace) 	! space-time distribution of the occured events
integer, intent(in) :: XTANT_OUTPUT_events

integer(8) time_index, space_index

!~ area = 2.d0*Nspace*2.d0*Nspace*dz*dz*1.d9*1.d9
!~ volume = dx*dy*dz*1.d9*1.d9*1.d9

if (XTANT_OUTPUT_events .eq. 2 .or. XTANT_OUTPUT_events .eq. 3) then

	open(unit=131, file='OUTPUTs/OUTPUT_events_zt_XTANT.dat', action="write", status = 'replace')
	print*, "Generating output file for XTANT: OUTPUT_events_zt_XTANT.dat"
	write(*,'(A,1x,f19.12,1x,A,1x,f19.12,A)') "  The grid covered", sum(Nevents(:,:)), "events and", sum(Eevents(:,:)),  ' eV energy'
	print*, ''
	
	write (131,'(4(A))') "# Coordinate (nm), Time (fs), Average number of events per dz, Average energy density released (eV/dz)"

	do space_index = -Nspace,Nspace
		do time_index = 1, Ntime
				write (131,'(5(E13.5))') dble(space_index)*dz*1.d9,dble(time_index)*dt+min_time,Nevents(time_index,space_index) &
					, Eevents(time_index,space_index)	! energy per length (eV/dz)   ! number of events is not calculated correctly yet
		enddo
		
		write (131,*) " " 	! needed for gnuplot style
	enddo
	
	close (unit=131)

endif

if (XTANT_OUTPUT_events .eq. 1 .or. XTANT_OUTPUT_events .eq. 3) then
	
	print *, "Warning: tz version of OUTPUT_events_zt_XTANT.dat is not implemented, so the file is not written"

endif


end subroutine writing_NEevents




! create a grid in one dimension with how many events occured at a particular distance
pure subroutine forming_events_1d(n_events,koe,dr,Nspace,r_event,curr_events_1d)

integer, intent (IN) :: n_events	! number of events
integer, intent (IN) :: koe(n_events)		! kind of event = photon/elastic electron/inelastic electron/hole
real(8), intent(in) :: dr 		! time step in the grid;currently, the spatial grid is the same in all directions
integer(8), intent (IN) :: Nspace	! number of space points
real(8), intent(in) :: r_event(n_events,3) 	! time and space steps
real(8), intent(out) :: curr_events_1d(0:Nspace)

integer curr_event
real(8) real_spher_coord
integer(8) spher_coord

	curr_events_1d(:) = 0.d0

	events_loop: do curr_event=1, n_events
	
		if ( koe(curr_event) .eq. 2 ) then! only plotting inelastic collisions

			real_spher_coord = dsqrt(r_event(curr_event,1)**2 + r_event(curr_event,2)**2 + r_event(curr_event,3)**2 )/dr

			! avoiding possible integer overflow:
			if (   real_spher_coord.gt.dble(Nspace+dr) ) cycle events_loop
				
			spher_coord = nint(real_spher_coord,8)
					
			if (   spher_coord .gt. Nspace ) cycle events_loop

			curr_events_1d(spher_coord) = curr_events_1d(spher_coord) + 1.d0
									
		endif
		
	enddo events_loop

end subroutine forming_events_1d


subroutine writing_events_1d(dr,Nspace,events_1d_in)	! writing spatial distribution of the events

real(8), intent (IN) :: dr	! number of space points
integer(8), intent (IN) :: Nspace	! number of space points
real(8), intent(in) :: events_1d_in(0:Nspace)

real(8) :: events_1d(0:Nspace), volume
integer(8) spher_coord

events_1d(:) = events_1d_in(:)

open(unit=122, file='OUTPUTs/OUTPUT_events1d.dat', action="write", status = 'replace')
print*, "Generating output file OUTPUT_events1d.dat"
print*, "  The grid covered", sum(events_1d(:)), " inelastic events"

write (122,'(4(A))') "# Coordinate r (nm), Average number of events per nm^3, volume"

	do spher_coord = 1, Nspace

		volume = 4.d0/3.d0*Pi * ((dble(spher_coord)*dr*1.d9)**3 - (dble(spher_coord-1)*dr*1.d9)**3)

                write (122,'(F9.2,1x,F11.5,1x,E15.7)') dble(spher_coord)*dr*1.d9,events_1d(spher_coord)/(dr*1.d9 * dr*1.d9 * (3.d0*dr*1.d9)),volume	! per nm^3

	enddo

close (unit=122)

end subroutine writing_events_1d


! create a grid in x,y with how many events occured in 2d
pure subroutine forming_events_2d(n_events,koe,dx,dy,NX,NY,r_event,curr_events_2d)

integer, intent (IN) :: n_events	! number of space points
integer, intent (IN) :: koe(n_events)		! kind of event = photon/elastic electron/inelastic electron/hole
real(8), intent(in) :: dx,dy 		! time step in the grid;currently, the spatial grid is the same in all directions
integer(8), intent (IN) :: NX,NY	! number of space points
real(8), intent(in) :: r_event(n_events,3) 	! time and space steps
real(8), intent(inout) :: curr_events_2d(-NX:NX,-NY:NY)

real(8) real_space_index_x,real_space_index_y
integer curr_event
integer(8) space_index_x,space_index_y
 
	curr_events_2d(:,:) = 0.d0

	events_loop: do curr_event=1, n_events
	
		if ( koe(curr_event) .eq. 2 ) then! only plotting inelastic collisions, or valence holes

			real_space_index_x = r_event(curr_event,1)/dx
			real_space_index_y = r_event(curr_event,2)/dy

			! avoiding possible integer overflow:
			if (   abs(real_space_index_x).gt.dble(NX+dx) .or. &
				abs(real_space_index_y).gt.dble(NY+dy)    ) cycle events_loop
				
			space_index_x = nint(real_space_index_x,8)
			space_index_y = nint(real_space_index_y,8)
			
			if (   abs(space_index_x) .gt. NX .or. &
				abs(space_index_y) .gt. NY    ) cycle events_loop
				
			curr_events_2d(space_index_x,space_index_y) = &
			curr_events_2d(space_index_x,space_index_y) + 1.d0
									
		endif

	enddo events_loop

end subroutine forming_events_2d





subroutine writing_events_2d(dx,dy,NX,NY,events_2d_in)	! writing spatial distribution of the events
real(8), intent (IN) :: dx,dy	! number of space points
integer(8), intent (IN) :: NX,NY	! number of space points
real(8), intent(in) :: events_2d_in(-NX:NX,-NY:NY)

real(8) :: volume, area
integer(8) space_index_x, space_index_y

open(unit=188, file='OUTPUTs/OUTPUT_events2d.dat', action="write", status = 'replace')
print*, "Generating output file OUTPUT_events2d.dat"
print*, "  The grid covered", sum(events_2d_in(:,:)), " inelastic events"

write (188,'(4(A))') "# Coordinate 1 (nm), Coordinate 2 (nm), Average number of events (x,y)"

	!volume = dx*1.d9 * dx*1.d9 * (3.d0*dx*1.d9)
	!area = dx*1.d9 * dy*1.d9

	do space_index_x = -NX, NX
	do space_index_y = -NY, NY
		
		write (188,'(2(F9.2,1x),3(F11.5,1x))') dble(space_index_x)*dx*1.d9,dble(space_index_y)*dy*1.d9,events_2d_in(space_index_x,space_index_y)	! per nm^3

	enddo
	write (188,*) ''	! for gnuplot contour style
	enddo

close (unit=188)
end subroutine writing_events_2d





! create a grid in x,y,z with how many events occured in a particular 3d-volume
pure subroutine forming_events_3d(n_events,koe,space,r_event,curr_events_3d)
integer, intent (IN) :: n_events	! number of events
integer, intent (IN) :: koe(n_events)		! kind of event = photon/elastic electron/inelastic electron/hole
type(spatial_rgrid_struct), intent (IN) :: space
real(8), intent(in) :: r_event(n_events,3) 	! time and space steps
real(8), intent(inout) :: curr_events_3d(-space%NX:space%NX,-space%NY:space%NY,-space%NZ:space%NZ)

real(8) :: real_space_index_x,real_space_index_y,real_space_index_z
integer(8) :: space_index_x,space_index_y,space_index_z
integer :: curr_event
integer(8) :: NX,NY,NZ
real(8) :: dx,dy,dz
NX = space%NX
NY = space%NY
NZ = space%NZ
dx = space%dx
dy = space%dy
dz = space%dz

curr_events_3d(:,:,:) = 0.d0

	events_loop: do curr_event=1, n_events
	
		if ( koe(curr_event) .eq. 2 ) then! only plotting inelastic collisions, or valence holes

			real_space_index_x = r_event(curr_event,1)/dx
			real_space_index_y = r_event(curr_event,2)/dy
			real_space_index_z = r_event(curr_event,3)/dz

			! avoiding possible integer overflow:
			if (   abs(real_space_index_x).gt.dble(NX+dx) .or. &
				abs(real_space_index_y).gt.dble(NY+dy) .or. &
				abs(real_space_index_z).gt.dble(NZ+dz) ) cycle events_loop
				
			space_index_x = nint(real_space_index_x,8)
			space_index_y = nint(real_space_index_y,8)
			space_index_z = nint(real_space_index_z,8)
			
			! avoiding array out of boundaries:
			if (   abs(space_index_x) .gt. NX .or. &
				abs(space_index_y) .gt. NY .or. &
				abs(space_index_z) .gt. NZ   ) cycle events_loop
				
			curr_events_3d(space_index_x,space_index_y,space_index_z) = &
			curr_events_3d(space_index_x,space_index_y,space_index_z) + 1.d0
									
		endif

	enddo events_loop

end subroutine forming_events_3d



subroutine writing_events_3d(space,events_3d_in)	! writing spatial distribution of the events
type(spatial_rgrid_struct), intent (IN) :: space
real(8), intent(in) :: events_3d_in(-space%NX:space%NX,-space%NY:space%NY,-space%NZ:space%NZ)

real(8) :: events_3d(-space%NX:space%NX,-space%NY:space%NY,-space%NZ:space%NZ), volume
integer(8) space_index_x, space_index_y, space_index_z
integer :: NX,NY,NZ
real(8) :: dx,dy,dz
NX = space%NX
NY = space%NY
NZ = space%NZ
dx = space%dx
dy = space%dy
dz = space%dz

events_3d(:,:,:) = events_3d_in(:,:,:)

open(unit=176, file='OUTPUTs/OUTPUT_events3d.dat', action="write", status = 'replace')
print*, "Generating output file OUTPUT_events3d.dat"
print*, "  The grid covered", sum(events_3d(:,:,:)), " inelastic events"
print*, ""

	write (176,'(A,1x,F9.2,A)') "# calculated at the end of simulation"
write (176,'(4(A))') "# Coordinate 1 (nm), Coordinate 2 (nm), Coordinate 3 (nm), Average number of events (x,y,z), Average number of events (x,y,z), Average number of events (x,y,z)"

	volume = dx*1.d9 * dy*1.d9 * dz*1.d9

	do space_index_x = -NX, NX
	do space_index_y = -NY, NY
	do space_index_z = -NZ, NZ

		write (176,'(3(F9.2,1x),3(F11.4,1x))') dble(space_index_x)*dx*1.d9,dble(space_index_y)*dy*1.d9,dble(space_index_z)*dz*1.d9,events_3d(space_index_x,space_index_y,space_index_z)	! per dx*dy*dz

	enddo
	!~ write (176,*) ''	! for gnuplot contour style
	enddo
	write (176,*) ''	! for gnuplot contour style
	enddo

close (unit=176)

end subroutine writing_events_3d


!!============================================
!!==============/EVENTS========================
!!============================================



subroutine writing_convergence(num_cascades,Ne_tot_conv,noec_avg_conv,electron_range_conv)
integer, intent(in) :: num_cascades
Real(8), intent(in) :: Ne_tot_conv(num_cascades)   							! average number of electrons in a cascade
Real(8), intent(in) :: noec_avg_conv(num_cascades)							! total number of elastic collisions in an average cascade
Real(8), intent(in) :: electron_range_conv(3,num_cascades)					! how far the electrons fly before the thermalization on average

integer curr_cascade

open(unit=424, file='OUTPUTs/OUTPUT_convergence.dat', action="write", status = 'replace')
print*, "Generating output file OUTPUT_convergence.dat"

		write (424, *) '#Line number (minus 2) is the number of cascades/repetitions used to obtain those numbers, i.e. statistics'
		write (424, *) '#average number of electrons in a cascade;	total number of elastic collisions;	electron range 1;	electron range 2;	electron range 3'

	do curr_cascade=1,num_cascades
	
		write (424, '(F13.6,1x,F13.6,3(1x,F13.6))')	Ne_tot_conv(curr_cascade), noec_avg_conv(curr_cascade) &
							, electron_range_conv(1,curr_cascade)*1.d9, electron_range_conv(2,curr_cascade)*1.d9, electron_range_conv(3,curr_cascade)*1.d9	! the line number will be the number of iterations/cascades
	
	enddo

close (unit=424 )


end subroutine writing_convergence


subroutine writing_totenergy(Ntime,dt,max_time,min_time,Etot_timegrid)
integer(8), intent(in) :: Ntime
Real(8), intent(in) :: dt,max_time,min_time
Real(8), intent(in) :: Etot_timegrid(Ntime)					! how far the electrons fly before the thermalization on average

integer(8) time_index

open(unit=404, file='OUTPUTs/OUTPUT_totenergy.dat', action="write", status = 'replace')
print*, "Generating output file OUTPUT_totenergy.dat"

		write (404, *) '#Time (fs), Total Energy (eV)'

	do time_index=1,Ntime
	
		write (404, '(F13.5,1x,F13.6)')	dble(time_index)*dt+min_time,Etot_timegrid(time_index)	! the line number will be the number of iterations/cascades
	
	enddo

close (unit=404 )


end subroutine writing_totenergy



end module Write_output
