!    XCASCADE-3D software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE-3D software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE-3D is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE-3D is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE-3D. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains subroutines needed to read cascades/OUTPUT_cascade*.dat and input3d.txt. The reading of the latter is implemented using Fortran namelist
MODULE Read_input
implicit none

 contains

! Input files:
subroutine read_input_3d(numeric,physical,switch,beam)
	use DefaultParameters	! object oriented stuff 8-)
	
	type(numeric_struct), intent(inout) :: numeric
	type(physical_parameters_struct), intent(inout) :: physical
	type(io_switches_struct), intent(inout) :: switch
	type(beam_struct), intent(inout) :: beam

	namelist / numerical_parameters / numeric
	namelist / physical_parameters / physical
	namelist / convolution / beam
	namelist / io_switches / switch

	open (2, file='input3d.txt', status="old")
		read(2, nml = numerical_parameters)
		read(2, nml = physical_parameters)
		read(2, nml = io_switches)
		read(2, nml = convolution)
	close (2)

	! some corrections to the input values:
	beam%spot%x = beam%spot%x / (2.d0*dsqrt(2.d0*dlog(2.d0)))
	beam%spot%y = beam%spot%y / (2.d0*dsqrt(2.d0*dlog(2.d0)))

	numeric%grid%time%maxt = numeric%grid%time%maxt + 0.75d0*numeric%grid%time%dt	! not to go over the array boundaries in case some events occur a little bit later
	numeric%grid%time%Ntime = DINT((numeric%grid%time%maxt - numeric%grid%time%mint) / numeric%grid%time%dt)	! niumber of time grid points
		
	numeric%grid%space%maxX = numeric%grid%space%maxX*1.d-9 ! nm -> m
	numeric%grid%space%maxY = numeric%grid%space%maxY*1.d-9 ! nm -> m
	numeric%grid%space%maxZ = numeric%grid%space%maxZ*1.d-9 ! nm -> m
	numeric%grid%space%dx = numeric%grid%space%dx*1.d-9 ! nm -> m
	numeric%grid%space%dy = numeric%grid%space%dy*1.d-9 ! nm -> m
	numeric%grid%space%dz = numeric%grid%space%dz*1.d-9 ! nm -> m

	beam%spot%x = beam%spot%x * 1.d-9 ! nm -> m
	beam%spot%y = beam%spot%y * 1.d-9 ! nm -> m
	beam%spot%z = beam%spot%z * 1.d-9 ! nm -> m
	
	if (switch%events%make_3d.le.0 .and. switch%events%make_2d.le.0 .and. switch%events%make_1d.le.0 & 
	.and. switch%events%XTANT_events.le.0 .and. switch%electrons%spherical.le.0 .and. switch%electrons%cylindrical.le.0 .and. & 
	switch%electrons%Z_dependence.le.0 .and. switch%electrons%make_3d.le.0 .and. switch%flyholes%Z_fholes.le.0 .and. switch%flyholes%cylindrical.le.0 .and. &
	switch%flyholes%make_2d.le.0 .and. switch%flyholes%make_3d.le.0) then
	    print*, "All grids are off!"
	    !numeric%grid%time%dt = 25.d0 ! 0.5d0*(max_time-min_time) ! fs
	    numeric%grid%space%dx = numeric%grid%space%maxX
	    numeric%grid%space%dy = numeric%grid%space%maxY
	    numeric%grid%space%dz = numeric%grid%space%maxZ
	endif
	
	! default value for omp_threads:
	if (beam%numeric%omp_threads .eq. -1) beam%numeric%omp_threads = numeric%omp_numthr	! if number of OMP threads is not chosen for the convolution, it is equal to general number of OMP threads in the program

	! total number of photons in our simulation box
	if (beam%spot%z .gt. numeric%grid%space%maxZ) then
		beam%Nphotons = beam%Nphotons * (2.d-3*beam%spot%x*1.d9*2.d-3*beam%spot%y*1.d9*1.d-3*numeric%grid%space%maxZ*1.d9)	
	else
		beam%Nphotons = beam%Nphotons * (2.d-3*beam%spot%x*1.d9*2.d-3*beam%spot%y*1.d9*1.d-3*beam%spot%z*1.d9)
	endif

	numeric%grid%space%NX = IDINT(numeric%grid%space%maxX/numeric%grid%space%dx)
	numeric%grid%space%NY = IDINT(numeric%grid%space%maxY/numeric%grid%space%dy)
	numeric%grid%space%NZ = IDINT(numeric%grid%space%maxZ/numeric%grid%space%dz)

	! default value for make3d_time:
	if (switch%make3d_time.le.-1.d98) switch%make3d_time = numeric%grid%time%maxt
	if (switch%make3d_time_index.le.-1) then        ! if make3d_time_index is not set, we calculate it:
		switch%make3d_time_index = (switch%make3d_time-numeric%grid%time%mint)/numeric%grid%time%dt
	else    ! in case make3d_time_index is set in the input3d.txt file, we set 3d_time accordingly:
		switch%make3d_time = numeric%grid%time%mint + switch%make3d_time_index*numeric%grid%time%dt
	endif

end subroutine read_input_3d


!=========================================
!=========================================


function Number_of_cascades()

integer Number_of_cascades

integer fn, fnm
logical file_exist
character(100) file_name,file_number

! removing existing old files with cascades
    fn = 0
    file_exist = .true.
    
    do while (file_exist)
        fn = fn+1

        write (file_number, "(I6)") fn
        File_name = 'cascades/OUTPUT_cascade'//trim(adjustl(file_number))//'.dat'
	
        inquire(file=trim(adjustl(File_name)),exist=file_exist) ! check if the file is there

        !~ if (file_exist) then
	    
	    !~ if (.not. fn .eq. 6) then
	        !~ fnm = fn
	    !~ else 
	        !~ fnm = 666
	    !~ endif
	
            !~ open(unit=fnm, file=File_name, status='old')
	    !~ close(unit=fnm)
        !~ endif

    enddo
    
Number_of_cascades = fn-1

end  function Number_of_cascades


subroutine reading_data_from_XCASCADE(fluence, mheff, tfp_h)
	real(8), intent(out) :: fluence
	real(8), intent(out) :: mheff	! effective mass of a valence hole
	real(8), intent(out) :: tfp_h	! time of mean free path for holes

   open(201, FILE = 'INOUT_xcascade3d.iout', action="read", status = 'old')
	read(201,*) fluence, mheff, tfp_h
   close (unit=201)


end subroutine reading_data_from_XCASCADE


function READING_number_of_events(curr_iter)	! find how many lines/events contains cascade number curr_iter

integer READING_number_of_events		! current MC iteration = current cascade
integer, intent (IN) :: curr_iter		! current MC iteration = current cascade

logical file_exist
integer fn3,end_file
integer n_events
character(100) File_name, File_number, Error_descript

    fn3 = 10+curr_iter	! could it be dangerous?
    write(file_number,"(i7)") curr_iter

! Read the cascades:
  file_name = 'cascades/OUTPUT_cascade'//trim(adjustl(file_number))//'.dat'
    inquire(file=file_name,exist=file_exist)
    if (file_exist) then
    
        open(unit = fn3, FILE = file_name, status = 'old')
	
      ! counting the number of events written in this file = number of lines - first line which is a comment
        n_events = -1
        do
            READ(fn3,*,IOSTAT=End_file)
            IF (End_file .EQ. 0) n_events = n_events + 1
            IF (End_file .NE. 0) exit
        enddo
	
   else
      Error_descript = 'File '//file_name//' is not found!'
      print*, trim(adjustl(Error_descript))
      stop 'Stopping due to unrecoverable error'
   endif
   
   close(fn3)
   
   READING_number_of_events = n_events

END function READING_number_of_events


!=========================================
!=========================================
!=========================================


subroutine READING_cascade_old(curr_iter,n_events,koe,nop,t_cur,E_i, E_f, E2, Eh,Ne_CB)

integer, intent (IN) :: curr_iter	! current MC iteration = current cascade
integer, intent (IN) :: n_events	! number of events in this cascade
integer, intent (inout), ALLOCATABLE :: koe(:)		! kind of particle = photon/electron/hole
integer, intent (inout), ALLOCATABLE :: nop(:)		! number of acting particle
real(8), intent (inout), ALLOCATABLE :: t_cur(:)			! time of a particular event
real(8), intent (inout), ALLOCATABLE :: E_i(:), E_f(:), E2(:), Eh(:)		! [eV] initial energy of particle, final energy of particle after an event, energy of a secondary excited particle (ionization potential was subtracted), kinetic energy of holes
Integer, intent (inout), ALLOCATABLE :: Ne_CB(:)		! number of the excited particle during certain event = number of carriers in conduction band after the event

integer fn4	! file number
character(100) File_name, File_number, Error_descript
logical file_exist
integer end_file
integer curr_event				! each line corresponds to an event	
    

    fn4 = 10+curr_iter	! could it be dangerous?
    write(file_number,"(i7)") curr_iter

! Read the cascades:
  file_name = 'cascades/OUTPUT_cascade'//trim(adjustl(file_number))//'.dat'
    inquire(file=file_name,exist=file_exist)
    if (file_exist) then
        open(unit = fn4, FILE = file_name, status = 'old')
   else
      Error_descript = 'File '//file_name//' is not found!'
      print*, trim(adjustl(Error_descript))
      stop 'Stopping due to unrecoverable error'
   endif
   
    if (.not. allocated(koe)) allocate(koe(n_events),nop(n_events),t_cur(n_events),E_i(n_events), E_f(n_events), E2(n_events),Ne_CB(n_events))
   
    rewind(fn4)	! back to the beginning of the file
    read(fn4,*)

    do curr_event=1,n_events
        read (fn4, "(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6)")  koe(curr_event), nop(curr_event), t_cur(curr_event), E_i(curr_event),E_f(curr_event), E2(curr_event),Eh(curr_event), Ne_CB(curr_event)
    enddo

   close(fn4)
   
END subroutine READING_cascade_old



subroutine READING_cascade(curr_iter,n_events,event)
use Objects

integer, intent (IN) :: curr_iter	! current MC iteration = current cascade
integer, intent (IN) :: n_events	! number of events in this cascade
type(event_type), intent(inout) :: event(n_events)

integer fn4	! file number
character(100) File_name, File_number, Error_descript
logical file_exist
integer end_file
integer curr_event				! each line corresponds to an event	
    
    fn4 = 10+curr_iter	! could it be dangerous?
    write(file_number,"(i7)") curr_iter

! Read the cascades:
  file_name = 'cascades/OUTPUT_cascade'//trim(adjustl(file_number))//'.dat'
    inquire(file=file_name,exist=file_exist)
    if (file_exist) then
        open(unit = fn4, FILE = file_name, status = 'old')
   else
      Error_descript = 'File '//file_name//' is not found!'
      print*, trim(adjustl(Error_descript))
      stop 'Stopping due to unrecoverable error'
   endif
   
    rewind(fn4)	! back to the beginning of the file
    read(fn4,*)

    do curr_event=1,n_events
        read (fn4, "(i1,4x,i6,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,f25.16,4x,i6)")  event(curr_event)%koe, event(curr_event)%nop, event(curr_event)%time, event(curr_event)%E_i,event(curr_event)%E_f, event(curr_event)%E2,event(curr_event)%Eh, event(curr_event)%Ne_CB
    enddo

   close(fn4)
   
END subroutine READING_cascade



!~ subroutine read_fluence_2d(laser_spot)
!~ real(8), intent(inout) :: laser_spot(512,512)

!~ character(100) File_name, File_number, Error_descript
!~ logical file_exist
!~ integer fnum	! file number
!~ integer :: row, column

!~ fnum = 49
!~ file_name = 'Inputs/along_polar_30shots.txt'
    !~ inquire(file=file_name,exist=file_exist)
    !~ if (file_exist) then
        !~ open(unit = fnum, FILE = file_name, status = 'old')
   !~ else
      !~ Error_descript = 'File '//file_name//' is not found!'
      !~ print*, trim(adjustl(Error_descript))
      !~ stop 'Stopping due to unrecoverable error'
   !~ endif
   
   !~ do row=1,512
		!~ read(fnum,*) laser_spot(row,:)
   !~ enddo
   
   !~ close(fnum)

!~ end subroutine read_fluence_2d


subroutine read_fluence_1d(laser_spot,direction)
real(8), intent(inout) :: laser_spot(512)
integer :: direction ! 1 = photon polarization across the knife edge, 2 = along

	if (direction .eq. 2) then
	
		call read_fluence_1d_along(laser_spot)
		
	else if (direction .eq. 1) then
	
		call read_fluence_1d_across(laser_spot)
		
	else
		print *, "wrong polarization direction"
		stop
	endif
	
end subroutine read_fluence_1d




subroutine read_fluence_1d_along(laser_spot)
real(8), intent(inout) :: laser_spot(512)

character(100) File_name, File_number, Error_descript
logical file_exist
integer fnum	! file number
integer :: row, indx, index0
real(8) :: dx, minx, maxx, norm, this, last, buf1,buf2

fnum = 49
file_name = 'Inputs/along_polar_600shots-5%attenuation_dose.csv'
	inquire(file=file_name,exist=file_exist)
	if (file_exist) then
		open(unit = fnum, FILE = file_name, status = 'old')
	else
		Error_descript = 'File '//file_name//' is not found!'
		print*, trim(adjustl(Error_descript))
		stop 'Stopping due to unrecoverable error'
	endif

	do row=1,512
		read(fnum,*) buf1,buf2,laser_spot(row)
	enddo

	close(fnum)

	! normalizing
	norm = 0.d0
	last = 0.d0
	dx = 0.25d-6	! 250 nm, from experiments
	
	minx = -58.75d-6
	maxx = 69.d-6
	index0 = 236
	
	print *, index0 + minx/dx, index0 + maxx/dx
	
	do indx = nint(index0 + minx/dx), nint(index0 + maxx/dx)		! starting at index corrsponding to x = -5 um
		this = laser_spot(indx)
		norm = norm + dx*0.5d0*(this+last)
		last = this
		
		print *, indx, (indx-index0)*dx*1.d6, laser_spot(indx), norm
	enddo
	
	laser_spot(:) = laser_spot(:)/norm
	

end subroutine read_fluence_1d_along


subroutine read_fluence_1d_across(laser_spot)
real(8), intent(inout) :: laser_spot(512)

character(100) File_name, File_number, Error_descript
logical file_exist
integer fnum	! file number
integer :: row, indx, index0
real(8) :: dx, minx, maxx, norm, this, last, buf1,buf2

fnum = 49
file_name = 'Inputs/across_polar_600shots-5%attenuation_dose.csv'
	inquire(file=file_name,exist=file_exist)
	if (file_exist) then
		open(unit = fnum, FILE = file_name, status = 'old')
	else
		Error_descript = 'File '//file_name//' is not found!'
		print*, trim(adjustl(Error_descript))
		stop 'Stopping due to unrecoverable error'
	endif

	do row=1,512
		read(fnum,*) buf1,buf2,laser_spot(row)
	enddo

	close(fnum)

	! normalizing
	norm = 0.d0
	last = 0.d0
	dx = 0.25d-6	! 250 nm, from experiments
	
	minx = -67.75d-6
	maxx = 60.d-6
	index0 = 272
	
	print *, index0 + minx/dx, index0 + maxx/dx
	
	do indx = nint(index0 + minx/dx), nint(index0 + maxx/dx)		! starting at index corrsponding to x = -5 um
		this = laser_spot(indx)
		norm = norm + dx*0.5d0*(this+last)
		last = this
		print *, indx, (indx-index0)*dx*1.d6, laser_spot(indx), norm
	enddo
	
	laser_spot(:) = laser_spot(:)/norm


end subroutine read_fluence_1d_across


END MODULE Read_input
