!    XCASCADE-3D software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE-3D software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE-3D is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE-3D is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE-3D. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains most Monte-Carlo subroutines
MODULE Monte_Carlo

use Constants

implicit none

contains


subroutine start_with_photon(nmc, curr_cascade, n_events, curr_event, Ne, curr_particle, r_event0, r_event, r, E_threshold, E2, vel, photoel_direction, time_of_event, mheff, Eh, N_vh, rh, velh, t_nexth, Ee, tfp_h)
use DefaultParameters

implicit none 

	integer, intent(in) :: nmc
	integer, intent(in) :: curr_cascade
	integer, intent(in) :: n_events
	integer(8), intent(in) :: curr_event
	integer(8), intent(in) :: Ne
	integer(8), intent(in) :: curr_particle
	real(8), intent(in) :: r_event0(3)	
	real(8), intent(inout) :: r_event(n_events,3)
	real(8), intent(inout) :: r(Ne,n_events,3)
	
	real(8), intent(in) :: E_threshold
	real(8), intent(inout) :: E2(n_events)
	real(8), intent(inout) :: Ee(Ne,n_events)
	real(8), intent(inout) :: vel(Ne,n_events,3)
	type(photoel_direction_struct), intent(in) :: photoel_direction

	real(8), intent(in) :: time_of_event(n_events)
	real(8), intent(in) :: mheff
	real(8), intent(inout) :: Eh(n_events)
	integer, intent(inout) :: N_vh(n_events)
	real(8), intent(inout) :: rh(Ne+Ne/2,n_events,3)
	real(8), intent(inout) :: velh(Ne+Ne/2,n_events,3)
	real(8), intent(inout) :: t_nexth(Ne+Ne/2)
	real(8), intent(in) :: tfp_h

	real(8) vel_mod
	
	r_event(curr_event,:) = r_event0(:)	! using the initial place of the cascade
	r(curr_particle,curr_event,:) = r_event(curr_event,:)

	
	if (E2(curr_event) .gt. E_threshold) then
		vel_mod = dsqrt(2.d0/me*E2(curr_event)*e)	! E = mv^2/2		! speed of the first created electron (photoelectron) in current cascade

		! polarization of the incoming photon
		if (photoel_direction%direction .eq. 0) then						! all photoelectrons fly in one direction
		        vel(curr_particle,curr_event,1:3) = 0.d0
			vel(curr_particle,curr_event,photoel_direction%axis) = vel_mod
		else if (photoel_direction%direction .eq. 1) then					! photoelectrons fly in the random directions
			vel(curr_particle,curr_event,:) = vel_mod*random_direction()
		else if (photoel_direction%direction .eq. 2) then					! half of the photoelectrons flies in (0,0,1), another half in (0,0,-1)
			if (dble(curr_cascade)/dble(nmc).le.0.5d0) then
					vel(curr_particle,curr_event,1:3) = 0.d0
					vel(curr_particle,curr_event,photoel_direction%axis) = vel_mod
				else
					vel(curr_particle,curr_event,1:3) = 0.d0
					vel(curr_particle,curr_event,photoel_direction%axis) = -vel_mod
			endif
		else if (photoel_direction%direction .eq. 3)	then						! anisotropic photoelectron distribution
			vel(curr_particle,curr_event,:) = vel_mod*anisotropic_photoel_direction(photoel_direction%axis)
		else
			stop "Unknown photon polarization"
		endif
	else	! electrons with energy below the threshold stop immediately after creation
		vel(curr_particle,curr_event,:) = 0.d0
	endif
	
	Ee(curr_particle,curr_event) = E2(curr_event)					! saving the energy of each electron to plot later
			
	
	if (mheff.gt.0.d0 .and. Eh(curr_event).gt.0.d0) then	! a valence hole is created at this event!
		N_vh(curr_event) =  N_vh(curr_event) + 1
		rh(N_vh(curr_event),curr_event,:) = r_event(curr_event,:)	! its coordinate
		vel_mod = dsqrt(2.d0/(mheff*me)*Eh(curr_event)*e)	! its speed
		velh(N_vh(curr_event),curr_event,:) = vel_mod*random_direction()	! its velocity
		t_nexth(N_vh(curr_event)) = time_of_event(curr_event) + next_scattering_time_hole(tfp_h)
	endif


end subroutine start_with_photon



subroutine start_with_electron(nmc, n_events, Ne_CB, curr_event, Ne, curr_particle, r_event0, r_event, r, E_threshold, E2, E_i, E_f, vel, time_of_event, mheff, Eh, N_vh, rh, velh, t_nexth, Ee, tfp_h, scattering_angles,nop)
use DefaultParameters

implicit none 

	integer, intent(in) :: nmc
	integer, intent(in) :: n_events
	integer, intent(in) :: Ne_CB(n_events)
	integer(8), intent(in) :: curr_event
	integer(8), intent(in) :: Ne
	integer(8), intent(in) :: curr_particle
	real(8), intent(in) :: r_event0(3)	
	real(8), intent(inout) :: r_event(n_events,3)
	real(8), intent(inout) :: r(Ne,n_events,3)
	
	real(8), intent(in) :: E_threshold
	real(8), intent(inout) :: E2(n_events)
	real(8), intent(in) :: E_i(n_events)
	real(8), intent(in) :: E_f(n_events)
	integer, intent(inout) :: nop(n_events)
	real(8), intent(inout) :: Ee(Ne,n_events)
	real(8), intent(inout) :: vel(Ne,n_events,3)
	integer, intent(in) :: scattering_angles

	real(8), intent(in) :: time_of_event(n_events)
	real(8), intent(in) :: mheff
	real(8), intent(in) :: Eh(n_events)
	integer, intent(inout) :: N_vh(n_events)
	real(8), intent(inout) :: rh(Ne+Ne/2,n_events,3)
	real(8), intent(inout) :: velh(Ne+Ne/2,n_events,3)
	real(8), intent(inout) :: t_nexth(Ne+Ne/2)
	real(8), intent(in) :: tfp_h

	real(8) v_e0(3), v_e0_mod, vel_mod
	
	! coordinates
	r_event(curr_event,:) = r_event0(:)	! using the initial place of the cascade
	r(curr_particle,curr_event,:) = r_event(curr_event,:)		! the incident electron first collides at place r_event0
	r(Ne_CB(curr_event),curr_event,:) = r_event(curr_event,:)	! the newly created electron is at the same place now
	
	! initial velocity of the incident electron
	if (E_i(curr_event) .gt. E_threshold) then
		v_e0_mod = dsqrt(2.d0/me*E_i(curr_event)*e)	! E = mv^2/2		! getting the speed of the incident electron before the first collision
		v_e0 = random_direction()									! velocity direction of the incident electron (the incident electron was flying in random direction before the beginning of time...)
	else 
		v_e0_mod = 0.d0
		v_e0(:) = 0.d0
	endif

			
	! final velocity of the incident electron
	if (E_f(curr_event) .gt. E_threshold) then
		vel_mod = dsqrt(2.d0/me*E_f(curr_event)*e)	! E = mv^2/2		! using the final speed of the incident electron after the collision
		if (scattering_angles.eq.0) then	! direct scattering
			vel(curr_particle,curr_event,:) = vel_mod * v_e0(:)
		else if (scattering_angles.eq.1) then	! isotropic scattering
			vel(curr_particle,curr_event,:) = vel_mod*random_direction()
		else if (scattering_angles.eq.2) then
			! new velocity of the incident electron according to the anisotropic scattering
			vel(curr_particle,curr_event,:) = vel_mod*scattering_direction(E_f(curr_event),v_e0(:))
		endif
	else
		vel(curr_particle,curr_event,:) = 0.d0
	endif

	
	! velocity of the secondary electron
	if (E2(curr_event) .gt. E_threshold) then
		vel_mod = dsqrt(2.d0/me*E2(curr_event)*e)	! E = mv^2/2		! using the speed of the secondary electron
		vel(Ne_CB(curr_event),curr_event,:) = v_e0_mod * v_e0(:) - vel(nop(curr_event),curr_event,:)	! it flies in the direction taking into account the momentum conservation
		vel(Ne_CB(curr_event),curr_event,:) = vel(Ne_CB(curr_event),curr_event,:) / dsqrt &
					( dot_product(vel(Ne_CB(curr_event),curr_event,:),vel(Ne_CB(curr_event),curr_event,:)) )	! unit vector in the right direction
		vel(Ne_CB(curr_event),curr_event,:)  = vel_mod * vel(Ne_CB(curr_event),curr_event,:)		! vector rescaled according to the final velocity
	else
		vel(Ne_CB(curr_event),curr_event,:) = 0.d0
	endif
	
	Ee(curr_particle,curr_event) = E_f(curr_event)				! saving the energy of each electron to plot later
	Ee(Ne_CB(curr_event),curr_event) = E2(curr_event)			! saving the energy of each electron to plot later
	
	! velocity of the valence hole
	if (mheff.gt.0.d0 .and. Eh(curr_event).gt.0.d0) then	! a valence hole is created at this event!
		N_vh(curr_event) =  N_vh(curr_event) + 1
		rh(N_vh(curr_event),curr_event,:) = r_event(curr_event,:)	! its coordinate
		vel_mod = dsqrt(2.d0/(mheff*me)*Eh(curr_event)*e)	! its speed
		velh(N_vh(curr_event),curr_event,:) = vel_mod*random_direction()	! its velocity
		t_nexth(N_vh(curr_event)) = time_of_event(curr_event) + next_scattering_time_hole(tfp_h)
	endif


end subroutine start_with_electron


function anisotropic_photoel_direction(axis) ! generate photoelectron direction vector according to cosine-square distribution; not finished
	real(8), dimension(3) :: anisotropic_photoel_direction
	integer, intent(in) :: axis	! 1=X, 2=Y, or 3=Z axis for the most probable electron direction 
	real(8), dimension(3) :: ran_num	! random number; todo: use random.org
	real(8) cos_xi	! the angle cosine
	
	real(8), dimension(3) :: v_init
	
	v_init = (/ 0.d0, 0.d0, 1.d0 /)
	cos_xi = 1.d0	! cosine of the scattering angle (angle is between 0 and Pi)
	
	anisotropic_photoel_direction = cos_direction(v_init,cos_xi)	! unit vector such that angle(v_inc, scattering_direction) = xi

end function anisotropic_photoel_direction


function scattering_direction(Ekin,v_inc) ! generates a unit vector giving the direction of scattered electron, based on the scattering angle
! Phys. Rev. E 65, 037402, formula (7) is used for cos_xi

	real(8), dimension(3) :: scattering_direction

	Real(8), intent(in) :: Ekin	! kinetic energy of the electron
	Real(8), intent(in), dimension(3) :: v_inc	! velocity of the incident electron

	real(8) :: ran_num, dzeta1, dzeta2	! random numbers for cos_xi and for phi; todo: use random.org

	real(8) E0, eps
	real(8) dzeta
	real(8) cos_xi	! the scattering angle cosine
	
	E0 = 27.21d0	! atomic unit of energy (eV)
	eps = Ekin/E0	! dimensionless energy
	dzeta = 4.d0*eps/(1.d0+4.d0*eps)		! conventional screened-Coulomb interaction, formula below (7)

	call random_number(ran_num)
	cos_xi = 1.d0 - (2.d0*ran_num*(1.d0-dzeta)) / (1.d0+dzeta*(1.d0-2.d0*ran_num))	! cosine of the scattering angle
	
	scattering_direction = cos_direction(v_inc,cos_xi)	! unit vector such that angle(v_inc, scattering_direction) = xi


end function scattering_direction




! J.F. Briesmeister, MCNP-A General Monte Carlo N-Partcle Transport Code Version 5 Volume I (2003) is used for the vector rotation formulas, page 2-38; https://nucleonica.com/wiki/images/8/89/MCNPvolI.pdf
function cos_direction(init_vect,cos_xi) ! generates a random unit vector which has angle xi to the given unit vector
	real(8), dimension(3) :: cos_direction 			 ! final unit vector
	real(8), dimension(3), intent(in) :: init_vect  		 ! initial unit vector
	real(8), intent(in) :: cos_xi						 ! the scattering angle cosine
	real(8) :: dzeta1								 ! random numbers for cos_xi and for phi; todo: use random.org
	real(8) :: alpha, beta					 		 ! applying Rodrigues' rotation formula
	real(8), dimension(3) :: h1, h2, z				 ! applying Rodrigues' rotation formula
	
	! METHOD 2: https://stackoverflow.com/questions/25039284/random-unit-vector-in-2-dimension
	! create a random vector on a circle:
	call random_number(dzeta1)
	dzeta1 = dzeta1*2.d0*Pi	! (0,2Pi)
	alpha = dcos(dzeta1)
	beta = dsin(dzeta1)
	! done: (alpha,beta)

	if (1.d0-init_vect(3)**2 .gt. 1.d-4) then
		h1 = 1.d0/dsqrt(1.d0-init_vect(3)**2) * (/ -init_vect(2), init_vect(1), 0.d0 /)
		h2 = 1.d0/dsqrt(1.d0-init_vect(3)**2) * (/ init_vect(1)*init_vect(3), init_vect(2)*init_vect(3), -(1.d0-init_vect(3)**2) /)
	else
		h1 = 1.d0/dsqrt(1.d0-init_vect(1)**2) * (/ -init_vect(2), init_vect(3), 0.d0 /)
		h2 = 1.d0/dsqrt(1.d0-init_vect(1)**2) * (/ -(1.d0-init_vect(1)**2), init_vect(2)*init_vect(1),  init_vect(3)*init_vect(1) /)
	endif

	z = alpha*h1 + beta*h2

	cos_direction = cos_xi*init_vect + dsqrt(1.d0-cos_xi*cos_xi) * z 	! Rodrigues' rotation formula

end function cos_direction




function next_scattering_time_hole(fft) 

	real(8) :: next_scattering_time_hole		! [fs] the time until the next collision, i.e., free flight time
	real(8), intent(in) :: fft		! [fs] the time until the next collision, i.e., free flight time

	real(8) rn

	if (fft .gt. 1.d-40) then
		call random_number(RN)
		next_scattering_time_hole = - fft * log(RN)		! n*lambda*sigma_tot = 1
	else
		next_scattering_time_hole = 1.d99		! the inelastic mean free path is infinite when the cross section is zero
	endif
	
end function next_scattering_time_hole



function random_direction_old() ! generate radnomly directed unit vector using rejection sampling method
! George Marsaglia, Choosing a point from the surface of a sphere
! The Annals of Mathematical Statistics 1972, Vol. 43, No. 2, 645-646
! (Method 1)

	real(8), dimension(3) :: random_direction_old

	real(8), dimension(3) :: ran_num	! random number; todo: use random.org

	real(8) radius

	radius = 2.d0

	do while (radius .gt. 1.d0)

		call random_number(ran_num(1))
		call random_number(ran_num(2))
		call random_number(ran_num(3))

		ran_num(:) = 2.d0*(ran_num(:)-0.5d0)	! now the three numbers are in the range (-1,1)

		random_direction_old(:) = ran_num(:) / dsqrt(ran_num(1)**2+ran_num(2)**2+ran_num(3)**2)
		radius = (random_direction_old(1)**2+random_direction_old(2)**2+random_direction_old(3)**2)

	enddo

	random_direction_old(:) = random_direction_old(:)/radius

end function random_direction_old



function random_direction() ! generate radnomly directed unit vector using rejection sampling method
! George Marsaglia, Choosing a point from the surface of a sphere
! The Annals of Mathematical Statistics 1972, Vol. 43, No. 2, 645-646
! (The new method)

!One can also use GNU Scientific Library to generate a unit vector:
	!~ Function: void gsl_ran_dir_3d (const gsl_rng * r, double * x, double * y, double * z)
	!~ This function returns a random direction vector v = (x,y,z) in three dimensions. The vector is normalized such that |v|^2 = x^2 + y^2 + z^2 = 1. The method employed is due to Robert E. Knop (CACM 13, 326 (1970)), and explained in Knuth, v2, 3rd ed, p136. It uses the surprising fact that the distribution projected along any axis is actually uniform (this is only true for 3 dimensions). 

	real(8), dimension(3) :: random_direction
	real(8), dimension(2) :: ran_num	! random number; todo: use random.org
	real(8) S
	S = 2.d0

	do while (S .ge. 1.d0)

		call random_number(ran_num(1))
		call random_number(ran_num(2))

		ran_num(:) = 2.d0*(ran_num(:)-0.5d0)	! now the two numbers are in the range (-1,1)

		S = (ran_num(1)**2+ran_num(2)**2)

	enddo

	random_direction(1) = 2.d0 * ran_num(1) * dsqrt(1.d0 - S)
	random_direction(2) = 2.d0 * ran_num(2) * dsqrt(1.d0 - S) 
	random_direction(3) = 1.d0 - 2.d0*S

end function random_direction



end module Monte_Carlo
