!    XCASCADE-3D software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE-3D software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE-3D is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE-3D is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE-3D. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains Monte-Carlo subroutines related to photon absorption
MODULE Laser_shape

use DefaultParameters	! object oriented stuff 8-)
use Constants

implicit none

contains


!~ function where_photons_absrorbed()
!~ integer, intent(in) :: axis	! 1=x, 2=y, 3=z



!~ end function where_photons_absrorbed


 function Photon_arrival_place(KOP, mu, sigma, Nspace, dx_glob, failed, beam)
 ! Function generates positions of photon absorption randomlyd according to chosen probability distribution
   real(8) Photon_arrival_place 		! [m] arrival place of the photon
   integer, intent(in) :: KOP		! which kind of pulse shape we use
   real(8), intent(in) :: mu, sigma  	! place of pulse max and pulse size
   integer(8), intent(in) :: Nspace
   real(8), intent(in) :: dx_glob
   integer, intent(inout), optional :: failed
   type(beam_struct), intent (IN), optional :: beam	

   real(8) rnd1, rnd2, rnd3
   real(8) coord, dx, Int_pulse, last_x, this_x, min_coord, max_coord
   real(8) integration_interval
   real(8) max_length
   integer i, j
   real(8) :: intvl_mult = 3.d0
   
   if (present(failed)) failed = 0
   
   max_length = dx_glob*Nspace
   Int_pulse = 0.0d0
   last_x = 0.0d0

   call random_number(rnd1)
   call random_number(rnd3)
   
   if (sigma .le. max_length .or. kop.eq.3) then
		dx = 0.25d0*dx_glob		! not sure if this is right...
		if (present(beam) .and. kop.eq.3) dx = 0.25d-6		! not sure if this is right...
		integration_interval = intvl_mult*sigma
		min_coord = mu - integration_interval
		max_coord = mu + integration_interval
	   
		   if (KOP .eq. 2) then
			   min_coord = mu
		   endif
		   
		   if (KOP .eq. 3) then 
			! across:
				min_coord = -67.75d-6
				max_coord = 60.d-6
			! along:
				!~ min_coord = -70.50d-6
				!~ max_coord = 57.25d-6
		   endif
		   
		   coord = min_coord
		   Photon_arrival_place = min_coord
		   
		   
		   do while (Int_pulse .LT. rnd1)
		      if (.not. present(beam)) this_x =	Photon_absorption_shape(KOP, mu, sigma, integration_interval, coord)
		      if (present(beam)) this_x = 		Photon_absorption_shape(KOP, mu, sigma, integration_interval, coord, beam)
		      Int_pulse = Int_pulse + dx*(this_x + last_x)*0.5d0 ! this line requires that the Photon_absorption_shape function is normalized to one
		      !~ if (rnd1.gt.0.95) print *, Int_pulse, rnd1, coord*1.d6,this_x*1.d6, dx*1.d6
		      !~ print *, int_pulse, rnd1, this_x, coord
		      last_x = this_x
		      if (coord .GT. max_coord) then ! just for case if numerical integration isn't perfect:
					!~ call random_number(rnd2)
					!~ Photon_arrival_place = mu - integration_interval + intvl_mult*integration_interval*rnd2		! Nikita's original code
					Photon_arrival_place = 1.d10		! Vladimir'r first suggestion
					if (present(failed)) failed = 1
					exit
		      endif

		      Photon_arrival_place = coord
		      coord = coord + dx
		   enddo

		   Photon_arrival_place = Photon_arrival_place - 0.5d0*dx + dx*rnd3
!~ if (rnd1.gt.0.95) stop
   else
		!~ dx = dx_glob!*0.01d0
		!~ integration_interval =  max_length
		!~ min_coord = mu - integration_interval
		!~ max_coord = mu + integration_interval
		Photon_arrival_place = 2.d0*(rnd1-0.5d0)*Nspace*dx_glob ! for large sigma and small computational box the absorption places appear random
   endif

end function Photon_arrival_place


function Photon_absorption_shape(KOP, mu, sigma, integration_interval, x, beam)
   REAL(8) Photon_absorption_shape
   integer, intent(in) :: KOP
   real(8), intent(in), optional :: mu, sigma, integration_interval, x
   type(beam_struct), intent (IN), optional :: beam	
   !if (.not.present(x) .and. .not. KOP.eq.3) stop "wrong call to Photon_absorption_shape()"
   select case (KOP)
	case(:0)
		Photon_absorption_shape = Rect_pulse(mu, sigma, integration_interval, x)
	case(1)
		Photon_absorption_shape = Gaus_pulse(mu, sigma, integration_interval, x)
	case(2)
		!~ if (integration_interval .le. sigma) then
			Photon_absorption_shape = LambertBeerLaw(mu, sigma, integration_interval, x)
		!~ else
			!~ Photon_absorption_shape = Rect_pulse(mu, sigma, integration_interval, x)
		!~ endif
	case(3)
		if (present(beam)) Photon_absorption_shape = knife_edge_shadow1d(beam,x)
		if (.not. present(beam)) Photon_absorption_shape = 1.d100
	case default
		print *, 'something is wrong with kind of pulse (KOP)'
   end select
end function Photon_absorption_shape 


pure function LambertBeerLaw(mu, sigma, integration_interval, x) ! number of photons at this coordinate x, according to  Lambert Beer Law of constant intensity
REAL(8) LambertBeerLaw
real(8), intent(in) :: mu, sigma, integration_interval, x
	if ((x-mu .ge. 0.d0) .AND. (x-mu .LE. integration_interval) ) then
		LambertBeerLaw = dexp(-(x-mu)/sigma) / sigma
	else
		LambertBeerLaw = 0.0d0
	endif
end function LambertBeerLaw


pure function Rect_pulse(mu, sigma, integration_interval, x) ! number of photons at this coordinate x, according to rectangular pulse of constant intensity
real(8) Rect_pulse
real(8), intent(in) :: mu, sigma, integration_interval, x
real(8) Gaus_sigma
	Gaus_sigma = sigma*2.35482d0
	if ((x .GT. (mu - Gaus_sigma*0.5d0)) .AND. (x .LE. (mu + Gaus_sigma*0.5d0)) ) then
	Rect_pulse = 1.0d0/Gaus_sigma
	else
		Rect_pulse = 0.0d0
	endif
end function Rect_pulse


pure function Gaus_pulse(mu, sigma, integration_interval, x) ! number of photons at the coordinate x according to Gaussian shape
real(8) Gaus_pulse
real(8), intent(in) :: mu, sigma, integration_interval, x
real(8), parameter :: g_Pi = 3.1415926535897932384626433832795d0
	if ( (x-mu .GE.  -integration_interval) .AND. (x-mu .LE. integration_interval) ) then
		Gaus_pulse = 1.0d0/(sigma*dsqrt(2.0d0*g_Pi))*dexp(-(x-mu)*(x-mu)/(2.d0*sigma*sigma))
	else
		Gaus_pulse = 0.0d0
	endif
end function Gaus_pulse


function knife_edge_shadow1d(beam,x) ! number of photons at the coordinate x according to Gaussian shape; not finished
real(8) knife_edge_shadow1d
type(beam_struct), intent (IN) :: beam	
real(8), intent(in) :: x
integer :: ind

	!~ ind = nint((x/.25d-6)+283)	! along
	ind = nint((x/.25d-6)+272)	! across
	if (ind .ge. 1 .and. ind .le.512) then
		knife_edge_shadow1d = beam%spot_1d(ind)
		!~ print *, ind, knife_edge_shadow1d
	else
		knife_edge_shadow1d = 0
	endif
	
end function knife_edge_shadow1d



end module Laser_shape
