!    XCASCADE-3D software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE-3D software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE-3D is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE-3D is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE-3D. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This module contains objects needed to read input3d.txt and set default input parameters. The reading of input3d.txt is implemented using Fortran namelist
module DefaultParameters
	implicit none
	
	type photoel_direction_struct
		integer :: direction = 2
		integer :: axis = 3
	end type

	type physical_parameters_struct
		type(photoel_direction_struct) photoel_direction
		real(8) :: E_threshold = 10.d0			! at which energy we stop the electrons [eV]; should correspond to the smallest ionization energy of the atoms
		logical :: inclHighEnEl = .true.			! whether to include contribution of high-energy electrons into all energy plots (they are always included into particle plots)
		integer :: scattering_angles = 2
	end type
	
	type switches_electrons_struct
		integer :: Z_dependence = 1			! 0 = do not prepare the files, 1 = prepare tz versions, 2 = prepare zt versions, 3 = both -- electron, hole densities and energies depending on Z
		integer :: cylindrical = 1				! 0 = do not prepare the file , 1 = prepare and write -- electron density and energy depending on cylindrical radius
		integer :: spherical = 1					! 0 = do not prepare the file , 1 = prepare and write -- electron density and energy depending on spherical radius	
		integer :: make_3d = 0					! 0 = do not prepare the file , 1 = prepare and write -- electron density and energy depending on X and Y at certain time moments
	end type	
	
	type switches_events_struct
		integer :: XTANT_events = 2				! 0 = do not prepare the file , 2 = prepare zt version -- output file for XTANT containing Z dependence of the events
		integer :: make_1d = 1					! 0 = do not prepare the files, 1 = prepare and write; spherical dependence of the events
		integer :: make_2d = 0					! 0 = do not prepare the files, 1 = prepare and write (may be time consuming)
		integer :: make_3d = 0					! 0 = do not prepare the files, 1 = prepare and write (very slow and memory hungry!)
	end type

	type switches_flyholes_struct
		integer :: Z_fholes = 0					! 0 = do not prepare the files, 1 = prepare tz versions, 2 = prepare zt versions, 3 = both -- density and energy of elastically scattering holes depending on Z
		integer :: cylindrical = 0				! 0 = do not prepare the files, 1 = prepare and write -- density and energy of elastically scattering holes depending on cylindrical radius at the end of the simulation
		integer :: make_2d = 0					! 0 = do not prepare the files, 1 = prepare and write (may be time consuming) -- density and energy of elastically scattering holes depending on x,y
		integer :: make_3d = 0					! 0 = do not prepare the files, 1 = prepare and write (very slow and memory hungry!) -- density and energy of elastically scattering holes depending on x,y,z
	end type

	type io_switches_struct
		integer :: rewrite_cascades = 0			! whether to add spatial information to files cascades/OUTPUT*.dat; useful for debugging
		type(switches_electrons_struct) electrons      ! output switches related to electrons
		type(switches_events_struct) events            	! output switches related to events
		type(switches_flyholes_struct) flyholes        	! output switches related to flying holes
		real(8) :: make3d_time = -1.d99			! at which moment of time, fs, to write 3d outputs (default = numeric%grid%time%maxt, see Read_input.f90)end type
		integer :: make3d_time_index = -1		! at which time index to write 3d outputs (default = numeric%grid%time%Ntime, see Read_input.f90)
	end type

	type temporal_tgrid_struct
		real(8) :: mint = -1.d0					! starting time in fs; should be lower for longer laser durations; should correspond to XCASCADE value
		real(8) :: maxt = 500.d0				! ending time in fs; should be larger for longer cascade durations; should correspond to XCASCADE value
		real(8) :: dt = 0.5d0					! time step in fs, used for plots only, not for the actual calculations
		integer :: Ntime = DINT((500.d0+1.d0)/0.5d0)
	end type
	
	type spatial_rgrid_struct
		real(8) :: maxX = 250.d0				! X size of the plotting box in nm, from -(this value) to +(this value) (not the computational box). If a particle is further than that, it is not contributing to the plots
		real(8) :: maxY = 250.d0				! Y size of the plotting box in nm, from -(this value) to +(this value) (not the computational box). If a particle is further than that, it is not contributing to the plots
		real(8) :: maxZ = 250.d0				! Z size of the plotting box in nm, from -(this value) to +(this value) (not the computational box). If a particle is further than that, it is not contributing to the plots
		real(8) :: dx = 1.d0					! X space grid step in nm, used for plots only, not for the actual calculations
		real(8) :: dy = 1.d0					! Y space grid step in nm, used for plots only, not for the actual calculations
		real(8) :: dz = 1.d0					! Z space grid step in nm, used for plots only, not for the actual calculations
		integer(8) :: NX = 250.d0 / 1.d0			! half of number of grid points in X
		integer(8) :: NY = 250.d0 / 1.d0 			! half of number of grid points in Y
		integer(8) :: NZ = 250.d0 / 1.d0			! half of number of grid points in Z 
	end type
	
	type rtgrids_struct
		type(spatial_rgrid_struct) space
		type(temporal_tgrid_struct) time
	end type
	
	type numeric_struct
		integer :: omp_numthr = 1
		type(rtgrids_struct) grid
	end type

	type spot_struct
		real(8) :: x = 200.d0  					! these are the beam parameters which were implemented for the first time
		real(8) :: y = 300.d0  
		real(8) :: z = 675.44d3
	end type

	type beamnumeric_struct
		integer :: omp_threads = -1				! number of OMP threads in Monte Carlo convolution (-1 means same as omp_numthr)
		integer :: mc_stat = 1000				! number of Monte Carlo repetitions in the Monte Carlo convolution (by default equal to numeric%omp_numthr)
		integer :: stat_multiplier = 2			! Monte Carlo statistics for per-cascade convolution (multiplied later by the number of photons in the current volume, beam%Nphotons*V)
	end type

	type beam_struct
		real(8) :: Nphotons = 1.38d6			! total number of photons per cubic micrometer; should correspond to the experimental fluence
		type(spot_struct) spot
		real(8) :: spot_1d(512)
		type(beamnumeric_struct) numeric
		integer :: show_laser_shape = 0			! instead of convolving the hole distribution, convolve single point, thus showing just the laser pulse shape
	end type
	
end module DefaultParameters
