!    XCASCADE-3D software implements Monte Carlo approach to model electron cascades in solids induced by X-ray impact or by an impact of high-energy electrons
!    Copyright (C) 2025 Deutsches Elektronen-Synchrotron DESY, a research centre of the Helmholtz Association.

!    This file is part of XCASCADE-3D software.

!    Authors:
!    Vladimir Lipp <vladimir.lipp@desy.de> (DESY)
!    Nikita Medvedev <medvedev@ipp.cas.cz> (DESY)
!    Beata Ziaja <ziaja@mail.desy.de> (DESY & IFJ)

!    SPDX-FileCopyrightText: 2025 Deutsches Elektronen-Synchrotron DESY
!    SPDX-License-Identifier: AGPL-3.0-only

!    XCASCADE-3D is free software: you can redistribute it and/or modify it under the terms of the Affero GNU General Public License version 3 only, as published by the Free Software Foundation.
!    XCASCADE-3D is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the Affero GNU General Public License version 3 for more details.
!    You should have received a copy of the Affero GNU General Public License version 3 along with XCASCADE-3D. If not, see <https://www.gnu.org/licenses/>.

!    For more information about this software, see https://doi.org/10.5281/zenodo.8204314 and https://xm.cfel.de/research/scientific_software/xcascade_amp_xcascade_3d/.

! This is the main program file

!#define OMP_inside
PROGRAM XCascade3D
! Initiate modules:
! module-name:            		| file where it's located:
use DefaultParameters
use Constants              		! Constants.f90
use Objects				! Objects.f90
use Read_input	   			! Read_input.f90
use Write_output   			! Write_output.f90
use Monte_Carlo		   	! Monte_Carlo.f90
implicit none

type(numeric_struct) numeric
type(physical_parameters_struct) physical
type(rtgrids_struct) grid
type(io_switches_struct) switch
type(beam_struct) beam

type(event_type), allocatable :: event(:)

! For OPENMP:
integer my_id, OMP_GET_THREAD_NUM, OMP_GET_NUM_THREADS

! calculation time
real(8) as1
integer ctim(8), c1(8)

! Input parameters:
integer :: NoEiC ! number of elements in the compound 
real(8) :: Dens  ! density [1/cm^3]

integer :: kop_inc	! kind of particle impacting: photon vs electron
real(8) :: Eph	! [eV] photon energy
real(8) :: Fluence	! [J/cm^2] photon fluence
real(8) :: mheff  ! effective mass of valence holes in the units of m_e
real(8) :: tfp_h	! time of free path of holes
real(8) :: Total_t
integer :: NMC

! Known data about a cascade
integer n_events						! number of events in a certain cascade
integer, ALLOCATABLE :: koe(:)			! kind of event = photon/elastic electron/inelastic electron/hole
integer, ALLOCATABLE :: nop(:)			! number of acting particle
real(8), ALLOCATABLE ::  time_of_event(:)			! time of a particular curr_event
real(8), ALLOCATABLE ::  E_i(:), E_f(:), E2(:), Eh(:), Ech(:)	! [eV] initial energy of particle, final energy of particle after an curr_event, kinetic energy of holes, energy stored in core shells
Integer, ALLOCATABLE ::  Ne_CB(:)			! number of the excited electrons during certain curr_event = number of carriers in conduction band after the curr_event
Integer, ALLOCATABLE ::  N_vh(:)			! number of the valence holes  
real(8) :: r_event0(3)	! coordinates of the only absorbed photon
real(8) :: v_e0(3)		! initial velocity direction of the incident electron (for the case of incident electron)
real(8) :: v_e0_mod	! initial speed of the incident electron (for the case of incident electron)

Real(8) :: dt,dx,dy,dz 	! time and space grid
Real(8) :: min_time,max_time,maxX,maxY,maxZ 	! latest time and space point to write
integer :: include_holes
real(8) :: Ekin
logical :: last_event_done	! whether we took into account the last event when creating time grids

! What we want to find
real(8), allocatable :: r_event(:,:)	! coordinates of each event
real(8), allocatable :: r(:,:,:)						! 3 coordinates of each electron at each event
real(8), allocatable :: rh(:,:,:)						! 3 coordinates of each valence hole at each event
real(8), allocatable :: r_timegrid(:,:,:)		! 3 coordinates of each electron at each time-grid moment
real(8), allocatable :: rh_timegrid(:,:,:)		! 3 coordinates of each valence hole at each time-grid moment
real(8), allocatable :: t_nexth(:)				! for each hole, when it will scatter next time
real(8), allocatable :: vel(:,:,:), vel_timegrid(:,:,:)	! velocities of each electron at each event and at each contour time moment
real(8), allocatable :: velh(:,:,:), velh_timegrid(:,:,:)				! velocities of each electron at each event and at each contour time moment
real(8), allocatable :: Ee(:,:), E_timegrid(:,:)	! energy of each electron at each event and at each contour time moment
Real(8), allocatable :: cur_Etot(:),cur_Etot_timegrid(:),Etot_timegrid(:)					
real(8), allocatable :: cur_Ech_timegrid(:),Ech_timegrid(:)	! energies of core holes at each event and at each contour time moment

integer(8) Ne_tot   ! average number of electrons in a cascade
integer(8), allocatable :: ne_avg_t(:)		! number of excited electrons depending on time

real(8), allocatable :: cur_cylind_Ne(:,:), avg_cylind_Ne(:,:) 	! current (in one cascade) and average number of electrons depending on time and distance to the photon/electron
real(8), allocatable :: cur_cylind_Ee2(:,:), avg_cylind_Ee2(:,:)	! current (in one cascade) and average energies of electrons depending on time and distance to the photon; calculated via velocities
real(8), allocatable :: cur_cylind_Ee(:,:), avg_cylind_Ee(:,:)		! current (in one cascade) and average energies of electrons depending on time and distance to the photon/electron

real(8), allocatable :: cur_spherical_Ne(:,:), avg_spherical_Ne(:,:)	! current (in one cascade) and average number of electrons depending on time and distance to the photon/electron
real(8), allocatable :: cur_spherical_Ee2(:,:), avg_spherical_Ee2(:,:)	! current (in one cascade) energies of electrons depending on time and distance to the photon; calculated via velocities
real(8), allocatable :: cur_spherical_Ee(:,:), avg_spherical_Ee(:,:)	! current (in one cascade) and average energies of electrons depending on time and distance to the photon/electron

real(8), allocatable :: cur_spherical_Eh(:,:), avg_spherical_Eh(:,:)	! current (in one cascade) and average kinetic energies of holes depending on time and distance to the photon/electron
real(8), allocatable :: cur_cyl_E123h(:,:), avg_cyl_E123h(:,:)	! current (in one cascade) and average kinetic energies of holes depending on time and distance to the photon/electron
real(8), allocatable :: cur_N123fh_3d(:,:,:,:), avg_N123fh_3d(:,:,:,:)	! current (in one cascade) and average kinetic energies of holes depending on time and distance to the photon/electron
real(8), allocatable :: cur_N123fh_3d_partconv(:,:,:,:), avg_N123fh_3d_partconv(:,:,:,:)	! current (in one cascade) and average kinetic energies of holes depending on time and distance to the photon/electron
real(8), allocatable :: cur_N123fh_3d_cleverconv(:,:,:), avg_N123fh_3d_cleverconv(:,:,:)	! current (in one cascade) and average kinetic energies of holes depending on time and distance to the photon/electron -- convolved

real(8), allocatable :: cur_Z_Ne(:,:), avg_Z_Ne(:,:)		
real(8), allocatable :: cur_Z_Ee2(:,:), avg_Z_Ee2(:,:)	
real(8), allocatable :: cur_Z_Ee(:,:), avg_Z_Ee(:,:)	
real(8), allocatable :: cur_3d_Ne(:,:,:), avg_3d_Ne(:,:,:)					! current (in one cascade) and average electron density depending on x,y,z
real(8), allocatable :: cur_3d_Ne_cleverconv(:,:,:), avg_3d_Ne_cleverconv(:,:,:)		! current (in one cascade) and average electron density depending on x,y,z -- convolved

real(8), allocatable :: cur_Z_Nh(:,:), avg_Z_Nh(:,:)	
real(8), allocatable :: cur_Z_Eh(:,:), avg_Z_Eh(:,:)	

real(8), allocatable :: cur_cyl_Nfh(:,:,:), avg_cyl_Nfh(:,:,:)	! fh = flying holes
real(8), allocatable :: cur_cyl_Efh(:,:,:), avg_cyl_Efh(:,:,:)	! fh = flying holes
real(8), allocatable :: cur_2d_Nfh(:,:,:), avg_2d_Nfh(:,:,:)	! fh = flying holes
real(8), allocatable :: cur_Z_Nfh(:,:), avg_Z_Nfh(:,:)		! fh = flying holes
real(8), allocatable :: cur_Z_Efh(:,:), avg_Z_Efh(:,:)		! fh = flying holes

real(8), allocatable :: curr_events_3d(:,:,:), avg_events_3d(:,:,:)	! how many events occured at certain volume in 3D in one cascade and on average
real(8), allocatable :: curr_events_2d(:,:), avg_events_2d(:,:)	! how many events occured at certain volume in 2D in one cascade and on average
real(8), allocatable :: curr_events_1d(:), avg_events_1d(:)		! how many events occured at certain volume in 2D in one cascade and on average, depending on a spherical coordinate
real(8), allocatable :: curr_Nevents(:,:), avg_Nevents(:,:)		! how many events occured at certain volume in z,t in one cascade and on average
real(8), allocatable :: curr_Eevents(:,:), avg_Eevents(:,:)		! how much energy is at certain volume in z,t in one cascade and on average
Real(8) :: curr_electron_range(3), avg_electron_range(3)				! how far the electrons fly before the thermalization in one cascade and on average. first one is the range of photoelectrons, second one is the range of fatherst electron, third is the range of initial Auger electrons 
integer noec_avg							! total number of elastic collisions in an average cascade

! Arrays to study the convergence
Real(8), allocatable :: Ne_tot_conv(:)   						! average number of electrons in a cascade
Real(8), allocatable :: noec_avg_conv(:)						! total number of elastic collisions in an average cascade
Real(8), allocatable :: electron_range_conv(:,:)					! how far the electrons fly before the thermalization on average

! Auxiliary variables
character(100) Output_file_name,File_name
integer(8) Ne   ! number of electrons
real(8) imin	! minimal ionization potential
integer(8) curr_particle	! current number of a particle
integer(8) curr_event	! current number of curr_event
integer curr_cascade	! current number of iteration
integer stati		! averaging over MC runs
real(8) vel_mod				! modulus of the velocity
real(8),dimension(3) :: vel_old

integer noec_tot			! total number of elastic collisions by this moment
integer, ALLOCATABLE :: noec(:)			! current number of elastic collisions at each moment
integer delta_noec_cur			! correction to the current number of elastic collisions, taking into account the case when additional elastic collision(s) happened between event nop(curr_event) and event nop(curr_event+noec(nop(curr_event)))
! for contour plots:
real(8) :: time_grid, space_grid
integer(8) time_index,space_index
integer(8) :: Ntime, NspaceX, NspaceY, NspaceZ	! number of grid points in time and space
real(8) :: dist2		! distance between two points, squared
integer Nvholes	! average number of valence holes
integer(8) :: NCascadesDone=0
real(8) :: Nphotons
real(8) :: phot_zspread
integer :: tot_photons,lost_photons, tot_lost_photons

integer(8) max_num_events	! maximum number of electrons in one cascade
max_num_events = 900000

!print*, 'For ifort, do not forget to increase the stack sizes'

print *, 'XCASCADE-3D version 1.0, https://doi.org/10.5281/zenodo.8204314'
print *, ''

call random_seed() ! standard FORTRAN seeding of random numbers
call date_and_time(values=c1) ! standard FORTRAN time and date
 ctim=c1	
write(*, 1005) ctim(5), ctim(6), ctim(7), ctim(3), ctim(2), ctim(1)
print*, '   '

!! initial value of some parameters
 Ne_tot = 0
 noec_avg = 0 
 tot_lost_photons = 0
 tot_photons = 0

call read_input_3d(numeric,physical,switch,beam)

!call read_fluence_1d(beam%spot_1d,physical%photoel_direction%axis)

!renaming variable for readability:
!associate( min_time => numeric%grid%time%mint) ?
min_time = numeric%grid%time%mint
max_time = numeric%grid%time%maxt
dt = numeric%grid%time%dt

maxX = numeric%grid%space%maxX
maxY = numeric%grid%space%maxY
maxZ = numeric%grid%space%maxZ
dx = numeric%grid%space%dx
dy = numeric%grid%space%dy
dz = numeric%grid%space%dz
NspaceX = numeric%grid%space%NX
NspaceY = numeric%grid%space%NY
NspaceZ = numeric%grid%space%NZ

Ntime = numeric%grid%time%Ntime

! Array allocations for future plotting:
if (.not. allocated(avg_cylind_Ne)) allocate(avg_cylind_Ne(Ntime,NspaceX))
if (.not. allocated(avg_cylind_Ee2)) allocate(avg_cylind_Ee2(Ntime,NspaceX))
if (.not. allocated(avg_cylind_Ee)) allocate(avg_cylind_Ee(Ntime,NspaceX))
avg_cylind_Ne(:,:) = 0.d0
avg_cylind_Ee(:,:) = 0.d0
avg_cylind_Ee2(:,:) = 0.d0

if (.not. allocated(avg_spherical_Ne)) allocate(avg_spherical_Ne(Ntime,NspaceX))
if (.not. allocated(avg_spherical_Ee)) allocate(avg_spherical_Ee(Ntime,NspaceX))
if (.not. allocated(avg_spherical_Ee2)) allocate(avg_spherical_Ee2(Ntime,NspaceX))
if (.not. allocated(avg_spherical_Eh)) allocate(avg_spherical_Eh(Ntime,NspaceX))
avg_spherical_Ne(:,:) = 0.d0
avg_spherical_Ee(:,:) = 0.d0
avg_spherical_Ee2(:,:) = 0.d0
avg_spherical_Eh(:,:) = 0.d0

if (switch%events%make_3d.eq.1) then
	if (.not. allocated(avg_events_3d)) allocate(avg_events_3d(-NspaceX:NspaceX,-NspaceY:NspaceY,-NspaceZ:NspaceZ))
else
	if (.not. allocated(avg_events_3d)) allocate(avg_events_3d(-1:1,-1:1,-1:1))	! I have to do it, because there is no simple way to make line "!$omp default(shared) private(..." dependent on parameters
endif
avg_events_3d(:,:,:) = 0.d0	

if (switch%events%make_2d.eq.1) then
	if (.not. allocated(avg_events_2d)) allocate(avg_events_2d(-NspaceX:NspaceX,-NspaceY:NspaceY))
else
	if (.not. allocated(avg_events_2d)) allocate(avg_events_2d(-1:1,-1:1))	! I have to do it, because there is no simple way to make line "!$omp default(shared) private(..." dependent on parameters
endif
avg_events_2d(:,:) = 0.d0

if (switch%events%make_1d.eq.1) then
	if (.not. allocated(avg_events_1d)) allocate(avg_events_1d(0:NspaceX))
else
	if (.not. allocated(avg_events_1d)) allocate(avg_events_1d(0:1))	! I have to do it, because there is no simple way to make line "!$omp default(shared) private(..." dependent on parameters
endif
avg_events_1d(:) = 0.d0
	
if (.not. allocated(avg_Nevents)) allocate(avg_Nevents(Ntime,-NspaceX:NspaceX))
if (.not. allocated(avg_Eevents)) allocate(avg_Eevents(Ntime,-NspaceX:NspaceX))
avg_Nevents(:,:) = 0
avg_Eevents(:,:) = 0

if (.not. allocated(avg_Z_Ne)) allocate(avg_Z_Ne(Ntime,-NspaceX:NspaceX))
if (.not. allocated(avg_Z_Ee)) allocate(avg_Z_Ee(Ntime,-NspaceX:NspaceX))
if (.not. allocated(avg_Z_Ee2)) allocate(avg_Z_Ee2(Ntime,-NspaceX:NspaceX))
if (.not. allocated(avg_Z_Nh)) allocate(avg_Z_Nh(Ntime,-NspaceX:NspaceX))
if (.not. allocated(avg_Z_Eh)) allocate(avg_Z_Eh(Ntime,-NspaceX:NspaceX))
avg_Z_Ne(:,:) = 0.d0
avg_Z_Ee(:,:) = 0.d0
avg_Z_Ee2(:,:) = 0.d0
avg_Z_Nh(:,:) = 0.d0
avg_Z_Eh(:,:) = 0.d0
if (switch%electrons%make_3d.gt.0) then
	if (.not. allocated(avg_3d_Ne)) allocate(avg_3d_Ne(-NspaceX:NspaceX,-NspaceY:NspaceY,-NspaceZ:NspaceZ))
	if (switch%electrons%make_3d.gt.1 .and. .not. allocated(avg_3d_Ne_cleverconv)) allocate(avg_3d_Ne_cleverconv(-NspaceX:NspaceX,-NspaceY:NspaceY,-NspaceZ:NspaceZ))
	if (switch%electrons%make_3d.eq.1 .and. .not. allocated(avg_3d_Ne_cleverconv)) allocate(avg_3d_Ne_cleverconv(-1:1,-1:1,-1:1))			! how to get rid of this array when it is not needed at all? I don't know... So I just make it tiny
else	! how to get rid of these arrays when they are not needed at all? I don't know... So I just make them tiny
	if (.not. allocated(avg_3d_Ne)) allocate(avg_3d_Ne(-1:1,-1:1,-1:1))
	if (.not. allocated(avg_3d_Ne_cleverconv)) allocate(avg_3d_Ne_cleverconv(-1:1,-1:1,-1:1))
endif
avg_3d_Ne(:,:,:) = 0.d0
avg_3d_Ne_cleverconv(:,:,:) = 0.d0

if (.not. allocated(avg_cyl_E123h)) allocate(avg_cyl_E123h(3,NspaceX))
avg_cyl_E123h(:,:) = 0.d0

if (switch%flyholes%make_3d.gt.0) then
        if (.not. allocated(avg_N123fh_3d)) allocate(avg_N123fh_3d(3,-NspaceX:NspaceX,-NspaceY:NspaceY,-NspaceZ:NspaceZ))
        if (.not. allocated(avg_N123fh_3d_partconv)) allocate(avg_N123fh_3d_partconv(3,-NspaceX:NspaceX,-NspaceY:NspaceY,-NspaceZ:NspaceZ))
        if (.not. allocated(avg_N123fh_3d_cleverconv)) allocate(avg_N123fh_3d_cleverconv(-NspaceX:NspaceX,-NspaceY:NspaceY,-NspaceZ:NspaceZ))
else	! how to get rid of this array when it is not needed at all? I don't know... So I just make it tiny
        if (.not. allocated(avg_N123fh_3d)) allocate(avg_N123fh_3d(3,-1:1,-1:1,-1:1))
        if (.not. allocated(avg_N123fh_3d_partconv)) allocate(avg_N123fh_3d_partconv(3,-1:1,-1:1,-1:1))
        if (.not. allocated(avg_N123fh_3d_cleverconv)) allocate(avg_N123fh_3d_cleverconv(-1:1,-1:1,-1:1))
endif
avg_N123fh_3d(1:3,:,:,:) = 0.d0
avg_N123fh_3d_partconv(1:3,:,:,:) = 0.d0
avg_N123fh_3d_cleverconv(:,:,:) = 0.d0

!if (switch%flyholes%cylindrical .gt. 0) then
	if (.not. allocated(avg_cyl_Nfh)) allocate(avg_cyl_Nfh(3,Ntime,NspaceX))
	if (.not. allocated(avg_cyl_Efh)) allocate(avg_cyl_Efh(3,Ntime,NspaceX))
avg_cyl_Nfh(:,:,:) = 0.d0
avg_cyl_Efh(:,:,:) = 0.d0

!if (switch%flyholes%Z_fholes .gt. 0) then
	if (.not. allocated(avg_Z_Nfh)) allocate(avg_Z_Nfh(Ntime,-NspaceX:NspaceX))
	if (.not. allocated(avg_Z_Efh)) allocate(avg_Z_Efh(Ntime,-NspaceX:NspaceX))
avg_Z_Nfh(:,:) = 0.d0
avg_Z_Efh(:,:) = 0.d0

if (switch%flyholes%make_2d .gt. 0) then
	if (.not. allocated(avg_2d_Nfh)) allocate(avg_2d_Nfh(3,-NspaceX:NspaceX,-NspaceY:NspaceY))
else
	if (.not. allocated(avg_2d_Nfh)) allocate(avg_2d_Nfh(3,-1:1,-1:1))
endif
avg_2d_Nfh(:,:,:) = 0.d0


avg_electron_range(1:3) = 0.d0
Nvholes = 0	! average number of valence holes in the cascades

nmc = Number_of_cascades()

! array allocation to read the cascades
if (.not. allocated(ne_avg_t)) allocate(ne_avg_t(max_num_events))
ne_avg_t(:) = 0

if (.not.allocated(Ne_tot_conv)) allocate(Ne_tot_conv(nmc))
if (.not.allocated(noec_avg_conv)) allocate(noec_avg_conv(nmc))
if (.not.allocated(electron_range_conv)) allocate(electron_range_conv(3,nmc))
if (.not.allocated(Etot_timegrid)) allocate(Etot_timegrid(Ntime))	
if (.not.allocated(Ech_timegrid)) allocate(Ech_timegrid(Ntime))	
Etot_timegrid(:)=0.d0
Ech_timegrid(:)=0.d0

! Print and check the current simulation conditions
call print_simulation_conditions(nmc, physical, switch, beam)

if (switch%flyholes%Z_fholes.gt.0 .or. switch%flyholes%make_2d.gt.0 .or. switch%flyholes%make_3d.gt.0) then
	call reading_data_from_XCASCADE(fluence, mheff, tfp_h)
	tfp_h = tfp_h*1.d15 ! [s -> fs]
else
	fluence = 1.d0
	mheff = 1.d99
	tfp_h = 0.d0
endif

call OMP_SET_DYNAMIC(0)      ! standard openmp subroutine
call OMP_SET_NUM_THREADS(numeric%omp_numthr) ! start using threads with openmp: Num_th is the number of threads, defined in the input file

!$omp parallel &
!$omp default(shared) private(curr_cascade,time_index,space_index,koe,nop,imin,time_of_event,E_i,E_f,E2,Eh,Ne,Ne_CB,N_vh,r,vel,rh,velh,t_nexth,Ee,r_event,curr_particle,curr_event,n_events,noec_tot,noec,delta_noec_cur,time_grid,space_grid,r_timegrid,rh_timegrid,vel_timegrid,velh_timegrid,E_timegrid,cur_spherical_Ne,cur_spherical_Ee,cur_spherical_Ee2,cur_cylind_Ne,cur_cylind_Ee,cur_cylind_Ee2,cur_spherical_Eh,cur_cyl_E123h,cur_N123fh_3d,cur_N123fh_3d_partconv,cur_N123fh_3d_cleverconv,cur_3d_Ne,cur_3d_Ne_cleverconv,cur_Z_Ne,cur_Z_Ee,cur_Z_Ee2,cur_Z_Eh,cur_Z_Nh,cur_cyl_Efh,cur_cyl_Nfh,cur_Z_Efh,cur_Z_Nfh,cur_2d_Nfh,cur_Etot,Ech,vel_old,vel_mod,curr_events_3d,curr_events_2d,curr_events_1d,curr_Nevents,curr_Eevents,curr_electron_range,Ekin,cur_Etot_timegrid,cur_Ech_timegrid,last_event_done,phot_zspread,Nphotons,lost_photons,event) 
!$omp do reduction( + :Ne_tot,Ne_avg_t,noec_avg, Nvholes, avg_cylind_Ne,avg_cylind_Ee,avg_cylind_Ee2, avg_spherical_Ne,avg_spherical_Ee,avg_spherical_Ee2,avg_spherical_Eh,avg_cyl_E123h,avg_N123fh_3d,avg_N123fh_3d_partconv,avg_N123fh_3d_cleverconv,avg_3d_Ne,avg_3d_Ne_cleverconv, avg_Z_Ne,avg_Z_Ee,avg_Z_Ee2,avg_Z_Eh,avg_Z_Nh,avg_cyl_Efh,avg_cyl_Nfh,avg_Z_Efh,avg_Z_Nfh,avg_2d_Nfh, avg_events_3d, avg_events_2d,avg_events_1d,avg_Nevents,avg_Eevents,avg_electron_range,Etot_timegrid,Ech_timegrid,tot_lost_photons,tot_photons)	

cascades_loop: do curr_cascade = 1, nmc

    n_events = READING_number_of_events(curr_cascade)
    if (n_events .gt. max_num_events) then
        print*, 'Stopping: max_num_events, ',max_num_events,', is smaller than n_events,', n_events, ', in cascade number', curr_cascade
	stop
    else if (n_events .eq. 0) then
	print *, "Warning: No events in cascade", curr_cascade
	if (nmc .eq. 1) stop
	cycle cascades_loop
    endif

    if (.not.allocated(event)) allocate(event(n_events))

    ! these should all be replaced by the object 'event' above
    if (.not. allocated(koe)) allocate(koe(n_events))
    if (.not. allocated(nop)) allocate(nop(n_events))
    if (.not. allocated(time_of_event)) allocate(time_of_event(n_events))
    if (.not. allocated(E_i)) allocate(E_i(n_events))
    if (.not. allocated(E_f)) allocate(E_f(n_events))
    if (.not. allocated(E2)) allocate(E2(n_events))
    if (.not. allocated(Eh)) allocate(Eh(n_events))
    if (.not. allocated(Ech)) allocate(Ech(n_events))
    if (.not. allocated(Ne_CB)) allocate(Ne_CB(n_events))
    if (.not. allocated(N_vh)) allocate(N_vh(n_events))
    if (.not. allocated(noec)) allocate(noec(n_events))
    if (.not.allocated(cur_Etot)) allocate(cur_Etot(n_events))	

    cur_Etot(:) = 0.d0
    Ech(:) = 0.d0
    
    noec_tot = 0
    noec(1:n_events) = 0
    N_vh(1:n_events) = 0

    call READING_cascade_old(curr_cascade,n_events,koe,nop,time_of_event,E_i, E_f, E2, Eh,Ne_CB)
    call READING_cascade(curr_cascade,n_events,event)
    ! now we know the types (koe) and times (time_of_event) of all the events in the cascade as well as the initial (E_i) and final (E_f) energies of the colliding electrons and the energies of the newly created electrons (E2)
    ! so the current cascade is now in the RAM; let us add spatial resolution to it
    !~ if (E_f(n_events).ge.physical%E_threshold) print*, "Warning: in cascade", curr_cascade, "particle number", nop(n_events), "never lost all its energy below treshold"

    Ne = Ne_CB(n_events)	! total number of electrons at the end of current cascade
    Ne_tot = Ne_tot + Ne
    Ne_tot_conv(curr_cascade) = dble(Ne)
    imin = minval(Eh(:), mask=(Eh>0.d0))

    ne_avg_t(1:n_events) = ne_avg_t(1:n_events) + Ne_CB(1:n_events)

    if (.not. allocated(r)) allocate(r(Ne,n_events,3))
    if (.not. allocated(vel)) allocate(vel(Ne,n_events,3))
    if (mheff.gt.0.d0) then	! if valence holes exist in this materials and their effective mass is given, then prepare for their motion
	if (.not. allocated(rh)) allocate(rh(Ne+Ne/2,n_events,3))
	if (.not. allocated(velh)) allocate(velh(Ne+Ne/2,n_events,3))
	if (.not. allocated(t_nexth)) allocate(t_nexth(Ne+Ne/2))
	rh(:,:,:) = 1.d100
	velh(:,:,:)  = 0.d0
   endif
    if (.not. allocated(Ee)) allocate(Ee(Ne,n_events))
    if (.not. allocated(r_event)) allocate(r_event(n_events,3))

		Ee(1:Ne, 1:n_events) = 0.d0
		
		r_event0(1:3) = 0.d0		! the first particle is created at the position (0,0,0)

		curr_event = 1
		curr_particle = nop(curr_event) ! curr_particle = 1

	        if (koe(curr_event) .eq. 1) then	! The cascade starts from a photon
			call start_with_photon(nmc, curr_cascade, n_events, curr_event, Ne, curr_particle, r_event0, r_event, r, physical%E_threshold, E2, vel, physical%photoel_direction, time_of_event, mheff, Eh, N_vh, rh, velh, t_nexth, Ee, tfp_h)			
		else if (koe(curr_event) .eq. 2) then	! The cascade starts from an incident electron
			call start_with_electron(nmc, n_events, Ne_CB, curr_event, Ne, curr_particle, r_event0, r_event, r, physical%E_threshold, E2, E_i, E_f, vel, time_of_event, mheff, Eh, N_vh, rh, velh, t_nexth, Ee, tfp_h, physical%scattering_angles, nop)
		else
			print*, "There must be an error in the first line of cascade",curr_cascade
			stop
		endif

		cur_Etot(curr_event) = sum(Ee(:,curr_event))
		Ech(:) = E_i(:)-E_f(:)-E2(:)-Eh(:)


!================================ Monte Carlo CORE ================================================

	events_loop: do curr_event=2, n_events

		cur_Etot(curr_event) = 0.d0
		N_vh(curr_event) = N_vh(curr_event-1)
		
		electrons_fly: do curr_particle=1, Ne_CB(curr_event)
			if (curr_particle .ne. ne_cb(curr_event) .or. Ne_CB(curr_event).eq.Ne_CB(curr_event-1)) then
				! particles that did not change their trajectory fly further with the same energy (otherwise, see below)
				r(curr_particle,curr_event,:) = r(curr_particle,curr_event-1,:) + vel(curr_particle,curr_event-1,:) * (time_of_event(curr_event)-time_of_event(curr_event-1))*1.d-15	! find the new coordinate
				vel(curr_particle,curr_event,:) = vel(curr_particle,curr_event-1,:)		! in the absence of collisions the velocity does not change
				Ee(curr_particle,curr_event) = Ee(curr_particle,curr_event-1)			! saving the energy of each electron to plot later
			endif
		enddo electrons_fly
				
		! holes can sometimes fly, too (experimental feature, do not use):
		if (mheff.gt.0.d0) then
			holes_fly: do curr_particle=1,N_vh(curr_event)
				rh(curr_particle,curr_event,:) = rh(curr_particle,curr_event-1,:) + velh(curr_particle,curr_event-1,:) * (time_of_event(curr_event)-time_of_event(curr_event-1))*1.d-15
				velh(curr_particle,curr_event,:) = velh(curr_particle,curr_event-1,:)		! in the absence of collisions the velocity does not change
			enddo holes_fly
		endif
		
		! if some event occurs, we will rewrite the new corrdinate, velocity and the energy of particles, depending on the kind of the event (koe)
		if (koe(curr_event).eq.2) then ! inelastic scattering of electron number nop(curr_event) occurs
			
			Ee(nop(curr_event),curr_event) = E_f(curr_event)	! saving the energy of each electron to plot later
			Ee(Ne_CB(curr_event),curr_event) = E2(curr_event)	! saving the energy of each electron to plot later


			if (E_f(curr_event) .gt. physical%E_threshold) then
				vel_mod = dsqrt(2.d0/me*E_f(curr_event)*e)	! E = mv^2/2		! new speed of the old electron
				vel_old(:) = vel(nop(curr_event),curr_event-1,:)/dsqrt &	! direction of the incident electron velocity
							( dot_product(vel(nop(curr_event),curr_event-1,:),vel(nop(curr_event),curr_event-1,:)) )
				if (physical%scattering_angles.eq.0) then
					! new velocity of the old electron according to the forward scattering
					vel(nop(curr_event),curr_event,:) = vel_mod * vel_old
				else if (physical%scattering_angles.eq.1) then
					! new velocity of the old electron according to the isotropic scattering
					vel(nop(curr_event),curr_event,:) = vel_mod*random_direction()		
				else if (physical%scattering_angles.eq.2) then
					! new velocity of the old electron according to the anisotropic scattering
					vel(nop(curr_event),curr_event,:) = vel_mod*scattering_direction(	 E_f(curr_event),vel_old(:) 	)
				endif
			else
				vel(nop(curr_event),curr_event,:) = 0.d0
			endif

			! the newly created electron
			r(Ne_CB(curr_event),curr_event,:) = r(nop(curr_event),curr_event,:)
			
			if (E2(curr_event) .gt. physical%E_threshold) then
				vel_mod = dsqrt(2.d0/me*E2(curr_event)*e)	! E = mv^2/2		
				if (physical%scattering_angles.eq.0 .or. physical%scattering_angles.eq.1) then
					vel(Ne_CB(curr_event),curr_event,:) = vel_mod*random_direction()	! new electron flies in a random direction
				else if (physical%scattering_angles.eq.2) then
					vel_old = vel(nop(curr_event),curr_event,:) / dsqrt &	! direction of the incident electron velocity
									( dot_product(vel(nop(curr_event),curr_event,:),vel(nop(curr_event),curr_event,:)) )
					! the velocity of the new electron is calculated from the momentum conservation
					vel(Ne_CB(curr_event),curr_event,:) = vel(nop(curr_event),curr_event-1,:) - vel(nop(curr_event),curr_event,:)	! the direction is correct, but the energy is not, because a part shoud have been lost to excite the new electron
					vel(Ne_CB(curr_event),curr_event,:) = vel(Ne_CB(curr_event),curr_event,:) / dsqrt &
									( dot_product(vel(Ne_CB(curr_event),curr_event,:),vel(Ne_CB(curr_event),curr_event,:)) )	! now we have the unit vector in the right direction
					vel(Ne_CB(curr_event),curr_event,:) = dsqrt(2.d0/me*E2(curr_event)*e) * vel(Ne_CB(curr_event),curr_event,:)	! now its modulus is correct
				endif
			else
				vel(Ne_CB(curr_event),curr_event,:) = 0.d0
			endif
			
			r_event(curr_event,:) = r(nop(curr_event),curr_event,:)			! saving the coordinate of current event		
			Ech(curr_event) = Ech(curr_event-1) + E_i(curr_event)-E_f(curr_event)-E2(curr_event)-Eh(curr_event)
			
			if (mheff.gt.0.d0 .and. Eh(curr_event).gt.0.d0) then	! a valence hole is created at this event!
				N_vh(curr_event) =  N_vh(curr_event) + 1
				rh(N_vh(curr_event),curr_event,:) = r_event(curr_event,:)	! its coordinate
				vel_mod = dsqrt(2.d0/(mheff*me)*Eh(curr_event)*e)	! its speed
				velh(N_vh(curr_event),curr_event,:) = vel_mod*random_direction()	! its velocity
				t_nexth(N_vh(curr_event)) = time_of_event(curr_event) + next_scattering_time_hole(tfp_h)
			endif

		else if (koe(curr_event).eq.3) then ! deep hole moved up creating electron number Ne_CB(curr_event)
			if (nop(curr_event)+ noec(nop(curr_event)) .le. n_events) then
                                delta_noec_cur = noec(nop(curr_event)+ noec(nop(curr_event))) - noec(nop(curr_event))	! attempt to guess, at which event this hole has been created; this is the difference in the number of event appeared due to elastic events
                        else
                                delta_noec_cur=-(nop(curr_event)+noec(nop(curr_event)))+1
                        endif
			if (.not. nop(curr_event).eq.ne_cb(nop(curr_event)+noec(nop(curr_event))+delta_noec_cur)) then	! if the attempt has failed, find it by brootforce; should be rare
				delta_noec_cur = -(nop(curr_event)+noec(nop(curr_event)))+1
				do while (.not. nop(curr_event).eq.ne_cb(nop(curr_event)+noec(nop(curr_event))+delta_noec_cur))
					delta_noec_cur = delta_noec_cur + 1
					if (nop(curr_event)+noec(nop(curr_event))+delta_noec_cur .gt. n_events) stop "Could not find the coordinate of the deep hole; please check the algorithm"	! otherwise we go out of array boundaries
				enddo
				if (nop(curr_event)+noec(nop(curr_event))+delta_noec_cur .gt. n_events) stop "Could not find the coordinate of the deep hole; please check the algorithm"	! otherwise we go out of array boundaries
			endif
			
			Ee(Ne_CB(curr_event),curr_event) = E2(curr_event)	! saving the energy of each electron to plot later
			E_i(curr_event) = Ech(curr_event-1)
			Ech(curr_event) = Ech(curr_event-1) - E_f(curr_event) - E2(curr_event) - Eh(curr_event)
			
			r(Ne_CB(curr_event),curr_event,:) = r_event( nop(curr_event) + noec(nop(curr_event)) + delta_noec_cur  ,:)  			! the coordinates of the newly created electron; its coordinates must coinside with those from the electron created this hole (its number is nop(curr_event), it was acting at event where particle nop(curr_event) was created; the number of such event may be harder to find, since there might have been elastic colisions in between)					
			
			if (E2(curr_event) .gt. physical%E_threshold) then
				vel_mod = dsqrt(2.d0/me*E2(curr_event)*e)	! E = mv^2/2		! speed of the newly created electron
				vel(Ne_CB(curr_event),curr_event,:) = vel_mod*random_direction()	! new deep-hole-created electron always flies in a random direction
			else
				vel(Ne_CB(curr_event),curr_event,:) = 0.d0
			endif
			
			r_event(curr_event,:) = r(Ne_CB(curr_event),curr_event,:)	! saving the coordinate of current event
		
			if (mheff.gt.0.d0 .and. Eh(curr_event).gt.0.d0) then	! a valence hole is created at this event!
				N_vh(curr_event) =  N_vh(curr_event) + 1
				rh(N_vh(curr_event),curr_event,:) = r_event(curr_event,:)	! its coordinate
				vel_mod = dsqrt(2.d0/(mheff*me)*Eh(curr_event)*e)	! its speed
				velh(N_vh(curr_event),curr_event,:) = vel_mod*random_direction()	! its velocity
				t_nexth(N_vh(curr_event)) = time_of_event(curr_event) + next_scattering_time_hole(tfp_h)
			endif
		
			if (mheff.gt.0.d0 .and. E_f(curr_event).gt.0.d0) then	! a core hole moved up and became a valence hole!
				N_vh(curr_event) =  N_vh(curr_event) + 1
				rh(N_vh(curr_event),curr_event,:) = r_event(curr_event,:)	! its coordinate
				vel_mod = dsqrt(2.d0/(mheff*me)*E_f(curr_event)*e)	! its speed
				velh(N_vh(curr_event),curr_event,:) = vel_mod*random_direction()	! its velocity
				t_nexth(N_vh(curr_event)) = time_of_event(curr_event) + next_scattering_time_hole(tfp_h)
			endif
		
		else if (koe(curr_event).eq.4) then	 ! elastic scattering of electron number nop(curr_event) occurs

			Ech(curr_event) = Ech(curr_event-1) ! core hole energy is not changed by an elastic process
		
			if (E_f(curr_event) .gt. physical%E_threshold) then
				vel_mod = dsqrt(2.d0/me*E_f(curr_event)*e)	! E = mv^2/2		! new speed of the scattered electron; should be the same as the old one, but we allow the input table to decide that
				vel_old(:) = vel(nop(curr_event),curr_event-1,:)/dsqrt &	! direction of the incident electron velocity
						( dot_product(vel(nop(curr_event),curr_event-1,:),vel(nop(curr_event),curr_event-1,:)) )
				if (physical%scattering_angles.eq.0) then
					! new velocity of the old electron according to the direct scattering
					vel(nop(curr_event),curr_event,:) = vel_mod * vel_old(:)
				else if (physical%scattering_angles.eq.1) then
					! new velocity of the old electron according to the isotropic scattering
					vel(nop(curr_event),curr_event,:) = vel_mod*random_direction()		! velocity of the electron				
				else if (physical%scattering_angles.eq.2) then
					! new velocity of the old electron according to the anisotropic scattering
					vel(nop(curr_event),curr_event,:) = vel_mod*scattering_direction(E_f(curr_event),vel_old(:))
				endif
			else
				vel(nop(curr_event),curr_event,:) = 0.d0
			endif

			Ee(nop(curr_event),curr_event) = E_f(curr_event)		! saving the energy of each electron to plot later
			
			r_event(curr_event,:) = r(nop(curr_event),curr_event,:)				! saving the coordinate of current event
			noec_tot = noec_tot + 1
		endif

		noec(curr_event) = noec_tot 		! total number of elastic collisions by time_of_event(curr_event)
		
		cur_Etot(curr_event) = sum(Ee(:,curr_event))


	enddo events_loop
 !================================ / Monte Carlo CORE ================================================


	! adding the energy of valence holes and core holes to the total energy:
	if (switch%rewrite_cascades.gt.0) then
		do curr_event = 1, n_events
			if (koe(curr_event).eq.3) cur_Etot(curr_event:n_events) = cur_Etot(curr_event:n_events) + E_f(curr_event) ! in case deep hole turns to valence hole, we add its energy to the total energy
			cur_Etot(curr_event) = cur_Etot(curr_event) + sum(Eh(1:curr_event)) + Ech(curr_event)
			! currently, the energy of the band gap is missing in the total energy, therefore its number is not constant...
		enddo
		call WRITING_cascade_old(curr_cascade,n_events,koe,nop,time_of_event,E_i, E_f, E2, Eh,Ne_CB,r_event, cur_Etot(:),Ech(:))
		
		! soon to be removed:
		do curr_event=1,n_events
			event(curr_event)%r(1:3)=r_event(curr_event,1:3)
		enddo
		event(:)%Etot = cur_Etot(:)
		event(:)%Ech = Ech(:)
		
		call WRITING_cascade(curr_cascade,n_events,event)
	endif
	
	
	curr_electron_range(:) = 0.d0
	! Find the electron range 1 (photoelectron range)
	el_rng1_loop: do curr_event = n_events, 1, -1
		if (koe(curr_event).eq.2 .and. nop(curr_event) .eq. 1)	then	! searching for inelastic scattering events of the first photoelectron
		
			dist2 = (r_event(curr_event,1)-r_event(1,1))**2 + (r_event(curr_event,2)-r_event(1,2))**2 + (r_event(curr_event,3)-r_event(1,3))**2
			curr_electron_range(1) =dist2
			exit el_rng1_loop
			
		endif
	enddo el_rng1_loop
	
	
	! Find the electron range 2 (furtherst electron range)
	el_rng2_loop: do curr_event = n_events, 1, -1
		
		if (koe(curr_event).eq.2) then
			
			dist2 = (r_event(curr_event,1)-r_event(1,1))**2 + (r_event(curr_event,2)-r_event(1,2))**2 + (r_event(curr_event,3)-r_event(1,3))**2
			
			if (curr_electron_range(2).lt.dist2) curr_electron_range(2) = dist2
		endif
			
	enddo el_rng2_loop
	
	
	! Find the electron range 3 (range of Auger electron)
	! (1) finding the Auger electron
	curr_particle = 1
	el_rng3_loop1: do curr_event = 1, n_events, 1
		if (koe(curr_event).eq.3 .and. nop(curr_event).eq.1) then	! this decay should be at 0,0,0 !
			curr_particle = Ne_CB(curr_event)
			exit el_rng3_loop1
		endif
	enddo el_rng3_loop1
	! (2) finding how far it went
	if (.not. curr_particle .eq. 1) then
		el_rng3_loop2: do curr_event = n_events, 1, -1
			if (koe(curr_event).eq.2 .and. nop(curr_event) .eq. curr_particle)	then	! searching for inelastic scattering events of the Auger electron
				dist2 = (r_event(curr_event,1)-r_event(1,1))**2 + (r_event(curr_event,2)-r_event(1,2))**2 + (r_event(curr_event,3)-r_event(1,3))**2
				curr_electron_range(3) =dist2
				exit el_rng3_loop2
				
			endif
		enddo el_rng3_loop2
	endif

	curr_electron_range(1:3) = dsqrt(curr_electron_range(1:3))
	!print *, curr_cascade, curr_electron_range*1e9
	
	! averaging the electron ranges
	avg_electron_range(1:3) = avg_electron_range(1:3) + curr_electron_range(1:3)
	electron_range_conv(1:3,curr_cascade) = curr_electron_range(1:3)
	
	
	! Counting events spatially and temporally; todo: move this away from the main subroutine...
	
	if (switch%events%XTANT_events.gt.0 .and. allocated(avg_Nevents) .and. allocated(avg_Eevents)) then
		if (.not. allocated(curr_Nevents)) allocate(curr_Nevents(Ntime,-NspaceX:NspaceX))
		if (.not. allocated(curr_Eevents)) allocate(curr_Eevents(Ntime,-NspaceX:NspaceX))
		curr_Nevents(:,:) = 0.d0
		curr_Eevents(:,:) = 0.d0
		!~ test_var = 0.d0
		events: do curr_event = 1, n_events
			if (.not. koe(curr_event).eq.4) then

				find_time_index_evt: do time_index = 1,Ntime
					time_grid = (time_index)*dt + min_time
					if (abs(time_of_event(curr_event)-time_grid).le.0.5d0*dt) exit find_time_index_evt
				enddo find_time_index_evt
					
				find_space_index_evt: do space_index = -NspaceX, NspaceX
					space_grid = space_index*dx
					if (abs(r_event(curr_event,3)-space_grid).le.0.5d0*dx) exit find_space_index_evt
				enddo find_space_index_evt

				if (abs(time_of_event(curr_event)-time_grid).le.0.5d0*dt .and. abs(r_event(curr_event,3)-space_grid).le.0.5d0*dx) then
					if (E2(curr_event) .lt. physical%E_threshold .and. E2(curr_event) .lt. imin) then	! if the secondary electron halts, we include its energy
						curr_Nevents(time_index,space_index) = curr_Nevents(time_index,space_index) + 1.d0
						curr_Eevents(time_index,space_index) = curr_Eevents(time_index,space_index) + E2(curr_event)
					endif
					if (Eh(curr_event).gt.0.d0) then	! if a valence hole is created, we include its energy
						curr_Eevents(time_index,space_index) = curr_Eevents(time_index,space_index) + Eh(curr_event)
					endif
					if (koe(curr_event).eq.3 .and. E_f(curr_event).gt.0.d0) then	! if a deep hole moved up and turned into a valence hole, we include its energy
						curr_Eevents(time_index,space_index) = curr_Eevents(time_index,space_index) + E_f(curr_event)
					endif
					if (koe(curr_event).eq.2 .and. E_f(curr_event).lt.physical%E_threshold .and. E_f(curr_event) .lt. imin) then	! if a deep hole moved up and turned into a valence hole, we include its energy
						curr_Eevents(time_index,space_index) = curr_Eevents(time_index,space_index) + E_f(curr_event)
					endif
				endif
			endif
		enddo events
	
		avg_Nevents(:,:) = avg_Nevents(:,:) + curr_Nevents(:,:)
		avg_Eevents(:,:) = avg_Eevents(:,:) + curr_Eevents(:,:)
		
		if (allocated(curr_Nevents)) deallocate(curr_Nevents)
		if (allocated(curr_Eevents)) deallocate(curr_Eevents)
	endif
	
	if (allocated(avg_events_1d) .and. switch%events%make_1d.gt.0) then
		if (.not. allocated(curr_events_1d)) allocate(curr_events_1d(0:NspaceX))
		call forming_events_1d(n_events,koe,dx,NspaceX,r_event,curr_events_1d)
		avg_events_1d(:) = avg_events_1d(:) + curr_events_1d(:)
		if (allocated(curr_events_1d)) deallocate(curr_events_1d)
	endif
	
	if (allocated(avg_events_2d) .and. switch%events%make_2d.gt.0) then
		allocate(curr_events_2d(-NspaceX:NspaceX,-NspaceY:NspaceY))
		call forming_events_2d(n_events,koe,dx,dy,NspaceX,NspaceY,r_event,curr_events_2d)
		avg_events_2d(:,:) = avg_events_2d(:,:) + curr_events_2d(:,:)
		deallocate(curr_events_2d)
	endif

	if (allocated(avg_events_3d) .and. switch%events%make_3d.gt.0) then
		allocate(curr_events_3d(-NspaceX:NspaceX,-NspaceY:NspaceY,-NspaceZ:NspaceZ))
		call forming_events_3d(n_events,koe,numeric%grid%space,r_event,curr_events_3d)
		avg_events_3d(:,:,:) = avg_events_3d(:,:,:) + curr_events_3d(:,:,:)
		deallocate(curr_events_3d)
	endif
	
	
	! allocation
	if (.not. allocated(r_timegrid)) allocate(r_timegrid(Ne,Ntime,3))
	if (.not. allocated(vel_timegrid)) allocate(vel_timegrid(Ne,Ntime,3))
	if (.not. allocated(rh_timegrid) .and. mheff.gt.0.d0) allocate(rh_timegrid(Ne+Ne/2,Ntime,3))
	if (.not. allocated(velh_timegrid) .and. mheff.gt.0.d0) allocate(velh_timegrid(Ne+Ne/2,Ntime,3))
	if (.not. allocated(E_timegrid)) allocate(E_timegrid(Ne,Ntime))

	if (.not. allocated(cur_Etot_timegrid)) allocate(cur_Etot_timegrid(Ntime))
	if (.not. allocated(cur_Ech_timegrid)) allocate(cur_Ech_timegrid(Ntime))
	cur_Etot_timegrid(:) = 0.d0
	cur_Ech_timegrid(:) = 0.d0

	r_timegrid(:,:,:) = 1.d100	! all particles are infinitely far away = they do not exist (implemented in functions Forming_*, file Write_output.f90)
	vel_timegrid(:,:,:) = 0.d0	! initial velocities of all particles
	if (allocated(rh_timegrid)) rh_timegrid(:,:,:) = 1.d100	! all particles are infinitely far away = they do not exist (implemented in function Forming_..., file Write_output.f90)
	if (allocated(velh_timegrid)) velh_timegrid(:,:,:) = 0.d0	! initial velocities of all particles
	E_timegrid(:,:) = 0.d0		! initial energies of all particles
	
	! Now replace the inhomogeneous time grid with homogeneous one
	
	! first event
	curr_event = 1
	curr_particle = nop(curr_event) ! curr_particle = 1
	
	find_time_index: do time_index = 1,Ntime
		time_grid = (time_index)*dt + min_time
		if (abs(time_of_event(curr_event)-time_grid).le.0.5d0*dt) exit find_time_index
enddo find_time_index

	if (time_index .lt. 1 .or. time_index .ge. Ntime) then
		print*, curr_cascade, curr_event, time_of_event(curr_event),time_grid, time_index, Ntime
		print *, "min_time and/or max_time must be increased! (cascade", curr_cascade, ")"
		stop
	else if ( time_of_event(n_events) .gt. max_time ) then
		!~ print*, curr_cascade, time_of_event(n_events), Ntime*dt + min_time, Ntime
		print *, "max_time must be increased at least up to", time_of_event(n_events), "fs , see cascade", curr_cascade, "; currently it's", max_time
		stop
	endif
	
	cur_Ech_timegrid(time_index) = Ech(curr_event)

	if (koe(curr_event) .eq. 1) then	! The cascade starts from a photon
	
		!only one photoelectron is there
		r_timegrid(curr_particle,time_index,:) = r(curr_particle,curr_event,:)
		vel_timegrid(curr_particle,time_index,:) = vel(curr_particle,curr_event,:)
		E_timegrid(curr_particle,time_index) = E2(curr_event)
		cur_Etot_timegrid(time_index) = E2(curr_event)+Eh(curr_event) + Ech(curr_event)
		
	else if (koe(curr_event) .eq. 2) then	! The cascade starts from an incident electron
		
		!impact electron
		r_timegrid(curr_particle,time_index,:) = r(curr_particle,curr_event,:)
		vel_timegrid(curr_particle,time_index,:) = vel(curr_particle,curr_event,:)
		E_timegrid(curr_particle,time_index) = E_f(curr_event)
		!secondary electron
		r_timegrid(Ne_CB(curr_event),time_index,:) = r(Ne_CB(curr_event),curr_event,:)		
		vel_timegrid(Ne_CB(curr_event),time_index,:) = vel(Ne_CB(curr_event),curr_event,:)
		E_timegrid(Ne_CB(curr_event),time_index) = E2(curr_event)
		cur_Etot_timegrid(time_index) = E_f(curr_event)+E2(curr_event)+Eh(curr_event) + Ech(curr_event)
	
	endif
		
		! valence holes flying: the first event
		if (N_vh(curr_event).gt.0) then
			if (allocated(rh_timegrid)) rh_timegrid(1:N_vh(curr_event),time_index,:) = rh(1:N_vh(curr_event),curr_event,:)
			if (allocated(velh_timegrid)) velh_timegrid(1:N_vh(curr_event),time_index,:) = velh(1:N_vh(curr_event),curr_event,:)
		endif

	! other events; preparing the time indexes
	last_event_done = .false.
	time_grid_loop: do time_index = 1,Ntime

		time_grid = (time_index)*dt + min_time
		curr_event = minloc(abs(time_of_event-time_grid),1)	!searching for the corresponding event at this time
		if (curr_event.le.1) cycle time_grid_loop

			!======electrons (and holes) are flying between the events and after the last event until max_time=====
			if ( abs(time_grid-time_of_event(curr_event)) .le. 0.5d0*dt ) then	! if any event occured, we rewrite all the coordinates, velocities, and kinetic energies to what they should be accordingly
				r_timegrid(1:Ne_CB(curr_event),time_index,:) = r(1:Ne_CB(curr_event),curr_event,:)
				vel_timegrid(1:Ne_CB(curr_event),time_index,:) = vel(1:Ne_CB(curr_event),curr_event,:)
				
				! new valence hole(s) probably created, let them fly
				if (allocated(rh_timegrid)) rh_timegrid(1:N_vh(curr_event),time_index,:) = rh(1:N_vh(curr_event),curr_event,:)
				if (allocated(velh_timegrid)) velh_timegrid(1:N_vh(curr_event),time_index,:) = velh(1:N_vh(curr_event),curr_event,:)

			else	! Overwise we just follow straight paths of the electrons (and holes)

				! before proceeding with the forward electron (and hole) motion, make sure that the last event has been taken into account (otherwise the direction of the motion will be wrong!)
				if ( curr_event.eq.n_events .and. (time_grid-time_of_event(curr_event)) .gt. 0.d0 .and. .not.last_event_done) then
					r_timegrid(1:Ne_CB(curr_event),time_index,:) = r(1:Ne_CB(curr_event),curr_event,:)
					vel_timegrid(1:Ne_CB(curr_event),time_index,:) = vel(1:Ne_CB(curr_event),curr_event,:)
					if (allocated(rh_timegrid)) rh_timegrid(1:N_vh(curr_event),time_index,:) = rh(1:N_vh(curr_event),curr_event,:)
					if (allocated(velh_timegrid)) velh_timegrid(1:N_vh(curr_event),time_index,:) = velh(1:N_vh(curr_event),curr_event,:)
					last_event_done = .true.
				else
					! before proceeding with the forward electron motion, check whether some particles must have been stopped
					do curr_particle = 1, ne_cb(curr_event)
						Ekin = 0.5d0*me/e*dot_product(vel_timegrid(curr_particle,time_index,:),vel_timegrid(curr_particle,time_index,:))
						if (Ekin .lt. physical%E_threshold .and. Ekin .gt. 0.d0) then
							print*, "chk", curr_cascade, curr_event, curr_particle, Ekin, physical%E_threshold
							vel_timegrid(curr_particle,time_index,:) = 0.d0
						endif
					enddo
					r_timegrid(1:Ne_CB(curr_event),time_index,:) = r_timegrid(1:Ne_CB(curr_event),time_index-1,:) + vel_timegrid(1:Ne_CB(curr_event),time_index-1,:) * dt*1.d-15	
					vel_timegrid(1:Ne_CB(curr_event),time_index,:) = vel_timegrid(1:Ne_CB(curr_event),time_index-1,:)
					
					if (allocated(velh_timegrid)) then
					! scatterings of holes
						do curr_particle = 1, N_vh(curr_event)
							if ( abs( time_grid-t_nexth(N_vh(curr_event)) ) .le. 0.5d0*dt ) then	! if any hole scatters now
								rh_timegrid(curr_particle,time_index,:) = rh_timegrid(curr_particle,time_index-1,:) + velh_timegrid(curr_particle,time_index-1,:) * dt*1.d-15	
								velh_timegrid(curr_particle,time_index,:) = velh_timegrid(curr_particle,time_index-1,:)		! in the absence of collisions the velocity does not change
							else	! other holes fly forward
								rh_timegrid(curr_particle,time_index,:) = rh_timegrid(curr_particle,time_index-1,:) + velh_timegrid(curr_particle,time_index-1,:) * dt*1.d-15	
								vel_mod = dsqrt(dot_product(velh_timegrid(curr_particle,time_index,:),velh_timegrid(curr_particle,time_index,:)))			! current speed of the hole
								velh_timegrid(curr_particle,time_index,:) = vel_mod*random_direction()		! in the absence of collisions the velocity does not change
								t_nexth(curr_particle) = time_grid + next_scattering_time_hole(tfp_h)
							endif
						enddo
					endif
					
				endif
			endif

		! changing the second index of energy from curr_event to time_index
		if (.not. physical%inclHighEnEl) then
			do curr_particle = 1, Ne_CB(curr_event)
				if (Ee(curr_particle,curr_event).le.physical%E_threshold) &		! only include low-energy electrons into the energy 
					E_timegrid(curr_particle,time_index) = Ee(curr_particle,curr_event)
			enddo
		else
			do curr_particle = 1, Ne_CB(curr_event)
					E_timegrid(curr_particle,time_index) = Ee(curr_particle,curr_event)
			enddo
		endif
			
			! calculating total energy at this time index:
			cur_Etot_timegrid(time_index) = cur_Etot(curr_event)


!do curr_particle = 1, Ne_CB(curr_event)
!	if (	abs(maxval(r_timegrid(:, time_index, 3))) .gt. 2000.d-9 .and. &	 ! a particle flew away very far in a very  short time. strange. might be a damaged OUTPUT_cascade*.dat file
!		abs(maxval(r_timegrid(:, time_index, 3))) .lt. 1d99 ) &
!		print *, "Warning:", curr_cascade, curr_event, time_index, curr_particle, maxval(r_timegrid(:, time_index, 3))*1d9
!enddo


	enddo time_grid_loop

	! collecting the sum over all cascades
	Etot_timegrid(:) = Etot_timegrid(:) + cur_Etot_timegrid(:)
	Nvholes = Nvholes + N_vh(n_events)


	if (switch%electrons%Z_dependence .gt. 0) then
		if (.not. allocated(cur_Z_Ne)) allocate(cur_Z_Ne(Ntime,-NspaceX:NspaceX))
		if (.not. allocated(cur_Z_Ee)) allocate(cur_Z_Ee(Ntime,-NspaceX:NspaceX))
		if (.not. allocated(cur_Z_Ee2)) allocate(cur_Z_Ee2(Ntime,-NspaceX:NspaceX))
		if (.not. allocated(cur_Z_Nh)) allocate(cur_Z_Nh(Ntime,-NspaceX:NspaceX))
		if (.not. allocated(cur_Z_Eh)) allocate(cur_Z_Eh(Ntime,-NspaceX:NspaceX))
		call Forming_Z_dependence(Ne,Ntime,NspaceX, min_time,dt,dx, r_timegrid, vel_timegrid, E_timegrid,  cur_Z_Ne,cur_Z_Ee,cur_Z_Ee2)
		call Forming_Z_dependence_holes(n_events, koe, E_f, r_event,time_of_event,Ntime,NspaceX,min_time,dt,dx, Eh, cur_Z_Nh,cur_Z_Eh)
		avg_Z_Ne(:,:) = avg_Z_Ne(:,:) + cur_Z_Ne(:,:)
		avg_Z_Ee(:,:) = avg_Z_Ee(:,:) + cur_Z_Ee(:,:)
		avg_Z_Ee2(:,:) = avg_Z_Ee2(:,:) + cur_Z_Ee2(:,:)
		avg_Z_Nh(:,:) = avg_Z_Nh(:,:) + cur_Z_Nh(:,:)
		avg_Z_Eh(:,:) = avg_Z_Eh(:,:) + cur_Z_Eh(:,:)
		if (allocated(cur_Z_Ne)) deallocate(cur_Z_Ne)
		if (allocated(cur_Z_Ee)) deallocate(cur_Z_Ee)
		if (allocated(cur_Z_Ee2)) deallocate(cur_Z_Ee2)
		if (allocated(cur_Z_Nh)) deallocate(cur_Z_Nh)
		if (allocated(cur_Z_Eh)) deallocate(cur_Z_Eh)
	endif
		
	if (switch%electrons%spherical .gt. 0) then
		if (.not. allocated(cur_spherical_Ne)) allocate(cur_spherical_Ne(Ntime,NspaceX))
		if (.not. allocated(cur_spherical_Ee)) allocate(cur_spherical_Ee(Ntime,NspaceX))
		if (.not. allocated(cur_spherical_Ee2)) allocate(cur_spherical_Ee2(Ntime,NspaceX))
		if (.not. allocated(cur_spherical_Eh)) allocate(cur_spherical_Eh(Ntime,NspaceX))
		call Forming_spherical(Ne,Ntime,NspaceX, min_time,dt,dx, r_timegrid, vel_timegrid, E_timegrid,  cur_spherical_Ne,cur_spherical_Ee,cur_spherical_Ee2)
		call Forming_spherical_holes(n_events, koe, r_event,time_of_event, Ntime,NspaceX,min_time,dt,dx, Eh, E_f, cur_spherical_Eh)
		avg_spherical_Ne(:,:) = avg_spherical_Ne(:,:) + dble(cur_spherical_Ne(:,:))
		avg_spherical_Ee(:,:) = avg_spherical_Ee(:,:) + cur_spherical_Ee(:,:)
		avg_spherical_Ee2(:,:) = avg_spherical_Ee2(:,:) + cur_spherical_Ee2(:,:)
		avg_spherical_Eh(:,:) = avg_spherical_Eh(:,:) + cur_spherical_Eh(:,:)
		if (allocated(cur_spherical_Ne)) deallocate(cur_spherical_Ne)
		if (allocated(cur_spherical_Ee)) deallocate(cur_spherical_Ee)
		if (allocated(cur_spherical_Ee2)) deallocate(cur_spherical_Ee2)
		if (allocated(cur_spherical_Eh)) deallocate(cur_spherical_Eh)
	endif
		
	if (switch%electrons%cylindrical .gt. 0) then
		if (.not. allocated(cur_cylind_Ne)) allocate(cur_cylind_Ne(Ntime,NspaceX))
		if (.not. allocated(cur_cylind_Ee)) allocate(cur_cylind_Ee(Ntime,NspaceX))
		if (.not. allocated(cur_cylind_Ee2)) allocate(cur_cylind_Ee2(Ntime,NspaceX))
		call Forming_cylindrical(Ne,Ntime,NspaceX, min_time,dt,dx, r_timegrid, vel_timegrid, E_timegrid,  cur_cylind_Ne,cur_cylind_Ee,cur_cylind_Ee2)
		avg_cylind_Ne(:,:) = avg_cylind_Ne(:,:) + cur_cylind_Ne(:,:)
		avg_cylind_Ee(:,:) = avg_cylind_Ee(:,:) + cur_cylind_Ee(:,:)
		avg_cylind_Ee2(:,:) = avg_cylind_Ee2(:,:) + cur_cylind_Ee2(:,:)
		if (allocated(cur_cylind_Ne)) deallocate(cur_cylind_Ne)
		if (allocated(cur_cylind_Ee)) deallocate(cur_cylind_Ee)
		if (allocated(cur_cylind_Ee2)) deallocate(cur_cylind_Ee2)
	endif

	do curr_event = 2, n_events
	! Adding the energy of deep holes that turned to valence holes to Eh
		if (koe(curr_event) .eq. 3) Eh(curr_event) = Eh(curr_event) + E_f(curr_event)
	enddo

	! 3d distribution of electrons
	if (switch%electrons%make_3d .gt. 0) then
		if (.not. allocated(cur_3d_Ne)) allocate(cur_3d_Ne(-NspaceX:NspaceX,-NspaceY:NspaceY,-NspaceZ:NspaceZ))
		if (beam%show_laser_shape.gt.0) then
			r_timegrid(:,:,:) = 1.d100
			r_timegrid(1,:,:) = 0.d0
		endif
		call Forming_3d_electron_distribution(switch%make3d_time_index,Ne,numeric%grid%space, numeric%grid%time, r_timegrid, cur_3d_Ne)
		avg_3d_Ne(:,:,:) = avg_3d_Ne(:,:,:) + cur_3d_Ne(:,:,:)
		if (allocated(cur_3d_Ne)) deallocate(cur_3d_Ne)
	endif

	! deallocations
	if (allocated(r_timegrid)) deallocate(r_timegrid)
	if (allocated(vel_timegrid)) deallocate(vel_timegrid)
	if (allocated(rh_timegrid)) deallocate(rh_timegrid)
	if (allocated(velh_timegrid)) deallocate(velh_timegrid)
	if (allocated(t_nexth)) deallocate(t_nexth)
	if (allocated(E_timegrid)) deallocate(E_timegrid)
	if (allocated(cur_Etot)) deallocate(cur_Etot)
	if (allocated(cur_Etot_timegrid)) deallocate(cur_Etot_timegrid)
	if (allocated(cur_Ech_timegrid)) deallocate(cur_Ech_timegrid)

	if (allocated(event)) deallocate(event)
	if (allocated(koe)) deallocate(koe)
	if (allocated(nop)) deallocate(nop)
	if (allocated(time_of_event)) deallocate(time_of_event)
	if (allocated(E_i)) deallocate(E_i)
	if (allocated(E_f)) deallocate(E_f)
	if (allocated(E2)) deallocate(E2)
	if (allocated(Eh)) deallocate(Eh)
	if (allocated(Ech)) deallocate(Ech)
	if (allocated(Ne_CB)) deallocate(Ne_CB)
	if (allocated(N_vh)) deallocate(N_vh)
	if (allocated(noec)) deallocate(noec)

	if (allocated(r)) deallocate(r)
	if (allocated(vel)) deallocate(vel)
	if (allocated(rh)) deallocate(rh)
	if (allocated(velh)) deallocate(velh)
	if (allocated(Ee)) deallocate(Ee)
	if (allocated(r_event)) deallocate(r_event)

    noec_avg = noec_avg + noec_tot
    noec_avg_conv(curr_cascade) = dble(noec_tot)

    NCascadesDone = NCascadesDone + 1
    !~ if (nmc .ge. 100 .and. mod(NCascadesDone,nmc/100).eq.0  ) write(*,'(1a1,i3,a)', advance='no') char(13), nint( dble(NCascadesDone)/dble(nmc) *100.d0 ),'%'	! writing current percentage of processed cascades
    
enddo cascades_loop	! going through all files=cascades

!$omp end do
!$omp end parallel

write(*,'(1a1,i3,a)', advance='no') char(13)	! removing the current percentage of cascades

! convergence study
	do curr_cascade = nmc, 1, -1
		Ne_tot_conv(curr_cascade) = sum(Ne_tot_conv(1:curr_cascade))/dble(curr_cascade)
		noec_avg_conv(curr_cascade) = sum(noec_avg_conv(1:curr_cascade))/dble(curr_cascade)
		electron_range_conv(1,curr_cascade) = sum(electron_range_conv(1,1:curr_cascade))/dble(curr_cascade)
		electron_range_conv(2,curr_cascade) = sum(electron_range_conv(2,1:curr_cascade))/dble(curr_cascade)
		electron_range_conv(3,curr_cascade) = sum(electron_range_conv(3,1:curr_cascade))/dble(curr_cascade)
	enddo

	call writing_convergence(nmc,Ne_tot_conv,noec_avg_conv,electron_range_conv(:,:))
	if (allocated(Ne_tot_conv)) deallocate(Ne_tot_conv)
	if (allocated(noec_avg_conv)) deallocate(noec_avg_conv)
	if (allocated(electron_range_conv)) deallocate(electron_range_conv)

	Etot_timegrid(:) = Etot_timegrid(:)/dble(nmc)
	!~ call writing_totenergy(Ntime,dt,max_time,min_time,Etot_timegrid)  ! there is probably a bug here
	if (allocated(Etot_timegrid)) deallocate(Etot_timegrid)
	

! Finding electron range
	avg_electron_range(1:3) = avg_electron_range(1:3) / dble(nmc)


print *, ''
print*, "Total", dble(noec_avg)/dble(nmc), "elastic collisions in a typical cascade"
print*, "Total", dble(Ne_tot)/dble(nmc), "electrons in a typical cascade"
print *, "Total", dble(Nvholes) / dble(nmc), "valence holes in a typical cascade"
print *, ''
write (*, '(A)') 'Photoelectron range, furtherst electron, Auger electron range: '
write (*, '(3(1x,F13.6),1x,A)') avg_electron_range(1:3)*1.d9, 'nm'
print *, '============================================'
print *, ''

if (allocated(ne_avg_t)) deallocate(ne_avg_t)
if (allocated(Ech_timegrid)) deallocate(Ech_timegrid)	

! ========Writing processed data==============
	
	if (switch%events%XTANT_events .gt. 0) then
		avg_Nevents(:,:) = avg_Nevents(:,:) / dble(nmc)
		avg_Eevents(:,:) = avg_Eevents(:,:) / dble(nmc)
		call writing_NEevents(min_time,dt,Ntime,dx,NspaceX, avg_Nevents,avg_Eevents,switch%events%XTANT_events)	! writing spatial distribution of the events
		if (allocated(avg_Nevents)) deallocate(avg_Nevents)
		if (allocated(avg_Eevents)) deallocate(avg_Eevents)
	endif
		
	if (switch%electrons%spherical .gt. 0) then
		avg_spherical_Ne(:,:) = avg_spherical_Ne(:,:)/dble(nmc)
		avg_spherical_Ee(:,:) = avg_spherical_Ee(:,:)/dble(nmc)	
		avg_spherical_Ee2(:,:) = avg_spherical_Ee2(:,:)/dble(nmc)	
		avg_spherical_Eh(:,:) = avg_spherical_Eh(:,:)/dble(nmc)	
		call writing_spherical(min_time,dt,Ntime,dx,NspaceX,avg_spherical_Ne,avg_spherical_Ee,avg_spherical_Ee2)
		call writing_spherical_holes(n_events,Ntime,NspaceX,min_time,dt,dx, avg_spherical_Eh)
		if (allocated(avg_spherical_Ne)) deallocate(avg_spherical_Ne)
		if (allocated(avg_spherical_Ee)) deallocate(avg_spherical_Ee)
		if (allocated(avg_spherical_Ee2)) deallocate(avg_spherical_Ee2)
		if (allocated(avg_spherical_Eh)) deallocate(avg_spherical_Eh)
	endif
		
	if (switch%electrons%cylindrical .gt. 0) then
		avg_cylind_Ne(:,:) = avg_cylind_Ne(:,:)/dble(nmc)
		avg_cylind_Ee(:,:) = avg_cylind_Ee(:,:)/dble(nmc)	
		avg_cylind_Ee2(:,:) = avg_cylind_Ee2(:,:)/dble(nmc)	
		call writing_cylindrical(min_time,dt,Ntime,dx,NspaceX,avg_cylind_Ne,avg_cylind_Ee,avg_cylind_Ee2)
		if (allocated(avg_cylind_Ne)) deallocate(avg_cylind_Ne)
		if (allocated(avg_cylind_Ee)) deallocate(avg_cylind_Ee)
		if (allocated(avg_cylind_Ee2)) deallocate(avg_cylind_Ee2)
	endif

	if (switch%electrons%Z_dependence .gt. 0) then
		avg_Z_Ne(:,:) = avg_Z_Ne(:,:)/dble(nmc)
		avg_Z_Ee(:,:) = avg_Z_Ee(:,:)/dble(nmc)
		avg_Z_Ee2(:,:) = avg_Z_Ee2(:,:)/dble(nmc)
		avg_Z_Nh(:,:) = avg_Z_Nh(:,:)/dble(nmc)
		avg_Z_Eh(:,:) = avg_Z_Eh(:,:)/dble(nmc)
		call writing_Z_dependence(min_time,dt,Ntime,dx,NspaceX,avg_Z_Ne,avg_Z_Ee,avg_Z_Ee2,switch%electrons%Z_dependence)
		call writing_Z_dependence_holes(min_time,dt,Ntime,dx,NspaceX,avg_Z_Nh,avg_Z_Eh,switch%electrons%Z_dependence)
                if (allocated(avg_Z_Ne)) deallocate(avg_Z_Ne)
                if (allocated(avg_Z_Ee)) deallocate(avg_Z_Ee)
                if (allocated(avg_Z_Ee2)) deallocate(avg_Z_Ee2)
		if (allocated(avg_Z_Nh)) deallocate(avg_Z_Nh)
		if (allocated(avg_Z_Eh)) deallocate(avg_Z_Eh)
	endif

	if (switch%electrons%make_3d .gt. 0) then
		avg_3d_Ne(:,:,:) = avg_3d_Ne(:,:,:)/dble(nmc)
		if (beam%show_laser_shape.gt.0) then
			avg_3d_Ne(:,:,:) = 0.d0				
			avg_3d_Ne(0,0,0) = 1.d0 * beam%Nphotons	
		endif
		call writing_electrons_3d_distribution(switch%make3d_time,Ne,numeric%grid%space, numeric%grid%time,avg_3d_Ne)
		if (allocated(avg_3d_Ne)) deallocate(avg_3d_Ne)
	endif
	
	if (allocated(avg_events_1d) .and. switch%events%make_1d.gt.0) then
		avg_events_1d(:) = avg_events_1d(:) / dble(nmc)
		call writing_events_1d(dx,NspaceX,avg_events_1d)	! writing spatial distribution of the events
		deallocate(avg_events_1d)
	endif
	
	if (allocated(avg_events_2d) .and. switch%events%make_2d.gt.0) then
		avg_events_2d(:,:) = avg_events_2d(:,:) / dble(nmc)
		call writing_events_2d(dx,dy,NspaceX,NspaceY,avg_events_2d)	! writing spatial distribution of the events
		deallocate(avg_events_2d)
	endif
	
	if (allocated(avg_events_3d) .and. switch%events%make_3d.gt.0) then
		avg_events_3d(:,:,:) = avg_events_3d(:,:,:) / dble(nmc)
		call writing_events_3d(numeric%grid%space,avg_events_3d)	! writing spatial distribution of the events
		deallocate(avg_events_3d)
	endif

! ========/Writing processed data==============



!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
! Printing out the duration of the program, starting and ending time and date:

call date_and_time(values=c1)	    ! For calculation of the time of execution of the program
as1=DBLE(24*60*60*(c1(3)-ctim(3))+3600*(c1(5)-ctim(5))+60*(c1(6)-ctim(6))+(c1(7)-ctim(7))+(c1(8)-ctim(8))*0.001d0)	! sec
print*, '   '
if (as1 .lt. 60.d0) then
        write(*,'(a, f10.2, a)') ' Duration of execution of program: ', as1, ' sec'
else if (as1 .lt. 3600.d0) then
        write(*,'(a, f12.4, a)') ' Duration of execution of program: ', as1/60.d0, ' min'
else if (as1 .lt. 86400.d0) then
      write(*,'(a, f14.6, a)') ' Duration of execution of program: ', as1/3600.d0, ' hours'
else
      write(*,'(a, f16.8, a)') ' Duration of execution of program: ', as1/86400.d0, 'days' 
endif
print *, "The calculation is done on", numeric%omp_numthr , "cores"

print*, '   '
write(*, 1001) ctim(5),ctim(6),ctim(7), ctim(3), ctim(2), ctim(1)

write(*, 1002) c1(5), c1(6), c1(7), c1(3),c1(2), c1(1)		
	
1001 format ('Begining: ', i2.2, ':', i2.2, ':', i2.2, '  ', i2.2, '/', i2.2, '/', i4.4)
1002 format ('The end:  ', i2.2, ':', i2.2, ':', i2.2, '  ', i2.2, '/', i2.2, '/', i4.4)
1005 format ('Start at: ', i2.2, ':', i2.2, ':', i2.2, '  ', i2.2, '/', i2.2, '/', i4.4)



END PROGRAM XCascade3D

